/*
** Copyright 1998 - 2003 Double Precision, Inc.
** See COPYING for distribution information.
*/

#if	HAVE_CONFIG_H
#include	"config.h"
#endif

#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<fcntl.h>
#include	<time.h>
#include	<errno.h>

#if	HAVE_UNISTD_H
#include	<unistd.h>
#endif
#include	<sys/types.h>
#include	<sys/stat.h>
#if HAVE_SYS_WAIT_H
#include	<sys/wait.h>
#endif
#ifndef WEXITSTATUS
#define WEXITSTATUS(stat_val) ((unsigned)(stat_val) >> 8)
#endif
#ifndef WIFEXITED
#define WIFEXITED(stat_val) (((stat_val) & 255) == 0)
#endif
#if HAVE_DIRENT_H
#include <dirent.h>
#define NAMLEN(dirent) strlen((dirent)->d_name)
#else
#define dirent direct
#define NAMLEN(dirent) (dirent)->d_namlen
#if HAVE_SYS_NDIR_H
#include <sys/ndir.h>
#endif
#if HAVE_SYS_DIR_H
#include <sys/dir.h>
#endif
#if HAVE_NDIR_H
#include <ndir.h>
#endif
#endif

#include	"liblock/config.h"
#include	"liblock/liblock.h"
#include	"maildir/config.h"
#include	"maildir/maildircreate.h"
#include	"maildir/maildirmisc.h"
#include	"maildir/maildirwatch.h"
#include	"liblock/mail.h"

#include	"imapscanclient.h"
#include	"imaptoken.h"
#include	"imapwrite.h"
#include	"imapd.h"

static const char rcsid[]="$Id: imapscanclient.c,v 1.27 2003/03/19 13:31:28 mrsam Exp $";

static int do_imapscan_maildir2(struct imapscaninfo *, const char *,
				int, int, struct uidplus_info *);
static void imapscanfail(const char *p);

void imapscan_init(struct imapscaninfo *i)
{
	memset(i, 0, sizeof(*i));
}

void imapscan_copy(struct imapscaninfo *a,
		   struct imapscaninfo *b)
{
	imapscan_free(a);
	*a=*b;
	memset(b, 0, sizeof(*b));
}


int imapscan_maildir(struct imapscaninfo *scaninfo,
		     const char *dir, int leavenew, int ro,
		     struct uidplus_info *uidplus)
{
	const	char *p;
	char *tmpname;
	char *newname;
	int rc;
	struct maildir_tmpcreate_info createInfo;

	if ((p=getenv("IMAP_USELOCKS")) != 0 && *p != '1')
		return (do_imapscan_maildir2(scaninfo, dir, leavenew, ro,
					     uidplus));

	if (scaninfo->watcher == NULL &&
	    (scaninfo->watcher=maildirwatch_alloc(dir)) == NULL)
		imapscanfail("maildirwatch");

	maildir_tmpcreate_init(&createInfo);

	createInfo.maildir=dir;
	createInfo.uniq="courierlock";
	createInfo.hostname=getenv("HOSTNAME");
	createInfo.doordie=1;

	if ((rc=maildir_tmpcreate_fd(&createInfo)) < 0)
	{
		write_error_exit(0);
	}
	close(rc);

	tmpname=createInfo.tmpname;
	newname=createInfo.newname;

	createInfo.tmpname=NULL;
	createInfo.newname=NULL;
	maildir_tmpcreate_free(&createInfo);

	/* HACK: newname now contains: ".../new/filename" */

	strcpy(strrchr(newname, '/')-3, WATCHDOTLOCK);

	while (ll_dotlock(newname, tmpname, 120) < 0)
	{
		if (errno == EEXIST)
		{
			if (scaninfo->watcher == NULL ||
			    maildirwatch_unlock(scaninfo->watcher, 120) < 0)
				sleep(1);
			continue;
		}

		if (errno == EAGAIN)
		{
			unlink(newname);
			sleep(5);
			continue;
		}
		imapscanfail(newname);
		free(newname);
		newname=NULL;
		break;
	}

	rc=do_imapscan_maildir2(scaninfo, dir, leavenew, ro, uidplus);

	if (newname)
	{
		unlink(newname);
		free(newname);
	}

	if (rc == 0)
		return 0;

	return (-1);
}

/* This structure is a temporary home for the filenames */

struct tempinfo {
	struct tempinfo *next;
	char *filename;
	unsigned long uid;
	int	found;
	int	isrecent;
	} ;

static char *imapscan_namedir(const char *dir, const char *new_or_cur)
{
char	*p=malloc(strlen(dir)+strlen(new_or_cur)+2);

	if (!p)	write_error_exit(0);
	strcat(strcat(strcpy(p, dir), "/"), new_or_cur);
	return (p);
}

static int fnamcmp(const char *a, const char *b)
{
	long ai, bi;
	ai = atol(a);
	bi = atol(b);
	if(ai - bi)
		return ai - bi;
	return strcmp(a, b);
}

static int sort_by_filename(struct tempinfo **a, struct tempinfo **b)
{
	return (fnamcmp( (*a)->filename, (*b)->filename));
}

static int sort_by_filename_status(struct tempinfo **a, struct tempinfo **b)
{
	if ( (*a)->found && (*b)->found )
	{
		if ( (*a)->uid < (*b)->uid )
			return (-1);
		if ( (*a)->uid > (*b)->uid )
			return (1);
		return (0);	/* What the fuck??? */
	}
	if ( (*a)->found )	return (-1);
	if ( (*b)->found )	return (1);

	return (fnamcmp( (*a)->filename, (*b)->filename));
}

/* Binary search on an array of tempinfos which is sorted by filenames */

static int search_by_filename(struct tempinfo **a, unsigned s, unsigned *i,
	const char *filename)
{
unsigned lo=0, hi=s, mid;
int	rc;

	while (lo < hi)
	{
		mid=(hi+lo)/2;
		rc=fnamcmp( a[mid]->filename, filename);
		if (rc < 0)
		{
			lo=mid+1;
			continue;
		}
		if (rc > 0)
		{
			hi=mid;
			continue;
		}
		*i=mid;
		return (0);
	}
	return (-1);
}

static void imapscanfail(const char *p)
{
	fprintf(stderr, "ERR: Failed to create cache file: %s (%s)\n", p,
		getenv("AUTHENTICATED"));
#if	HAVE_STRERROR
	fprintf(stderr, "ERR: Error: %s\n", strerror(errno));
#endif

#if	HAVE_FAM
	if (errno == EIO)
	{
		fprintf(stderr,
			"ERR: Check for proper operation and configuration\n"
			"ERR: of the File Access Monitor daemon (famd).\n");
	}
#endif
}

static char *readbuf;
static unsigned readbufsize=0;

char *readline(unsigned i, FILE *fp)
{
int	c;

	for (;;)
	{
		if (i >= 10000)
			--i;	/* DOS check */

		if (i >= readbufsize)
		{
		char	*p= readbuf ? realloc(readbuf, readbufsize+256):
					malloc(readbufsize+256);

			if (!p)	write_error_exit(0);
			readbuf=p;
			readbufsize += 256;
		}

		c=getc(fp);
		if (c == EOF || c == '\n')
		{
			readbuf[i]=0;
			return (c == EOF ? 0:readbuf);
		}
		readbuf[i++]=c;
	}
}

static int do_imapscan_maildir2(struct imapscaninfo *scaninfo,
				const char *dir, int leavenew, int ro,
				struct uidplus_info *uidplus)
{
char	*dbfilepath, *newdbfilepath;
struct	tempinfo *tempinfo_list=0, **tempinfo_array=0, *tempinfoptr;
struct	tempinfo *newtempinfo_list=0;
unsigned	tempinfo_cnt=0, i;
FILE	*fp;
char	*p, *q;
unsigned long uidv, nextuid;
int	version;
struct	stat	stat_buf;
DIR	*dirp;
struct	dirent *de;
unsigned long left_unseen=0;
int	dowritecache=0;

	if (is_sharedsubdir(dir))
		maildir_shared_sync(dir);


	/* Step 0 - purge the tmp directory */

	maildir_purgetmp(dir);

	dbfilepath=malloc(strlen(dir)+sizeof("/" IMAPDB));
	if (!dbfilepath)	write_error_exit(0);
	strcat(strcpy(dbfilepath, dir), "/" IMAPDB);

	/*
	** We may need to rebuild the UID cache file.  Create the new cache
	** file in the tmp subdirectory.
	*/

	{
		char	uniqbuf[80];
		static  unsigned tmpuniqcnt=0;
		struct maildir_tmpcreate_info createInfo;
		int fd;

		maildir_tmpcreate_init(&createInfo);

		createInfo.maildir=dir;
		createInfo.hostname=getenv("HOSTNAME");
		sprintf(uniqbuf, "imapuid_%u", tmpuniqcnt++);
		createInfo.uniq=uniqbuf;
		createInfo.doordie=1;

		if ((fd=maildir_tmpcreate_fd(&createInfo)) < 0)
		{
			write_error_exit(0);
		}
		close(fd);

		newdbfilepath=createInfo.tmpname;
		createInfo.tmpname=NULL;
		maildir_tmpcreate_free(&createInfo);
	}

	/* Step 1 - read the cache file */

	if ((fp=fopen(dbfilepath, "r")) != 0 &&
		(p=readline(0, fp)) != 0 &&
		sscanf(p, "%d %lu %lu", &version, &uidv, &nextuid) == 3 &&
		version == IMAPDBVERSION)
	{
		while ((p=readline(0, fp)) != 0)
		{
		char	*q=strchr(p, ' ');
		unsigned long uid;
		struct	tempinfo	*newtmpl;

			if (!q)	continue;
			*q++=0;
			if (sscanf(p, "%lu", &uid) != 1)	continue;
			if ((newtmpl=(struct tempinfo *)
				malloc(sizeof(struct tempinfo))) == 0
				|| (newtmpl->filename=strdup(q)) == 0)
			{
				unlink(newdbfilepath);
				write_error_exit(0);
			}
			newtmpl->next=tempinfo_list;
			tempinfo_list=newtmpl;
			newtmpl->uid=uid;
			newtmpl->found=0;
			newtmpl->isrecent=0;
			++tempinfo_cnt;
		}
		fclose(fp);
		fp=0;
	}
	else if(!ro)
	{

	/* First time - create the cache file */

		if (fp)	fclose(fp);
		nextuid=1;
		if ((fp=fopen(newdbfilepath, "w")) == 0 ||
			fstat(fileno(fp), &stat_buf) != 0)
		{
			if (fp)	fclose(fp);
			imapscanfail(newdbfilepath);

			/* bk: ignore error */
			unlink(newdbfilepath);
			unlink(dbfilepath);
			fp = 0;
			/*
			free(dbfilepath);
			unlink(newdbfilepath);
			free(newdbfilepath);
			return (-1);
			*/
		}
		uidv=stat_buf.st_mtime;
		dowritecache=1;
	}
	else
	{
		nextuid=1;
		time(&uidv);
	}

	while (uidplus)
	{
		struct	tempinfo	*newtmpl;

		maildir_movetmpnew(uidplus->tmpfilename,
				   uidplus->curfilename);

		if ((newtmpl=(struct tempinfo *)
		     malloc(sizeof(struct tempinfo))) == 0
		    || (newtmpl->filename=strdup(strrchr(uidplus->curfilename,
							 '/')+1)) == 0)
		{
			unlink(newdbfilepath);
			write_error_exit(0);
		}

		if ((p=strrchr(newtmpl->filename, MDIRSEP[0])) != 0)
			*p=0;

		newtmpl->next=tempinfo_list;
		tempinfo_list=newtmpl;
		newtmpl->uid=nextuid;
		uidplus->uid=nextuid;
		nextuid++;
		newtmpl->found=0;
		newtmpl->isrecent=0;
		++tempinfo_cnt;

		uidplus=uidplus->next;
		dowritecache=1;
	}

	if (!fp && (fp=fopen(newdbfilepath, "w")) == 0)
	{
		imapscanfail(newdbfilepath);

		/* bk: ignore error */
		unlink(newdbfilepath);
		unlink(dbfilepath);
		/*
		free(dbfilepath);
		unlink(newdbfilepath);
		free(newdbfilepath);
		while (tempinfo_list)
		{
			tempinfoptr=tempinfo_list;
			tempinfo_list=tempinfoptr->next;
			free(tempinfoptr->filename);
			free(tempinfoptr);
		}
		return (-1);
		*/
	}

	/*
	** Convert the link list of cached files to an array, then
	** sort it by filename.
	*/

	if ((tempinfo_array=(struct tempinfo **)malloc(
		(tempinfo_cnt+1)*sizeof(struct tempinfo *))) == 0)
	{
		unlink(newdbfilepath);
		write_error_exit(0);
	}

	for (i=0, tempinfoptr=tempinfo_list; tempinfoptr;
		tempinfoptr=tempinfoptr->next, i++)
		tempinfo_array[i]=tempinfoptr;

	if (tempinfo_cnt)
		qsort(tempinfo_array, tempinfo_cnt,
			sizeof(tempinfo_array[0]),
			( int (*)(const void *, const void *))
				&sort_by_filename);

	/* Step 2 - read maildir/cur.  Search the array.  Mark found files. */

	p=imapscan_namedir(dir, "cur");
	dirp=opendir(p);
	free(p);
	while (dirp && (de=readdir(dirp)) != 0)
	{
	int	rc;
	struct	tempinfo	*newtmpl;

		if (de->d_name[0] == '.')	continue;

		p=my_strdup(de->d_name);

		/* IMAPDB doesn't store the filename flags, so strip them */
		q=strrchr(p, MDIRSEP[0]);
		if (q)	*q=0;
		rc=search_by_filename(tempinfo_array, tempinfo_cnt, &i, p);
		if (q)	*q=MDIRSEP[0];
		if (rc == 0)
		{
			tempinfo_array[i]->found=1;
			free(tempinfo_array[i]->filename);
			tempinfo_array[i]->filename=p;
				/* Keep the full filename */
			continue;
		}

		if ((newtmpl=(struct tempinfo *)
			malloc(sizeof(struct tempinfo))) == 0)
		{
			unlink(newdbfilepath);
			write_error_exit(0);
		}
		newtmpl->filename=p;
		newtmpl->next=newtempinfo_list;
		newtmpl->found=0;
		newtmpl->isrecent=1;
		newtempinfo_list=newtmpl;
		dowritecache=1;
	}
	if (dirp)	closedir(dirp);

	/* Step 3 - purge messages that no longer exist in the maildir */

	free(tempinfo_array);

	for (tempinfo_array= &tempinfo_list; *tempinfo_array; )
	{
		if ( (*tempinfo_array)->found )
		{
			tempinfo_array= & (*tempinfo_array)->next;
			continue;
		}
		tempinfoptr= *tempinfo_array;
		*tempinfo_array=tempinfoptr->next;
		free(tempinfoptr->filename);
		free(tempinfoptr);
		--tempinfo_cnt;
		dowritecache=1;
	}

	/* Step 4 - add messages in cur that are not in the cache file */

	while (newtempinfo_list)
	{
		tempinfoptr=newtempinfo_list;
		newtempinfo_list=tempinfoptr->next;

		tempinfoptr->next=tempinfo_list;
		tempinfo_list=tempinfoptr;
		++tempinfo_cnt;
	}

	/* Step 5 - read maildir/new.  */

	p=imapscan_namedir(dir, "new");
	dirp=opendir(p);
	while (dirp && (de=readdir(dirp)) != 0)
	{
	struct	tempinfo	*newtmpl;
	char	*newname, *curname;
	char	*z;

		if (de->d_name[0] == '.')	continue;
		if (leavenew)
		{
			++left_unseen;
			continue;
		}

		z=de->d_name;

		newname=imapscan_namedir(p, z);
		curname=malloc(strlen(newname)+sizeof(MDIRSEP "2,"));
		if (!curname)
		{
			unlink(newdbfilepath);
			write_error_exit(0);
		}
		strcpy(curname, newname);
		z=strrchr(curname, '/');

		memcpy(z-3, "cur", 3); /* Mother of all hacks */
		if (strchr(z, MDIRSEP[0]) == 0)
			strcat(z, MDIRSEP "2,");

		if (rename(newname, curname))
		{
			free(newname);
			free(curname);
			continue;
		}

		if ((newtmpl=(struct tempinfo *)
			malloc(sizeof(struct tempinfo))) == 0)
		{
			unlink(newdbfilepath);
			write_error_exit(0);
		}
		newtmpl->filename=my_strdup(z+1);
		newtmpl->next=tempinfo_list;
		tempinfo_list=newtmpl;
		++tempinfo_cnt;
		newtmpl->found=0;
		newtmpl->isrecent=1;
		free(newname);
		free(curname);
		dowritecache=1;
	}
	free(p);
	if (dirp)	closedir(dirp);

	/*
	** Step 6: sort existing messages by UIDs, new messages will
	** sort after all messages with UIDs, and new messages are
	** sorted by filename, so that they end up roughly in the order
	** they were received.
	*/

	if ((tempinfo_array=(struct tempinfo **)malloc(
		(tempinfo_cnt+1)*sizeof(struct tempinfo *))) == 0)
	{
		unlink(newdbfilepath);
		write_error_exit(0);
	}

	for (i=0, tempinfoptr=tempinfo_list; tempinfoptr;
		tempinfoptr=tempinfoptr->next, i++)
		tempinfo_array[i]=tempinfoptr;

	if (tempinfo_cnt)
		qsort(tempinfo_array, tempinfo_cnt,
			sizeof(tempinfo_array[0]),
			( int (*)(const void *, const void *))
				&sort_by_filename_status);

	/* Assign new UIDs */

	for (i=0; i<tempinfo_cnt; i++)
		if ( !tempinfo_array[i]->found )
		{
			tempinfo_array[i]->uid= nextuid++;
			dowritecache=1;
		}

	/* bk: ignore if failed to open file */
	if (!ro && dowritecache && fp)
	{
	/* Step 7 - write out the new cache file */

		version=IMAPDBVERSION;
		fprintf(fp, "%d %lu %lu\n", version, uidv, nextuid);

		for (i=0; i<tempinfo_cnt; i++)
		{
			q=strrchr(tempinfo_array[i]->filename, MDIRSEP[0]);
			if (q)	*q=0;
			fprintf(fp, "%lu %s\n", tempinfo_array[i]->uid,
				tempinfo_array[i]->filename);
			if (q)	*q=MDIRSEP[0];
		}
		if (fflush(fp) || ferror(fp) || fclose(fp))
		{
			imapscanfail(dir);
			fclose(fp);
			/* bk: ignore if failed */
			unlink(newdbfilepath);
			unlink(dbfilepath);
			/*
			free(tempinfo_array);
			free(dbfilepath);
			unlink(newdbfilepath);
			free(newdbfilepath);
			while (tempinfo_list)
			{
				tempinfoptr=tempinfo_list;
				tempinfo_list=tempinfoptr->next;
				free(tempinfoptr->filename);
				free(tempinfoptr);
			}
			return (-1);
			*/
		}
		/* bk */
		else

		rename(newdbfilepath, dbfilepath);
	}
	else
	{
		if (fp)
			fclose(fp);
		unlink(newdbfilepath);
	}
	free(dbfilepath);
	free(newdbfilepath);

	/* Step 8 - create the final scaninfo array */

	scaninfo->msgs=0;
	if (tempinfo_cnt && (scaninfo->msgs=(struct imapscanmessageinfo *)
		malloc(tempinfo_cnt * sizeof(*scaninfo->msgs))) == 0)
		write_error_exit(0);
	scaninfo->nmessages=tempinfo_cnt;
	scaninfo->uidv=uidv;
	scaninfo->left_unseen=left_unseen;
	scaninfo->nextuid=nextuid+left_unseen;

	for (i=0; i<tempinfo_cnt; i++)
	{
		scaninfo->msgs[i].uid=tempinfo_array[i]->uid;
		scaninfo->msgs[i].filename=tempinfo_array[i]->filename;
		scaninfo->msgs[i].recentflag=tempinfo_array[i]->isrecent;
		scaninfo->msgs[i].changedflags=0;

		free(tempinfo_array[i]);
	}
	free(tempinfo_array);

	return (0);
}

static int try_maildir_open(const char *dir, struct imapscanmessageinfo *n)
{
int	fd;
char	*filename=maildir_filename(dir, 0, n->filename);
char	*p;

	if (!filename)
	{
		return (0);
	}

	p=strrchr(filename, '/')+1;

	if (strcmp(p, n->filename))
	{
		n->changedflags=1;
		free(n->filename);
		n->filename=malloc(strlen(p)+1);
		if (!n->filename)	write_error_exit(0);
		strcpy(n->filename, p);
	}

	fd=maildir_semisafeopen(filename, O_RDONLY, 0);
	free(filename);
	return (fd);
}

int imapscan_openfile(const char *dir, struct imapscaninfo *i, unsigned j)
{
struct imapscanmessageinfo *n;

	if (j >= i->nmessages)
	{
		errno=EINVAL;
		return (-1);
	}

	n=i->msgs+j;

	return (try_maildir_open(dir, n));
}

void imapscan_free(struct imapscaninfo *i)
{
	unsigned	n;

	if (i->watcher)
	{
		maildirwatch_free(i->watcher);
		i->watcher=NULL;
	}

	if (i->msgs == 0)	return;

	for (n=0; n<i->nmessages; n++)
	{
		if (i->msgs[n].filename)
			free(i->msgs[n].filename);
	}
	free(i->msgs);
	i->msgs=0;
}
