/*
** Copyright 1998 - 1999 Double Precision, Inc.
** See COPYING for distribution information.
*/

#if	HAVE_CONFIG_H
#include	"config.h"
#endif

#include	<stdio.h>
#include	<string.h>
#include	<stdlib.h>
#include	<ctype.h>
#include	<time.h>
#if	HAVE_UNISTD_H
#include	<unistd.h>
#endif
#include	<sys/types.h>
#include	<sys/stat.h>
#include	"rfc822/rfc822.h"
#include	"rfc2045/rfc2045.h"
#include	"numlib/numlib.h"
#include	"searchinfo.h"
#include	"imapwrite.h"
#include	"imaptoken.h"
#include	"imapscanclient.h"

static const char rcsid[]="$Id: search.c,v 1.5 2000/06/09 12:29:14 mrsam Exp $";

extern time_t rfc822_parsedt(const char *);
extern struct imapscaninfo current_mailbox_info;
extern char *current_mailbox;

extern int get_flagname(const char *, struct imapflags *);
extern void get_message_flags( struct imapscanmessageinfo *,
	char *, struct imapflags *);


static void fill_search_quick(struct searchinfo *,
	unsigned long, struct imapflags *, struct stat *);

static void fill_search_header(struct searchinfo *, struct rfc2045 *, FILE *,
	struct imapscanmessageinfo *);

static void fill_search_body(struct searchinfo *,
	struct rfc2045 *, FILE *, struct imapscanmessageinfo *);

static int search_evaluate(struct searchinfo *);

static void searcherror(const char *errmsg,
                struct imapscanmessageinfo *mi)
{
        fprintf(stderr, "IMAP SEARCH ERROR: %s, uid=%u, filename=%s\n",
                errmsg, (unsigned)getuid(), mi->filename);
}

static void search_callback(struct searchinfo *, struct searchinfo *, int,
	unsigned long, void *);

/*
**	search_internal() does the main heavylifting of searching the
**	maildir for qualifying messages.  It calls a callback function
**	when a matching message is found.
**
**	For a plain SEARCH, the callback function merely prints the message
**	number.
*/

void dosearch(struct searchinfo *si, struct searchinfo *sihead, int isuid)
{
	search_internal(si, sihead, isuid, search_callback, 0);
}

static void search_callback(struct searchinfo *si, struct searchinfo *sihead,
	int isuid, unsigned long i, void *dummy)
{
	writes(" ");
	writen(isuid ? current_mailbox_info.msgs[i].uid:i+1);
}

void search_internal(struct searchinfo *si, struct searchinfo *sihead,
	int isuid, void (*callback_func)(struct searchinfo *,
		struct searchinfo *, int, unsigned long, void *), void *voidarg)
{
unsigned long i;
struct	searchinfo *p;
struct	imapflags	flags;
int	fd;
FILE	*fp;
struct	stat	stat_buf;
int	rc;

	for (i=0; i<current_mailbox_info.nmessages; i++)
	{
		for (p=sihead; p; p=p->next)
			p->value= -1;	/* Search result unknown */

		fd=imapscan_openfile(current_mailbox,
			&current_mailbox_info, i);
		if (fd < 0)	continue;

		if ((fp=fdopen(fd, "r")) == 0)
			write_error_exit(0);

		get_message_flags(current_mailbox_info.msgs+i, 0, &flags);
		if (fstat(fileno(fp), &stat_buf))
		{
			fclose(fp);
			continue;
		}

		/* First, see if non-content search will be sufficient */

		for (p=sihead; p; p=p->next)
			fill_search_quick(p, i, &flags, &stat_buf);

		if ((rc=search_evaluate(si)) < 0)
		{
		struct	rfc2045 *rfcp=rfc2045_fromfp(fp);

			/* No, search the headers then */

			fill_search_header(sihead, rfcp, fp,
				current_mailbox_info.msgs+i);
			rc=search_evaluate(si);

			if (rc < 0)
			{
				/* Ok, search message contents */

				fill_search_body(si, rfcp, fp,
					current_mailbox_info.msgs+i);
				rc=search_evaluate(si);
			}
			rfc2045_free(rfcp);
		}

		if (rc > 0)
		{
			(*callback_func)(si, sihead, isuid, i, voidarg);
		}
		fclose(fp);
		close(fd);
	}
}

/* Check if the given index is included in the specified message set */

static int is_in_set(const char *msgset, unsigned long n)
{
unsigned long i, j;

	while (isdigit((int)(unsigned char)*msgset))
	{
		i=0;
		while (isdigit((int)(unsigned char)*msgset))
		{
			i=i*10 + (*msgset++-'0');
		}
		if (*msgset != ':')
			j=i;
		else
		{
			j=0;
			++msgset;
			while (isdigit((int)(unsigned char)*msgset))
			{
				j=j*10 + (*msgset++-'0');
			}
		}
		if (n >= i && n <= j)	return (1);
		if (*msgset == 0 || *msgset++ != ',')	break;
	}
	return (0);
}

/*
** Search date comparisons compare the dates only, not the time.
** We convert all timestamps to midnight GMT on their respective dates.
** Use convenient RFC822 functions for that purpose.
*/

static time_t decode_date(char *p)
{
char	*s=malloc(strlen(p)+sizeof(" 00:00:00"));
unsigned        i;
time_t	t;

	if (!s)	write_error_exit(0);

        /* Convert to format rfc822_parsedt likes */
 
        for (i=1; p[i] != ' '; i++)
        {
                if (!p[i])	break;
        }
	memcpy(s, p, i);
	strcpy(s+i, " 00:00:00");
	while (i)
	{
		if (s[--i] == '-')
			s[i]=' ';
	}

	t=rfc822_parsedt(s);
	free(s);
	return (t);
}

/* Given a time_t that falls on, say, 3-Aug-1999 9:50:43 local time,
** calculate the time_t for midnight 3-Aug-1999 UTC.  Search date comparisons
** are done against midnight UTCs */

static time_t timestamp_to_day(time_t t)
{
char	buf1[60], buf2[80];

	rfc822_mkdate_buf(t, buf1);	/* Converts to local time */
	(void)strtok(buf1, " ");	/* Skip weekday */
	strcpy(buf2, strtok(0, " "));
	strcat(buf2, " ");
	strcat(buf2, strtok(0, " "));
	strcat(buf2, " ");
	strcat(buf2, strtok(0, " "));
	strcat(buf2, " 00:00:00");
	return (rfc822_parsedt(buf2));
}

static char *timestamp_for_sorting(time_t t)
{
struct tm *tm=localtime(&t);
char	buf[200];

	buf[0]=0;
	if ( strftime(buf, sizeof(buf), "%Y.%m.%d.%H.%M.%S", tm) == 0)
		buf[0]=0;
	return (my_strdup(buf));
}

/* Evaluate non-content search nodes */

static void fill_search_quick(struct searchinfo *p,
	unsigned long msgnum, struct imapflags *flags, struct stat *stat_buf)
{
	switch (p->type)	{
	case search_messageset:
		if (is_in_set(p->as, msgnum+1))
			p->value=1;
		else
			p->value=0;
		break;
	case search_all:
		p->value=1;
		break;
	case search_keyword:
		{
		struct imapflags f;

			memset(&f, 0, sizeof(f));
			p->value=0;
			get_flagname(p->as, &f);
			if (strcmp(p->as, "\\RECENT") == 0 &&
				current_mailbox_info.msgs[msgnum].recentflag)
				p->value=1;
			if (f.seen && flags->seen)
				p->value=1;
			if (f.answered && flags->answered)
				p->value=1;
			if (f.deleted && flags->deleted)
				p->value=1;
			if (f.flagged && flags->flagged)
				p->value=1;
		}
		break;
	case search_before:
		p->value=0;
		{
		time_t t=decode_date(p->as);

			if (t && timestamp_to_day(stat_buf->st_mtime) < t)
				p->value=1;
		}
		break;
	case search_since:
		p->value=0;
		{
		time_t t=decode_date(p->as);

			if (t && timestamp_to_day(stat_buf->st_mtime) >= t)
				p->value=1;
		}
		break;
	case search_on:
		p->value=0;
		{
		time_t t=decode_date(p->as);

			if (t && timestamp_to_day(stat_buf->st_mtime) == t)
				p->value=1;
		}
		break;
	case search_smaller:
		p->value=0;
		{
		unsigned long n;

			if (sscanf(p->as, "%lu", &n) > 0 &&
				stat_buf->st_size < n)
				p->value=1;
		}
		break;
	case search_larger:
		p->value=0;
		{
		unsigned long n;

			if (sscanf(p->as, "%lu", &n) > 0 &&
				stat_buf->st_size > n)
				p->value=1;
		}
		break;
	case search_uid:
		if (is_in_set(p->as, current_mailbox_info.msgs[msgnum].uid))
			p->value=1;
		break;
	case search_orderedsubj:
	case search_arrival:
	case search_cc:
	case search_date:
	case search_from:
	case search_reverse:
	case search_size:
	case search_to:

		/* DUMMY nodes for SORT/THREAD.  Make sure that the
		** dummy node is CLEARed */

		if (p->as)
		{
			free(p->as);
			p->as=0;
		}

		switch (p->type)	{
		case search_arrival:
			p->as=timestamp_for_sorting(stat_buf->st_mtime);
			p->value=1;
			break;
		case search_size:
			{
			char	buf[NUMBUFSIZE], buf2[NUMBUFSIZE];
			char *q;

				str_size_t(stat_buf->st_size, buf);
				sprintf(buf2, "%*s", sizeof(buf2)-1, buf);
				for (q=buf2; *q == ' '; *q++='0')
					;
				p->as=my_strdup(buf2);
				p->value=1;
			}
			break;
		case search_reverse:
			p->value=1;
			break;
		default:
			break;
		}
		break;
	default:
		break;
	}
}

/* Evaluate search results.  Returns: 0 - false, 1 - true, -1 - unknown
** (partial search on message metadata, like size or flags, in hopes of
** preventing a search
** of message contents).
*/

static int search_evaluate(struct searchinfo *si)
{
int	rc, rc2;

	switch (si->type)	{
	case search_orderedsubj:	/* DUMMY for THREAD and SORT */
        case search_arrival:
        case search_cc:
        case search_date:
        case search_from:
        case search_reverse:
        case search_size:
        case search_to:
		if (si->value < 0)	return (-1);
		return (search_evaluate(si->a));
	case search_not:
		rc=search_evaluate(si->a);
		if (rc >= 0)	rc= 1-rc;
		break;
	case search_and:
		rc=search_evaluate(si->a);
		rc2=search_evaluate(si->b);

		rc=  rc > 0 && rc2 > 0 ? 1:
			rc == 0 || rc2 == 0 ? 0:-1;
		break;
	case search_or:
		rc=search_evaluate(si->a);
		rc2=search_evaluate(si->b);

		rc=  rc > 0 || rc2 > 0 ? 1:
			rc == 0 && rc2 == 0 ? 0:-1;
		break;
	default:
		rc=si->value;
		break;
	}
	return (rc);
}

/* Remove artifacts from the subject header */

static void stripsubj(char *s)
{
char	*p;
char	*q;

	for (p=s; *p; p++)
		;
	while (p > s)
	{
		if ( isspace((int)(unsigned char)p[-1]))
		{
			--p;
			continue;
		}
		if (p-s >= 3 && strcmp(p-3, "(FWD)") == 0)
		{
			p -= 3;
			continue;
		}
		break;
	}
	*p=0;

	for (p=s; *p; )
	{
		for (;;)
		{
			if (isspace((int)(unsigned char)*p))
			{
				++p;
				continue;
			}
			if (strncmp(p, "RE", 2) == 0)
			{
				q=p+2;
			}
			else if (strncmp(p, "FWD", 3) == 0)
			{
				q=p+3;
			}
			else if (strncmp(p, "FW", 2) == 0)
			{
				q=p+2;
			}
			else	break;

			if (*q == '[')
			{
				++q;
				if (!isdigit((int)(unsigned char)*q))
					break;
				do
				{
					++q;
				} while (isdigit((int)(unsigned char)*q));
				if (*q != ']')	break;
				++q;
			}
			if (*q++ != ':')	break;

			while (*q && isspace((int)(unsigned char)*q))
				++q;
			p=q;
		}

		if (*p != '[')	break;
		for (q=p; *q; q++)
			if (*q == ']')
				break;
		if (*q != ']')	break;
		++q;
		while (*q && isspace((int)(unsigned char)*q))
			++q;
		p=q;
	}

	q=s;
	while ( (*q++ = *p++) != 0)
		;
}

/* ------- header search -------- */

static void fill_search_header_recursive(struct searchinfo *,
	struct rfc2045 *, FILE *, struct imapscanmessageinfo *);

static void fill_search_header(struct searchinfo *si,
	struct rfc2045 *rfcp, FILE *fp, struct imapscanmessageinfo *mi)
{
off_t start_pos, end_pos, start_body;
off_t nlines, nbodylines;
struct searchinfo *sip;

	rfc2045_mimepos(rfcp, &start_pos, &end_pos, &start_body,
		&nlines, &nbodylines);

	if (fseek(fp, start_pos, SEEK_SET) < 0)
	{
		searcherror("fseek failed", mi);
		return;
	}
	while (start_pos < start_body)
	{
	char	*p=readline(0, fp);
	char	*q=0;
	char	*r;
	int	isdate;

		/* Collect multiline header into the readline buffer */

		while (p)
		{
		int	c;

			start_pos=ftell(fp);
			if (start_pos >= start_body)	break;

			c=getc(fp);
			if (c != EOF)	ungetc(c, fp);
			if (c == '\n' || !isspace((int)(unsigned char)c))
				break;
			p=readline(strlen(p), fp);
		}
		if (!p)	break;

		/* search_text applies to headers too */

		for (sip=si; sip; sip=sip->next)
			if (sip->type == search_text && sip->value <= 0)
				search_reset(sip);

		for (q=p; *q; q++)
		{
			*q=toupper((int)(unsigned char)*q);
			for (sip=si; sip; sip=sip->next)
				if (sip->type == search_text &&
					sip->value <= 0)
					search_step(sip, *q);
		}

		if ((q=strchr(p, ':')) != 0)
		{
			*q++=0;
			while (*q && isspace((int)(unsigned char)*q))
				++q;
		}	else q="";

		isdate=strcmp(p, "DATE");
		for (sip=si; sip; sip=sip->next)
		{
			if (sip->type == search_text)
			{
				if (sip->value <= 0 && search_found(sip))
					sip->value=1;
			}

			if (isdate == 0) switch (sip->type)	{
			time_t given_time, msg_time;

			case search_sentbefore:
			case search_sentsince:
			case search_senton:

				if (sip->value > 0)	break;

				given_time=decode_date(sip->as);
				msg_time=rfc822_parsedt(q);

				if (given_time == 0 || msg_time == 0)
					break;

				msg_time=timestamp_to_day(msg_time);
				sip->value=0;
				if ((sip->type == search_sentbefore &&
					msg_time < given_time) ||
					(sip->type == search_sentsince&&
						msg_time>=given_time)||
					(sip->type == search_senton &&
						msg_time == given_time))
					sip->value=1;
				break;
			default:
				break;
			}

			if (sip->type == search_cc && strcmp(p, "CC") == 0 &&
				sip->as == 0)
			{
			struct rfc822t *t=rfc822t_alloc(q, 0);
			struct rfc822a *a=t ? rfc822a_alloc(t):0;

				sip->as= a && a->naddrs > 0 ?
					rfc822_getaddr(a, 0):0;
				if (a)	rfc822a_free(a);
				if (t)	rfc822t_free(t);
			}

			if (sip->type == search_date && strcmp(p, "DATE") == 0
				&& sip->as == 0)
			{
			time_t msg_time=rfc822_parsedt(q);

				sip->as=timestamp_for_sorting(msg_time);
			}

			if (sip->type == search_from && strcmp(p, "FROM") == 0
				&& sip->as == 0)
			{
			struct rfc822t *t=rfc822t_alloc(q, 0);
			struct rfc822a *a=t ? rfc822a_alloc(t):0;

				sip->as= a && a->naddrs > 0 ?
					rfc822_getaddr(a, 0):0;
				if (a)	rfc822a_free(a);
				if (t)	rfc822t_free(t);
			}

			if (sip->type == search_to && strcmp(p, "TO") == 0 &&
				sip->as == 0)
			{
			struct rfc822t *t=rfc822t_alloc(q, 0);
			struct rfc822a *a=t ? rfc822a_alloc(t):0;

				sip->as= a && a->naddrs > 0 ?
					rfc822_getaddr(a, 0):0;
				if (a)	rfc822a_free(a);
				if (t)	rfc822t_free(t);
			}

			switch (sip->type)	{
			case search_cc:
			case search_date:
			case search_from:
			case search_to:
				sip->value=1;
				break;
			default:
				break;
			}

			if (sip->type == search_orderedsubj &&
				strcmp(p, "SUBJECT") == 0 && sip->as == 0)
			{
			/* Dummy node used to suck in the Subject:
			** header for the THREAD or SORT command.
			*/
				sip->as=my_strdup(q);
				sip->value=1;
				stripsubj(sip->as);
			}

			if (sip->type != search_header ||
				sip->value > 0)	continue;

			if (strcmp(sip->as, p) == 0)
			{
				sip->value=0;
				search_reset(sip);
				for (r=q; *r; r++)
					search_step(sip, *r);
				if (search_found(sip))
					sip->value=1;
			}
		}
	}
	fill_search_header_recursive(si, rfcp, fp, mi);
}

/* Scan the headers of attached messages too! */

static void fill_search_header_recursive(struct searchinfo *si,
	struct rfc2045 *rfcp, FILE *fp, struct imapscanmessageinfo *mi)
{
	if (!rfcp)	return;

	if (rfcp->firstpart && rfcp->firstpart->next == 0 &&
		!rfcp->firstpart->isdummy)
	{
		fill_search_header(si, rfcp->firstpart, fp, mi);
		return;
	}

	rfcp=rfcp->firstpart;

	while (rfcp)
	{
		if (!rfcp->isdummy)
			fill_search_header_recursive(si, rfcp, fp, mi);
		rfcp=rfcp->next;
	}
}

static int search_body_func(const char *, size_t, void *);

/* Last straw - search message contents */

static void fill_search_body(struct searchinfo *si,
	struct rfc2045 *rfcp, FILE *fp, struct imapscanmessageinfo *mi)
{
struct searchinfo *sip;
off_t start_pos, end_pos, start_body;
off_t nlines, nbodylines;
const char *content_type_s;
const char *content_transfer_encoding_s;
const char *charset_s;
char	buf[BUFSIZ];
int	rc;

	for (sip=si; sip; sip=sip->next)
		if ((sip->type == search_text || sip->type == search_body)
			&& sip->value <= 0)
			break;

	if (sip == 0)	return;	/* Nothing to search any more */

	if (rfcp->isdummy)	return;	/* stub RFC2045 section */

	if (rfcp->firstpart)	/* Search multiline contents */
	{
		for (rfcp=rfcp->firstpart; rfcp; rfcp=rfcp->next)
			fill_search_body(si, rfcp, fp, mi);
		return;
	}

	while (sip)
	{
		if ((sip->type == search_text || sip->type == search_body)
			&& sip->value <= 0)
			search_reset(sip);
		sip=sip->next;
	}
	rfc2045_mimeinfo(rfcp, &content_type_s,
		&content_transfer_encoding_s, &charset_s);
	rfc2045_mimepos(rfcp, &start_pos, &end_pos, &start_body,
		&nlines, &nbodylines);

	if (fseek(fp, start_body, SEEK_SET) < 0)
	{
		searcherror("fseek failed", mi);
		return;
	}
	rfc2045_cdecode_start(rfcp, search_body_func, si);

	rc=0;
	while (start_body < end_pos)
	{
	unsigned n=sizeof(buf);

		if (n > end_pos - start_body)
			n=end_pos - start_body;

		rc=fread(buf, 1, n, fp);
		if (rc <= 0)
		{
			searcherror("fread failed", mi);
			break;
		}
		start_body += rc;
		rc=rfc2045_cdecode(rfcp, buf, rc);
		if (rc)	break;
	}

	if (rc == 0)	(void)rfc2045_cdecode_end(rfcp);
	for (sip=si; sip; sip=sip->next)
		if ((sip->type == search_text || sip->type == search_body)
			&& sip->value <= 0)
		{
			if (search_found(sip))
				sip->value=1;
			else
				sip->value=0;
		}
}

static int search_body_func(const char *ptr, size_t cnt, void *voidptr)
{
struct searchinfo *si=(struct searchinfo *)voidptr;
size_t	i;

	while (si)
	{
		if ((si->type == search_text || si->type == search_body)
			&& si->value <= 0)
			break;
		si=si->next;
	}
	if (si == 0)	return (1);
	do
	{
		for (i=0; i<cnt; i++)
		{
		unsigned char c=ptr[i];

			if (c == '\n')
			{
				if (search_found(si))
				{
					si->value=1;
					break;
				}
				si->value=0;
				search_reset(si);
			}
			else
			{
				c=toupper((int)c);
				search_step(si, c);
			}
		}

		while ( (si=si->next) != 0)
		{
			if ((si->type == search_text || si->type == search_body)
				&& si->value <= 0)
				break;
		}
	} while (si);
	return (0);
}
