/*  $Id: UserController.h,v 1.10 2009/02/27 03:32:37 sarrazip Exp $
    UserController.h - Keyboard-based game controller

    cosmosmash - A space rock shooting video game.
    Copyright (C) 2007-2008 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
    02111-1307, USA.
*/

#ifndef _H_UserController
#define _H_UserController

#include "Controller.h"

#include <flatzebra/KeyState.h>

#include <SDL.h>


class UserController : public Controller
{
public:

    // These button numbers are not portable accross different gamepads
    enum
    {
	A_BTN = 0,
	B_BTN = 1,
	START_BTN = 6,
	BACK_BTN = 10
    };

    UserController()
      : leftKS(SDLK_LEFT),
	rightKS(SDLK_RIGHT),
	startKS(SDLK_SPACE),
	hyperspaceKS(SDLK_DOWN),
	extendedKS(SDLK_e),
	pauseKS(SDLK_p),
	noshootKS(SDLK_LCTRL, SDLK_RCTRL),
	joystick(NULL),
	joystickButtons(),
	xAxis(0)
    {
	if (SDL_Init(SDL_INIT_JOYSTICK) == 0 && SDL_NumJoysticks() > 0)
	{
	    joystick = SDL_JoystickOpen(0);
	    if (joystick != NULL)
		joystickButtons.resize(size_t(SDL_JoystickNumButtons(joystick)), false);
	}
    }

    virtual ~UserController()
    {
	if (joystick != NULL)
	    SDL_JoystickClose(joystick);
    }

    virtual void processKey(SDLKey keysym, bool pressed)
    {
	leftKS.check(keysym, pressed);
	rightKS.check(keysym, pressed);
	hyperspaceKS.check(keysym, pressed);
	startKS.check(keysym, pressed);
	extendedKS.check(keysym, pressed);
	pauseKS.check(keysym, pressed);
	noshootKS.check(keysym, pressed);
    }

    /** Indicates if the player has just asked to start a game.
	Returns true if yes, and then 'isExtendedMode' indicates
	if the non-Intellivision extensions must be activated.
	Otherwise, returns false and 'isExtendedMode' is false.
    */
    virtual bool isStartRequested(bool &isExtendedMode)
    {
	bool ext = extendedKS.justPressed();
	bool start = (startKS.justPressed() || ext || jButtonJustPressed(START_BTN));
	isExtendedMode = start && ext;
	return start;
    }

    virtual bool isPauseRequested()
    {
	return pauseKS.justPressed() || jButtonJustPressed(BACK_BTN);
    }

    virtual bool isResumeRequested()
    {
	return pauseKS.justPressed() || jButtonJustPressed(START_BTN);
    }

    virtual bool isShootingActive()
    {
	return !noshootKS.isPressed() && !getJButton(A_BTN);
    }

    virtual bool isHyperspaceRequested()
    {
	return hyperspaceKS.justPressed() || jButtonJustPressed(B_BTN);
    }

    virtual bool isLeftMoveRequested()
    {
	return leftKS.isPressed() || xAxis < -AXIS_MIN;
    }

    virtual bool isRightMoveRequested()
    {
	return rightKS.isPressed() || xAxis > +AXIS_MIN;
    }

    /** Do any processing that needs to be done at the end of
	a tick.  In this case, we remember the state of some
	keys (pressed or not), so that justPressed() will work
	in the next tick.
	We also update the state of the joystick buttons and axes.
    */
    virtual void endOfTick()
    {
	startKS.remember();
	hyperspaceKS.remember();
	extendedKS.remember();
	pauseKS.remember();

	if (joystick != NULL)
	{
	    // Remember current joystick button states for next tick.
	    for (size_t i = joystickButtons.size(); i--; )
		joystickButtons[static_cast<size_t>(i)] = (getJButton(i) != 0);

	    Sint16 dpadAxis = SDL_JoystickGetAxis(joystick, 6);  // NOTE: specific to Xbox 360 USB controller
	    xAxis = (abs(dpadAxis) == 32767 ? dpadAxis : SDL_JoystickGetAxis(joystick, 0));

	    // Update joystick button states.
	    SDL_JoystickUpdate();
	}
    }

private:

    enum { AXIS_MIN = 16000 };  // mininum absolute value of axis

    bool getJButton(int buttonNumber) const
    {
	if (buttonNumber < 0 || buttonNumber >= static_cast<int>(joystickButtons.size()))
	    return false;
	return SDL_JoystickGetButton(joystick, buttonNumber);
    }

    bool jButtonJustPressed(int buttonNumber) const
    {
	return getJButton(buttonNumber) && !joystickButtons[static_cast<size_t>(buttonNumber)];
    }


private:
    flatzebra::KeyState leftKS;
    flatzebra::KeyState rightKS;
    flatzebra::KeyState startKS;
    flatzebra::KeyState hyperspaceKS;
    flatzebra::KeyState extendedKS;
    flatzebra::KeyState pauseKS;
    flatzebra::KeyState noshootKS;

    SDL_Joystick *joystick;
    std::vector<bool> joystickButtons;
    Sint16 xAxis;

};


#endif  /* _H_UserController */
