/*  $Id: sdlmain.cpp,v 1.14 2007/06/20 03:02:43 sarrazip Exp $
    sdlmain.cpp - main() function for Cosmosmash

    cosmosmash - A space rock shooting video game.
    Copyright (C) 2000-2004 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
    02111-1307, USA.
*/

#ifndef PACKAGE
    #define PACKAGE "cosmosmash"
#endif

#include "CosmoSmashEngine.h"

#include <stdlib.h>
#include <errno.h>
#include <time.h>

using namespace std;


static const int defaultMillesecondsPerFrame = 58;


#ifdef HAVE_GETOPT_LONG
#include <getopt.h>


static struct option knownOptions[] =
{
    { "help",              no_argument,       NULL, 'h' },
    { "version",           no_argument,       NULL, 'v' },
    { "initial-score",     required_argument, NULL, 'i' },
    { "ms-per-frame",      required_argument, NULL, 'm' },
    { "mirror-hyperspace", no_argument,       NULL, 'y' },
    { "no-sound",          no_argument,       NULL, 'n' },
    { "full-screen",       no_argument,       NULL, 'f' },
    { "auto",              no_argument,       NULL, 'a' },

    { NULL, 0, NULL, 0 }  // marks the end
};


static void
displayVersionNo()
{
    printf("%s %s\n", PACKAGE, VERSION);
}


static void
displayHelp()
{
    displayVersionNo();

    printf(
"\n"
"Copyright (C) 2001, 2002 Pierre Sarrazin <http://sarrazip.com/>\n"
"This program is free software; you may redistribute it under the terms of\n"
"the GNU General Public License.  This program has absolutely no warranty.\n"
    );

    printf(
"\n"
"Known options:\n"
"--help               Display this help page and exit\n"
"--version            Display this program's version number and exit\n"
"--initial-score=N    Start game at score N (N >= 0) [default=0]\n"
"--ms-per-frame=N     N milliseconds per animation frame [default=%d]\n"
"                     Min. 1, max. 1000.  50 ms means 20 frames per second\n"
"--mirror-hyperspace  Hyperspace pivots around the center of the ground\n"
"                     instead of begin random [defaut=random]\n"
"--no-sound           Disable sound effects [default is to enable them]\n"
"--full-screen        Attempt to use full screen mode [default is window mode]\n"
"--auto               Automatic mode (the computer plays the game itself)\n"
"\n"
    ,
    defaultMillesecondsPerFrame);
}
#endif  /* __GNUG__ */


int main(int argc, char *argv[])
{
    /*  Default game parameters:
    */
    long initScore = 0;
    int millisecondsPerFrame = defaultMillesecondsPerFrame;
    bool mirrorHyperspace = false;
    bool useSound = true;
    bool fullScreen = false;
    bool automaticPlay = false;


    #ifdef HAVE_GETOPT_LONG

    /*  Interpret the command-line options:
    */
    int c;
    do
    {
	c = getopt_long(argc, argv, "", knownOptions, NULL);

	switch (c)
	{
	    case EOF:
		break;  // nothing to do

	    case 'i':
		{
		    errno = 0;
		    long n = strtol(optarg, NULL, 10);
		    if (errno == ERANGE || n <= 0 || n >= 1000000)
		    {
			displayHelp();
			printf("\n\nInvalid initial score.\n");
			return EXIT_FAILURE;
		    }

		    initScore = n;
		}
		break;

	    case 'm':
		{
		    errno = 0;
		    long n = strtol(optarg, NULL, 10);
		    if (errno == ERANGE || n < 1 || n > 1000)
		    {
			displayHelp();
			printf("\n\nInvalid number of ms per frame.\n");
			return EXIT_FAILURE;
		    }

		    millisecondsPerFrame = int(n);
		}
		break;
	    
	    case 'y':
		mirrorHyperspace = true;
		break;

	    case 'n':
		useSound = false;
		break;

	    case 'f':
		fullScreen = true;
		break;

	    case 'a':
		automaticPlay = true;
		break;

	    case 'v':
		displayVersionNo();
		return EXIT_SUCCESS;

	    case 'h':
		displayHelp();
		return EXIT_SUCCESS;

	    default:
		displayHelp();
		return EXIT_FAILURE;
	}
    } while (c != EOF && c != '?');

    #endif  /* __GNUG__ */


    /*  Initialize the random number generator:
    */
    const char *s = getenv("SEED");
    unsigned int seed = (s != NULL ? atol(s) : time(NULL));
    #ifndef NDEBUG
    fprintf(stderr, "seed = %u\n", seed);
    fprintf(stderr, "init-score = %ld\n", initScore);
    fprintf(stderr, "ms-per-frame = %d\n", millisecondsPerFrame);
    #endif
    srand(seed);


    try
    {
	CosmoSmashEngine theCosmoSmashEngine(
				initScore, mirrorHyperspace, useSound,
				fullScreen, automaticPlay);
	theCosmoSmashEngine.run(millisecondsPerFrame);
    }
    catch (const string &s)
    {
	fprintf(stderr, "%s: engine init error:\n%s\n", PACKAGE, s.c_str());
	return EXIT_FAILURE;
    }
    catch (int e)
    {
	fprintf(stderr, "%s: init error # %d\n", PACKAGE, e);
	return EXIT_FAILURE;
    }

    return EXIT_SUCCESS;
}
