/*  $Id: CosmoSmashEngine.h,v 1.23 2007/06/20 03:02:43 sarrazip Exp $
    CosmoSmashEngine.h - A space rock shooting video game engine.

    cosmosmash - A space rock shooting video game.
    Copyright (C) 2000-2004 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
    02111-1307, USA.
*/

#ifndef _H_CosmoSmashEngine
#define _H_CosmoSmashEngine

#include "UserController.h"
#include "AutoController.h"

#include <flatzebra/GameEngine.h>
#include <flatzebra/Sprite.h>
#include <flatzebra/SoundMixer.h>
#include <flatzebra/KeyState.h>

#include <string>
#include <vector>
#include <memory>

using namespace flatzebra;


class CosmoSmashEngine : public GameEngine
/*  A space rock shooting video game engine.
*/
{
public:
    CosmoSmashEngine(long initialScore,
			bool mirrorHyperspace, bool useSound,
			bool fullScreen, bool automaticPlay)
    							throw(int, std::string);
    /*  See base class.

	Throws a string or an integer code if an error occurs.

	If 'mirrorHyperspace' is true, then hyperspace pivots around
	the center of the ground.  If 'mirrorHyperspace' is false,
	then hyperspace is random.
    */

    virtual ~CosmoSmashEngine();
    /*  Nothing interesting.
    */

    virtual void processKey(SDLKey keysym, bool pressed);
    /*  Inherited.
    */

    virtual bool tick();
    /*  Inherited.
    */

protected:

    // Inherited:
    bool animatePlayer();
    void animateAutomaticCharacters();
    void detectCollisions();
    void restoreBackground();
    void drawSprites();

private:

    typedef std::vector<Couple> CoupleList;

    enum PlayerBoostType
    {
	NO_BOOST, TRIPLE_BULLETS, CLOSER_BULLETS
    };


    bool paused;
    unsigned long tickCount;
    bool useSound;

    // SDL color values (see SDL_MapRGB()):
    Uint32 blackColor;
    Uint32 whiteColor;
    Uint32 greenColor;

    bool mirrorHyperspace;
    bool useGameExtensions;
    unsigned long playerBoostTicks;
    PlayerBoostType playerBoostType;

    PixmapArray playerPA;
    Sprite *playerSprite;

    PixmapArray baseBulletPA;
    SpriteList  baseBulletSprites;

    PixmapArray bigRockPA;
    PixmapArray smallRockPA;
    SpriteList rockSprites;  // contains big and small rock sprites

    PixmapArray bigSpinnerPA;
    PixmapArray smallSpinnerPA;
    SpriteList spinnerSprites;

    PixmapArray pulsarPA;
    SpriteList pulsarSprites;

    PixmapArray saucerPA;
    SpriteList saucerSprites;

    PixmapArray saucerBulletPA;
    SpriteList saucerBulletSprites;

    PixmapArray explosionPA;
    SpriteList  explosionSprites;

    PixmapArray questionPA;
    SpriteList questionSprites;

    int groundPos;     // y position of highest pixel row of ground
    int groundHeight;  // height of ground in pixels
    int mountainTopPos;  // y position of highest possible pixel for mountains

    std::auto_ptr<Controller> controller;
    KeyState quitKS;

    long   initScore;     // score at which a game starts (determines level)

    long   theScore;      // player's score in points
    long   thePeakScore;  // not displayed
    int    theLevel;      // level number (1--6) (depends on theScore)
    bool   updateScore;   // indicates if player's score needs to be rewritten
    Couple scoreAreaPos;

    int    numLives;  // number of player lives left
    bool   updateNumLives;
    Couple numLivesAreaPos;

    unsigned long timeBeforeNextPulsar;  // in ticks
    unsigned long timeBeforeNextSaucer;
    unsigned long timeBeforeNextQuestion;

    CoupleList starPositions;
    CoupleList mountainRangePositions;


    /*  SOUND EFFECTS:
    */
    SoundMixer *theSoundMixer;  // see method playSoundEffect()
    enum { NUM_ROCK_HIT_SOUNDS = 2 };
    SoundMixer::Chunk rockHitSounds[NUM_ROCK_HIT_SOUNDS];
    SoundMixer::Chunk playerHitSound;
    SoundMixer::Chunk pulsarBeepSound;
    SoundMixer::Chunk saucerShootingSound;
    SoundMixer::Chunk hyperspaceSound;
    SoundMixer::Chunk cadenceSound;


    /*  Implementation functions:
    */
    void fireBullet(Couple pos);
    void initializeSprites() throw(PixmapLoadError);
    void initializeMisc(bool useSound) throw(std::string);
    void setTimeBeforeNextPulsar();
    void setTimeBeforeNextSaucer();
    void setTimeBeforeNextQuestion();
    int getSpeedUp() const;
    void readKeyboardCommands();
    Sprite *createExplosionSprite(Couple posOfCenterOfExplosion,
				    int timeToLive);
    std::pair<Sprite *, Sprite *> splitBigRock(const Sprite &bigRock) const;
    void makeSpriteExplode(const Sprite *target,
			    const Sprite *bullet,
			    long points);
    void addToScore(long n);
    void addToNumLives(int n);
    void killSpritesInList(SpriteList &sl);
    void centerPlayerSprite();
    void repositionPlayerSprite(Couple newpos);
    void killAllAutomaticCharacters();
    void killPlayerBase();
    bool playerIsActive() const;
    std::pair<bool, size_t> animateFallingObjects(SpriteList &sprites);
    void animateSaucers();
    void displayPauseMessage(bool display);
    void displayStartMessage(bool display);
    void displayMessage(int row, const char *msg);
    bool detectCollisionsWithBullet(const Sprite &bullet,
				    SpriteList &slist,
				    int scoreOnCollision);
    bool detectCollisionsWithExplosions();
    void detectCollisionsBetweenFallingObjects(SpriteList &slist);
    void boostPlayer();
    Couple accelerate(Couple speed) const;
    void playShortExplosionSound();
    void playSoundEffect(SoundMixer::Chunk &wb);


    /*	Forbidden operations:
    */
    CosmoSmashEngine(const CosmoSmashEngine &);
    CosmoSmashEngine &operator = (const CosmoSmashEngine &);
};


#endif  /* _H_CosmoSmashEngine */
