/*  $Id: CosmoSmashEngine.cpp,v 1.30 2007/06/20 03:02:43 sarrazip Exp $
    CosmoSmashEngine.cpp - A space rock shooting video game engine.

    cosmosmash - A space rock shooting video game.
    Copyright (C) 2000-2004 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
    02111-1307, USA.
*/

#include "CosmoSmashEngine.h"

#include <flatzebra/PixmapLoadError.h>

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <math.h>

#include <map>
#include <vector>
#include <string>
#include <algorithm>
#include <iomanip>

#include "base.xpm"
#include "boosted-base.xpm"
#include "basebullet.xpm"
#include "bigrock0.xpm"
#include "bigrock1.xpm"
#include "bigrock2.xpm"
#include "bigrock3.xpm"
#include "bigrock4.xpm"
#include "bigrock5.xpm"
#include "smallrock0.xpm"
#include "smallrock1.xpm"
#include "smallrock2.xpm"
#include "smallrock3.xpm"
#include "smallrock4.xpm"
#include "smallrock5.xpm"
#include "explosion0.xpm"
#include "explosion1.xpm"
#include "bigspinner_backslash.xpm"
#include "bigspinner_horizontal.xpm"
#include "bigspinner_slash.xpm"
#include "bigspinner_vertical.xpm"
#include "smallspinner_backslash.xpm"
#include "smallspinner_horizontal.xpm"
#include "smallspinner_slash.xpm"
#include "smallspinner_vertical.xpm"
#include "pulsar_empty.xpm"
#include "pulsar_mid.xpm"
#include "pulsar_full.xpm"
#include "saucer0.xpm"
#include "saucer1.xpm"
#include "saucerbullet_empty.xpm"
#include "saucerbullet_mid.xpm"
#include "saucerbullet_full.xpm"
#include "question0.xpm"
#include "question1.xpm"

using namespace std;

#define __(s) (s)  /* Gettext-like macro */

#ifdef _MSC_VER
    #define snprintf _snprintf
#endif


#undef line  /* We define such a function later */


/*  Various constants:
*/

static const size_t
FRAMES_PER_SECOND = 20;

static const Couple theDrawingPixmapSize(675, 520);


static const int
PLAYER_SPEED = 15,
PLAYER_BULLET_SPEED = 36,
PULSAR_SPEED = 20,
SAUCER_SPEED = 8,
SAUCER_BULLET_SPEED = 20,
QUESTION_SPEED = 16;

static const int
SAUCER_Y_POS = 75,
NUM_INIT_LIVES = 5;

static const double TWO_PI = 8.0 * atan(1.0);

static const long
SCORE_LOST_BASE = -100,
SCORE_BIG_ROCK_LANDING = -5,
SCORE_SMALL_ROCK_LANDING = -10,
SCORE_BIG_ROCK = 10,
SCORE_SMALL_ROCK = 20,
SCORE_BIG_SPINNER = 40,
SCORE_SMALL_SPINNER = 80,
SCORE_PULSAR = 50,
SCORE_SAUCER_BULLET = 50,
SCORE_SAUCER = 100,
SCORE_QUESTION = 50;


// Defined as constants for efficiency:
static const size_t
NUM_SPINNER_IMAGES = 4,
NUM_PULSAR_IMAGES = 3,
NUM_ROCK_IMAGES = 6,
NUM_SAUCER_IMAGES = 2,
NUM_SAUCER_BULLET_IMAGES = 3,
NUM_STARS = 40,
NUM_MOUNTAIN_RANGE_POINTS = 10;


inline int
getLevelNum(long score)
{
    if (score >= 100000)
	return 6;
    if (score >= 50000)
	return 5;
    if (score >= 20000)
	return 4;
    if (score >= 5000)
	return 3;
    if (score >= 1000)
	return 2;
    return 1;
}


inline int
discreteUniformRand(const Couple &limits)
/*  
    Returns a random integer between limits.x and limits.y inclusively.
    limits.x must be <= limits.y.
*/
{
    return rand() % (limits.y - limits.x + 1) + limits.x;
}


inline int
rnd(int lowerLimit, int upperLimit)
{
    return rand() % (upperLimit - lowerLimit + 1) + lowerLimit;
}


inline int
mulRound(int n, double x)
/*  Multiply the arguments, and round the result to the nearest integer
    according to its sign.
*/
{
    double product = n * x;
    if (product >= 0.0)
	return (int) (product + 0.5);
    return (int) (product - 0.5);
}

inline int
(min)(int a, int b)
{
	return (a < b ? a : b);
}


inline int
(max)(int a, int b)
{
	return (a > b ? a : b);
}


inline int
divideAmplitude(int n, int divisor)
{
    if (n == 0)
	return 0;
    if (n < 0)
	return min(n / divisor, -1);
    return max(n / divisor, +1);
}


inline Couple
divideAmplitude(const Couple &c, int divisor)
{
    return Couple(divideAmplitude(c.x, divisor), divideAmplitude(c.y, divisor));
}


inline int
decrementAmplitude(int n)
{
    if (n > 0)
	return n - 1;
    if (n < 0)
	return n + 1;
    return 0;
}


inline Couple
decrementAmplitude(const Couple &c)
{
    return Couple(decrementAmplitude(c.x), decrementAmplitude(c.y));
}


inline void
removeNullElementsFromSpriteList(SpriteList &slist)
{
    SpriteList::iterator it =
		    remove(slist.begin(), slist.end(), (Sprite *) NULL);
	/*  remove() has "packed" the remaining elements at the beginning
	    of the sequence, but has not shortened the list.  This must
	    be done by a call to the erase() method.  Doesn't this seem
	    unintuitive?  I thought remove() removed stuff.
	    @sarrazip 20010501
	*/
    slist.erase(it, slist.end());
}


///////////////////////////////////////////////////////////////////////////////


CosmoSmashEngine::CosmoSmashEngine(
		    long initialScore,
		    bool _mirrorHyperspace, bool useSound,
		    bool fullScreen, bool automaticPlay)
							throw(int, string)
  : GameEngine(theDrawingPixmapSize, "Cosmosmash", fullScreen),
  		// may throw string exception

    paused(false),
    tickCount(0),

    mirrorHyperspace(_mirrorHyperspace),
    useGameExtensions(false),
    playerBoostTicks(0),
    playerBoostType(NO_BOOST),

    playerPA(2),
    playerSprite(NULL),

    baseBulletPA(1),
    baseBulletSprites(),

    bigRockPA(NUM_ROCK_IMAGES),
    smallRockPA(NUM_ROCK_IMAGES),
    rockSprites(),

    bigSpinnerPA(NUM_SPINNER_IMAGES),
    smallSpinnerPA(NUM_SPINNER_IMAGES),
    spinnerSprites(),

    pulsarPA(NUM_PULSAR_IMAGES),
    pulsarSprites(),

    saucerPA(NUM_SAUCER_IMAGES),
    saucerSprites(),

    saucerBulletPA(NUM_SAUCER_BULLET_IMAGES),
    saucerBulletSprites(),

    explosionPA(2),
    explosionSprites(),

    questionPA(2),
    questionSprites(),

    groundPos(0),
    groundHeight(0),
    mountainTopPos(0),

    controller(automaticPlay
		? static_cast<Controller *>(new AutoController())
		: static_cast<Controller *>(new UserController())),
    quitKS(SDLK_ESCAPE),

    initScore(initialScore),
    theScore(0),
    thePeakScore(0),
    updateScore(true),

    numLives(0),
    updateNumLives(true),

    timeBeforeNextPulsar(0),
    timeBeforeNextSaucer(0),
    timeBeforeNextQuestion(0),

    starPositions(),

    theSoundMixer(NULL)
{
    this->useSound = useSound;

    assert(initScore >= 0);
    assert(theSDLScreen != NULL);
    assert(theSDLScreen->format != NULL);

    blackColor = SDL_MapRGB(theSDLScreen->format, 0, 0, 0);
    whiteColor = SDL_MapRGB(theSDLScreen->format, 255, 255, 255);
    greenColor = SDL_MapRGB(theSDLScreen->format, 0, 255, 0);

    try
    {
	initializeSprites();
    }
    catch (PixmapLoadError &e)
    {
	string msg = __("Could not load pixmap ") + e.getFilename();
	throw msg;
    }

    initializeMisc(useSound);  // may throw a 'string'
}


CosmoSmashEngine::~CosmoSmashEngine()
{
    delete playerSprite;
}


void
CosmoSmashEngine::initializeSprites() throw(PixmapLoadError)
{
    groundPos = theDrawingPixmapSize.y - 50;
    groundHeight = 10;

    // Load some pixmaps:
    loadPixmap(base_xpm, playerPA, 0);
    loadPixmap(boosted_base_xpm, playerPA, 1);

    // Create and position the player sprite:
    playerSprite = new Sprite(playerPA,
	    Couple((theDrawingPixmapSize.x - playerPA.getImageSize().x) / 2,
		    groundPos - playerPA.getImageSize().y),
	    Couple(0, 0), Couple(0, 0),
	    Couple(6, 6), playerPA.getImageSize() - Couple(12, 12));
	// this Sprite object does not own playerPA
    

    // Base bullets:
    loadPixmap(basebullet_xpm, baseBulletPA, 0);

    // Rocks:
    loadPixmap(bigrock0_xpm, bigRockPA, 0);
    loadPixmap(bigrock1_xpm, bigRockPA, 1);
    loadPixmap(bigrock2_xpm, bigRockPA, 2);
    loadPixmap(bigrock3_xpm, bigRockPA, 3);
    loadPixmap(bigrock4_xpm, bigRockPA, 4);
    loadPixmap(bigrock5_xpm, bigRockPA, 5);
    loadPixmap(smallrock0_xpm, smallRockPA, 0);
    loadPixmap(smallrock1_xpm, smallRockPA, 1);
    loadPixmap(smallrock2_xpm, smallRockPA, 2);
    loadPixmap(smallrock3_xpm, smallRockPA, 3);
    loadPixmap(smallrock4_xpm, smallRockPA, 4);
    loadPixmap(smallrock5_xpm, smallRockPA, 5);

    // Big spinner:
    loadPixmap(bigspinner_horizontal_xpm, bigSpinnerPA, 0);
    loadPixmap(bigspinner_slash_xpm,      bigSpinnerPA, 1);
    loadPixmap(bigspinner_vertical_xpm,   bigSpinnerPA, 2);
    loadPixmap(bigspinner_backslash_xpm,  bigSpinnerPA, 3);

    // Small spinner:
    loadPixmap(smallspinner_horizontal_xpm, smallSpinnerPA, 0);
    loadPixmap(smallspinner_slash_xpm,      smallSpinnerPA, 1);
    loadPixmap(smallspinner_vertical_xpm,   smallSpinnerPA, 2);
    loadPixmap(smallspinner_backslash_xpm,  smallSpinnerPA, 3);

    // Pulsar:
    loadPixmap(pulsar_empty_xpm, pulsarPA, 0);
    loadPixmap(pulsar_mid_xpm,   pulsarPA, 1);
    loadPixmap(pulsar_full_xpm,  pulsarPA, 2);

    // Saucer:
    loadPixmap(saucer0_xpm, saucerPA, 0);
    loadPixmap(saucer1_xpm, saucerPA, 1);

    // Saucer bullets:
    loadPixmap(saucerbullet_empty_xpm, saucerBulletPA, 0);
    loadPixmap(saucerbullet_mid_xpm,   saucerBulletPA, 1);
    loadPixmap(saucerbullet_full_xpm,  saucerBulletPA, 2);

    // Explosion:
    loadPixmap(explosion0_xpm, explosionPA, 0);
    loadPixmap(explosion1_xpm, explosionPA, 1);

    // Question (surprise) sprites:
    loadPixmap(question0_xpm, questionPA, 0);
    loadPixmap(question1_xpm, questionPA, 1);
}


static
string
getDir(const char *defaultValue, const char *envVarName)
{
    string dir;
    const char *s = getenv(envVarName);
    if (s != NULL)
	dir = s;
    else
	dir = defaultValue;
    
    if (!dir.empty() && dir[dir.length() - 1] != '/')
	dir += '/';

    return dir;
}


void
CosmoSmashEngine::initializeMisc(bool useSound) throw(string)
/*  
    Throws an error message in a 'string' if an error occurs.
*/
{
    theScore = initScore;
    thePeakScore = initScore;
    theLevel = getLevelNum(theScore);
    updateScore = true;
    scoreAreaPos = Couple(4, theDrawingPixmapSize.y - 16);
    numLives = 0;
    updateNumLives = true;
    numLivesAreaPos = Couple(theDrawingPixmapSize.x - 72, scoreAreaPos.y);
    playerBoostTicks = 0;
    playerBoostType = NO_BOOST;


    setTimeBeforeNextPulsar();
    setTimeBeforeNextSaucer();
    setTimeBeforeNextQuestion();


    /*  Initialize mountain range positions:
    */
    int y0 = groundPos - 100;
    int y1 = groundPos -  70;
    int y2 = groundPos -  50;

    mountainTopPos = y0;

    size_t i;
    for (i = 0; i < NUM_MOUNTAIN_RANGE_POINTS; i++)
    {
	int x = i * theDrawingPixmapSize.x / (NUM_MOUNTAIN_RANGE_POINTS - 1);
	int y;
	if (i & 1)
	    y = rnd(y0, y1);
	else
	    y = rnd(y1, y2);
	mountainRangePositions.push_back(Couple(x, y));
    }


    /*  Initialize star positions:
    */
    const int padding = 8;
    for (i = 0; i < NUM_STARS; i++)
    {
	int x = rnd(padding, theDrawingPixmapSize.x - padding);
	int y = rnd(padding, y2 - 10);
	starPositions.push_back(Couple(x, y));
    }


    /*  Sound effects:
    */
    if (useSound)
    {
	try
	{
	    theSoundMixer = NULL;
	    theSoundMixer = new SoundMixer();  // may throw string
	}
	catch (const SoundMixer::Error &e)
	{
	    return;
	}

	try
	{
	    string pkgsounddir = getDir(PKGSOUNDDIR, "PKGSOUNDDIR");
	    for (size_t i = 0; i < NUM_ROCK_HIT_SOUNDS; i++)
	    {
		string filename =
			    pkgsounddir + "rock-hit-" + char('0' + i) + ".wav";
		rockHitSounds[i].init(filename);
	    }

	    playerHitSound.init(pkgsounddir + "player-hit.wav");
	    pulsarBeepSound.init(pkgsounddir + "pulsar-beep.wav");
	    saucerShootingSound.init(pkgsounddir + "saucer-shooting.wav");
	    hyperspaceSound.init(pkgsounddir + "hyperspace.wav");
	}
	catch (const SoundMixer::Error &e)
	{
	    throw e.what();
	}
    }
}


void
CosmoSmashEngine::setTimeBeforeNextPulsar()
{
    int min = (theLevel < 3 ? 10 :  1);
    int max = (theLevel < 3 ? 45 : 15);
    timeBeforeNextPulsar = rnd(20 * min, 20 * max);
}


void
CosmoSmashEngine::setTimeBeforeNextSaucer()
{
    timeBeforeNextSaucer = rnd(20 * 5, 20 * 30);
}


void
CosmoSmashEngine::setTimeBeforeNextQuestion()
{
    static const char *q = getenv("Q");
    timeBeforeNextQuestion = (q != NULL ? 100 : rnd(20 * 10, 20 * 30));
}


///////////////////////////////////////////////////////////////////////////////


/*virtual*/
void
CosmoSmashEngine::processKey(SDLKey keysym, bool pressed)
{
    if (UserController *uc = dynamic_cast<UserController *>(controller.get()))
	uc->processKey(keysym, pressed);

    quitKS.check(keysym, pressed);
}


/*virtual*/
bool
CosmoSmashEngine::tick()
{
    if (quitKS.isPressed())
	return false;

    if (paused)
    {
	if (controller->isPauseRequested())
	{
	    paused = false;
	    displayPauseMessage(false);
	}
    }
    else
    {
	tickCount++;

	detectCollisions();

	if (!animatePlayer())
	    return false;
	
	animateAutomaticCharacters();
	restoreBackground();
	drawSprites();


	/*  If the game is in demo mode, the Space bar starts a new game.
	*/
	if (numLives == 0)
	{
	    if (controller->isStartRequested(useGameExtensions))
	    {
		displayStartMessage(false);

		killAllAutomaticCharacters();
		centerPlayerSprite();
		theScore = initScore;
		thePeakScore = initScore;
		numLives = NUM_INIT_LIVES;

		addToScore(0);
		addToNumLives(0);
	    }
	    else 
	    {
		/*  At the end of a game, wait for the last base to have
		    finished exploding before displaying the start
		    message again.
		*/
		if (explosionSprites.size() == 0)
		    displayStartMessage(true);
	    }
	}

	if (paused)
	    displayPauseMessage(true);
    }

    controller->endOfTick();

    return true;
}


void
CosmoSmashEngine::fireBullet(Couple pos)
{
    Couple bbSize = baseBulletPA.getImageSize();
    Sprite *s = new Sprite(baseBulletPA,
		pos, Couple(0, -PLAYER_BULLET_SPEED), Couple(0, 0),
		Couple(0, 0), bbSize);
    baseBulletSprites.push_back(s);

    if (theLevel >= 5)  // double bullet in higher levels
    {
	int n = (theLevel == 5 ? bbSize.y / 2 : bbSize.y);
	Couple pos2 = pos;
	pos2.y -= n;
	Sprite *s = new Sprite(baseBulletPA,
		pos2, Couple(0, -PLAYER_BULLET_SPEED), Couple(0, 0),
		Couple(0, 0), bbSize);
	baseBulletSprites.push_back(s);
    }

    if (playerBoostTicks > 0 && playerBoostType == TRIPLE_BULLETS)
    {
	int vx = 3;
	Couple pos2 = pos;
	pos2.x -= bbSize.x + 1;
	Sprite *s = new Sprite(baseBulletPA,
		    pos2, Couple(-vx, -PLAYER_BULLET_SPEED), Couple(0, 0),
		    Couple(0, 0), bbSize);
	baseBulletSprites.push_back(s);

	pos2 = pos;
	pos2.x += bbSize.x + 1;
	s = new Sprite(baseBulletPA,
		    pos2, Couple(+vx, -PLAYER_BULLET_SPEED), Couple(0, 0),
		    Couple(0, 0), bbSize);
	baseBulletSprites.push_back(s);
    }
}


bool
CosmoSmashEngine::animatePlayer()
/*  
    Returns true if the game must continue, or false to have it stop.
*/
{
    if (AutoController *ac = dynamic_cast<AutoController *>(controller.get()))
    {
	ac->beginGameState();

	ac->setPlayerPos(playerSprite->getCenterPos());

	SpriteList::iterator its;
	for (its = rockSprites.begin(); its != rockSprites.end(); its++)
	    ac->addRockPos((*its)->getCenterPos());
	for (its = spinnerSprites.begin(); its != spinnerSprites.end(); its++)
	    ac->addSpinnerPos((*its)->getCenterPos());
	for (its = pulsarSprites.begin(); its != pulsarSprites.end(); its++)
	    ac->addPulsarPos((*its)->getCenterPos());
	for (its = saucerBulletSprites.begin(); its != saucerBulletSprites.end(); its++)
	    ac->addPulsarPos((*its)->getCenterPos());

	ac->endGameState();
    }

    if (controller->isResumeRequested())
    {
	paused = true;
	return true;
    }

    if (playerSprite->decTimeToLive() != 0)  // if player base exploding
    {
	addToScore(0);  // patch: redraw score, because explosion munges it
	return true;
    }

    if (numLives == 0)
	return true;


    Couple &playerPos = playerSprite->getPos();
    const Couple &playerSize = playerSprite->getSize();


    /*  Fire a bullet once in a while:
    */
    int ticksBetweenBullets =
	   (playerBoostTicks > 0 && playerBoostType == CLOSER_BULLETS ? 3 : 6);
    if (tickCount % ticksBetweenBullets == 0 && controller->isShootingActive())
    {
	Couple bbSize = baseBulletPA.getImageSize();
	Couple pos = playerPos + Couple((playerSize.x - bbSize.x) / 2, 0);

	fireBullet(pos);
    }
    if (playerBoostTicks > 0)
	playerBoostTicks--;


    Couple &playerSpeed = playerSprite->getSpeed();

    /*	Get and process user commands:
    */

    playerSpeed.zero();

    if (controller->isHyperspaceRequested())
    {
	Couple size = playerSprite->getSize();
	int newX;
	if (mirrorHyperspace)
	{
	    // Mirror the player's position WRT the center:
	    Couple center = playerSprite->getCenterPos();
	    int newXCenter = theScreenSizeInPixels.x - center.x;
	    newX = newXCenter - size.x / 2;
	    if (newX < 0)
		newX = 0;  // just to be sure...
	}
	else
	    newX = rnd(0, theScreenSizeInPixels.x - size.x);

	Couple newPos(newX, groundPos - playerPA.getImageSize().y);
	repositionPlayerSprite(newPos);

	playSoundEffect(hyperspaceSound);

	return true;
    }

    if (controller->isLeftMoveRequested())
	playerSpeed.x = -PLAYER_SPEED;
    if (controller->isRightMoveRequested())
	playerSpeed.x = +PLAYER_SPEED;

    if (playerSpeed.isZero())
	return true;

    Couple newPos = playerPos + playerSpeed;
    if (newPos.x < 0)
	newPos.x = 0;
    else if (newPos.x + playerSize.x > theDrawingPixmapSize.x)
	newPos.x = theDrawingPixmapSize.x - playerSize.x;

    playerPos = newPos;

    return true;
}


void
CosmoSmashEngine::killSpritesInList(SpriteList &sl)
{
    for (SpriteList::iterator its = sl.begin(); its != sl.end(); its++)
	delete *its;

    sl.clear();
    assert(sl.size() == 0);
}


void
CosmoSmashEngine::centerPlayerSprite()
{
    Couple newPos(
	    (theDrawingPixmapSize.x - playerPA.getImageSize().x) / 2,
	    groundPos - playerPA.getImageSize().y);
    repositionPlayerSprite(newPos);
}


void
CosmoSmashEngine::repositionPlayerSprite(Couple newPos)
{
    playerSprite->setPos(newPos);
}


void
CosmoSmashEngine::killAllAutomaticCharacters()
{
    killSpritesInList(rockSprites);
    killSpritesInList(spinnerSprites);
    killSpritesInList(pulsarSprites);
    killSpritesInList(saucerSprites);
    killSpritesInList(saucerBulletSprites);
    killSpritesInList(questionSprites);
}


void
CosmoSmashEngine::killPlayerBase()
{
    static const char *noKillEnvVar = getenv("NOKILL");

    if (noKillEnvVar != NULL)
    	return;

    playSoundEffect(playerHitSound);

    addToScore(SCORE_LOST_BASE);

    Couple pos = playerSprite->getCenterPos();
    pos.y = playerSprite->getLowerRightPos().y - 1;
    static const Couple directions[] =
    {
	Couple(  10,   0 ),
	Couple(   7,  -7 ),
	Couple(   0, -10 ),
	Couple(  -7,  -7 ),
	Couple( -10,   0 ),
	Couple(   0,   0 )  // zero marks the end
    };

    const int explosionTime = 40;  // in ticks

    for (size_t i = 0; directions[i].isNonZero(); i++)
    {
	Sprite *explosion = createExplosionSprite(pos, explosionTime);
	explosion->setSpeed(directions[i]);
    }

    playerSprite->setTimeToLive(explosionTime);
	    /*  The player's "time to live" value is used to mark
		the period during which it is exploding. The player's
		sprite is not displayed during that time. It is
		repositioned for when it starts being displayed again.
	    */
    centerPlayerSprite();
    addToNumLives(-1);

    playerBoostTicks = 0;
    playerBoostType = NO_BOOST;

    killAllAutomaticCharacters();
	    /*  We kill falling sprites to avoid situations where e.g.,
		spinners keep landing while the player is exploding.
		Such situations could make the player lose many lives in a row.
	    */

    killSpritesInList(baseBulletSprites);
}


inline bool
CosmoSmashEngine::playerIsActive() const
/*  
    Returns true iff the player still has lives and not currently exploding...
*/
{
    return (numLives > 0 && playerSprite->getTimeToLive() == 0);
}


pair<bool, size_t>
CosmoSmashEngine::animateFallingObjects(SpriteList &sprites)
/*  
    Moves the sprites in the designated sprite list.

    If one of the sprites collides with the player, the player dies
    and the sprite is eliminated.
    Sprites that land are eliminated.
    Sprites that go off to the left or right of the screen are
    silently eliminated.

    Returns a boolean value that indicates if the player dies, and the
    number of falling objects that landed.

    When this method exits, pointers to eliminated sprites have been
    removed from 'sprites'.
*/
{
    bool playerDies = false;
    size_t numLandings = 0;  // number of objects that touch the ground

    for (SpriteList::iterator it = sprites.begin(); it != sprites.end(); it++)
    {
	Sprite *s = *it;
	assert(s != NULL);
	s->addSpeedToPos();

	if (playerIsActive() && s->collidesWithSprite(*playerSprite))
	{
	    playerDies = true;

	    delete s;
	    *it = NULL;
	    continue;
	}

	int height = groundPos - s->getLowerRightPos().y;
		// height is positive when object's bottom is above ground
	if (height <= - groundHeight)  // if rock low enough, it dies
	{
	    numLandings++;

	    delete s;
	    *it = NULL;
	    continue;
	}

	const Couple newPos = s->getPos();
	if (newPos.x >= theScreenSizeInPixels.x
			|| newPos.x + s->getSize().x <= 0)
	{
	    // object disappears without consequence
	    delete s;
	    *it = NULL;
	    continue;
	}
    }

    removeNullElementsFromSpriteList(sprites);

    return pair<bool, size_t>(playerDies, numLandings);
}


void
CosmoSmashEngine::animateSaucers()
{
    bool playerDies = false;

    for (SpriteList::iterator it = saucerSprites.begin();
				it != saucerSprites.end(); it++)
    {
	Sprite *saucer = *it;
	Couple &pos = saucer->getPos();
	const Couple &size = saucer->getSize();

	pos += saucer->getSpeed();

	// if the saucer is now off the screen:
	if (pos.x >= theScreenSizeInPixels.x || pos.x + size.x <= 0)
	{
	    delete saucer;
	    *it = NULL;
	    continue;
	}

	// make saucer shoot periodically:
	if (tickCount % 8 == 0 && playerSprite->getTimeToLive() == 0)
	{
	    const Couple size = saucerBulletPA.getImageSize();
	    Couple bulletPos = pos;

	    // Choose speed in direction of player base:
	    Couple diff = playerSprite->getPos() - bulletPos;
	    double k = SAUCER_BULLET_SPEED / hypot(diff.x, diff.y);
	    Couple speed(mulRound(diff.x, k), mulRound(diff.y, k));
	    speed = accelerate(speed);
	    if (speed.isZero())
		speed = Couple(0, SAUCER_BULLET_SPEED);
	    Sprite *s = new Sprite(saucerBulletPA,
				    pos, speed, Couple(0, 0),
				    Couple(2, 2), size - Couple(2, 2));
	    saucerBulletSprites.push_back(s);

	    playSoundEffect(saucerShootingSound);
	}
    }
    removeNullElementsFromSpriteList(saucerSprites);

    if (playerDies)
	killPlayerBase();
}


void
CosmoSmashEngine::animateAutomaticCharacters()
{
    /*  If there are not enough ordinary falling sprites, add some:
    */
    size_t numFallingObjects = rockSprites.size() + spinnerSprites.size();
    size_t min = (theLevel < 5 ? 2 : 3);
    if (numFallingObjects < min && playerSprite->getTimeToLive() == 0)
    {
	if (rand() % 4 == 0)
	{
	    // New spinner (more small spinners starting at level 2):
	    int n = (theLevel < 2 ? 3 : 2);
	    const PixmapArray *pa =
			(rand() % n == 0 ? &smallSpinnerPA : &bigSpinnerPA);
	    const Couple size = pa->getImageSize();
	    Couple pos(rnd(0, theScreenSizeInPixels.x - size.x), 0);
	    Couple speed(rnd(-2, +2), rnd(3, 10));
	    speed = accelerate(speed);
	    Sprite *s = new Sprite(*pa, pos, speed, Couple(0, 0),
				    size / 3, size * 2 / 3);
	    spinnerSprites.push_back(s);
	}
	else
	{
	    // New rock:
	    const PixmapArray *pa =
			(rand() % 3 == 0 ? &smallRockPA : &bigRockPA);
	    const Couple size = pa->getImageSize();
	    Couple pos(rnd(0, theScreenSizeInPixels.x - size.x), 0);
	    Couple speed(rnd(-2, +2), rnd(3 + theLevel, 20 + theLevel));
	    speed = accelerate(speed);
	    Sprite *s = new Sprite(*pa, pos, speed, Couple(0, 0),
				    Couple(4, 4), size - Couple(4, 4));
	    s->currentPixmapIndex = rnd(0, NUM_ROCK_IMAGES - 1);
	    rockSprites.push_back(s);
	}
    }


    /*  Send a new pulsar once in a while.
    */
    if (playerSprite->getTimeToLive() == 0 && --timeBeforeNextPulsar == 0)
    {
	setTimeBeforeNextPulsar();
	const Couple size = pulsarPA.getImageSize();
	Couple pos(rnd(0, theScreenSizeInPixels.x - size.x), 0);
	Couple speed(0, PULSAR_SPEED + 3 * (theLevel >= 4));
	speed = accelerate(speed);
	Sprite *s = new Sprite(pulsarPA,
				pos, speed, Couple(0, 0),
				Couple(2, 2), size - Couple(2, 2));
	s->setTimeToLive(4 * 20);
	pulsarSprites.push_back(s);
    }


    /*  Send a new saucer once in a while.
    */
    if (theLevel >= 4 && playerSprite->getTimeToLive() == 0
				&& --timeBeforeNextSaucer == 0)
    {
	setTimeBeforeNextSaucer();
	const Couple size = saucerPA.getImageSize();

	Couple pos, speed;
	int v = SAUCER_SPEED + 2 * (theLevel >= 5);
	if (rnd(0, 1) == 0)
	{
	    pos = Couple(- size.x, SAUCER_Y_POS);
	    speed = Couple(v, 0);
	}
	else
	{
	    pos = Couple(theScreenSizeInPixels.x, SAUCER_Y_POS);
	    speed = Couple(- v, 0);
	}

	Sprite *s = new Sprite(saucerPA,
				pos, speed, Couple(0, 0),
				Couple(2, 2), size - Couple(2, 2));
	s->currentPixmapIndex = 0;
	saucerSprites.push_back(s);
    }


    /*  Send a new question mark once in a while.
    */
    if (useGameExtensions
		&& theLevel >= 0
		&& playerSprite->getTimeToLive() == 0
		&& --timeBeforeNextQuestion == 0)
    {
	setTimeBeforeNextQuestion();
	const Couple size = questionPA.getImageSize();

	Couple pos, speed;
	pos = Couple(
		rnd(size.x * 4, theScreenSizeInPixels.x - size.x * 4),
		- size.y);
	speed = Couple(rnd(-2, +2), QUESTION_SPEED + 2 * (theLevel >= 5));

	Sprite *s = new Sprite(questionPA,
				pos, speed, Couple(0, 0),
				Couple(2, 2), size - Couple(2, 2));
	s->currentPixmapIndex = 0;
	questionSprites.push_back(s);
    }


    /*  Let the rocks fall. If they hit the ground or go off the screen
	on the left or right, kill them. If they hit the player (while it
	is not currently exploding), then make the player explode.
    */
    pair<bool, size_t> p = animateFallingObjects(rockSprites);
    if (p.second > 0 && numLives > 0)  // if rocks landed and player has lives
	addToScore(SCORE_BIG_ROCK_LANDING * p.second);
		// NOTE: landing of small rock still scored as big rock
    if (p.first)  // if collision with player
	killPlayerBase();


    /*  Let the question marks fall...
    */
    if (useGameExtensions)
    {
	pair<bool, size_t> p = animateFallingObjects(questionSprites);
	// We don't care when the question marks land.
	if (p.first)  // if collision with player
	    killPlayerBase();
    }


    /*  Let the spinners fall. If they hit the ground, the player dies.
	If they go off the screen on the left or right, kill them.
	If they hit the player (while it is not currently exploding),
	then the player dies.
    */
    p = animateFallingObjects(spinnerSprites);
    if (p.second > 0 && playerIsActive())
	killPlayerBase();
    else if (p.first)
	killPlayerBase();


    /*  Let the saucer bullets fall. If they hit the ground, kill them.
	If they go off the screen on the left or right, kill them.
	If they hit the player (while it is not currently exploding),
	then the player dies.
    */
    p = animateFallingObjects(saucerBulletSprites);
    if (p.first)
	killPlayerBase();


    /*  Let the pulsars fall. If they hit the ground, kill them.
	If they go off the screen on the left or right, kill them.
	If they hit the player (while it is not currently exploding),
	then the player dies.
	Pulsars disappear after some time.
	Once in a while, redirect them towards the player.
    */
    p = animateFallingObjects(pulsarSprites);
    if (p.first)
	killPlayerBase();


    int n = (theLevel < 3 ? 8 : 6);  // ticks between direction changes
    SpriteList::iterator it;
    for (it = pulsarSprites.begin(); it != pulsarSprites.end(); it++)
    {
	Sprite *pulsar = *it;
	assert(pulsar != NULL);
	if (pulsar->decTimeToLive() == 0)
	{
	    delete pulsar;
	    *it = NULL;
	}
	else if (tickCount % n == 0)
	{
	    Couple diff = playerSprite->getPos() - pulsar->getPos();
	    // NOTE: use projection instead of trig; see animateSaucers()
	    double angle = atan2((double) diff.y, (double) diff.x);
	    Couple newSpeed(int(PULSAR_SPEED * cos(angle)),
					int(PULSAR_SPEED * sin(angle)));
	    if (newSpeed.isZero())
		newSpeed = Couple(0, PULSAR_SPEED);
	    pulsar->setSpeed(newSpeed);
	}
    }
    removeNullElementsFromSpriteList(pulsarSprites);


    /*  Move the saucers.
    */
    animateSaucers();


    /*  Move the player's bullets:
    */
    for (it = baseBulletSprites.begin(); it != baseBulletSprites.end(); it++)
    {
	Sprite *s = *it;
	assert(s != NULL);
	s->addSpeedToPos();
	if (s->getPos().y + s->getSize().y <= 0)  // if completely above screen
	{
	    delete s;
	    *it = NULL;
	}
    }
    removeNullElementsFromSpriteList(baseBulletSprites);


    /*  Animate explosions:
    */
    for (it = explosionSprites.begin(); it != explosionSprites.end(); it++)
    {
	Sprite *s = *it;
	assert(s != NULL);

	if (s->currentPixmapIndex > 1)
	{
	    fprintf(stderr, "s->currentPixmapIndex = %u\n",
	    					s->currentPixmapIndex);
	    assert(false);
	}
	if (s->decTimeToLive() == 0)
	{
	    delete s;
	    *it = NULL;
	    continue;
	}

	s->addSpeedToPos();  // some explosions move

	s->currentPixmapIndex ^= 1;  // switch between 0 and 1
	assert(s->currentPixmapIndex <= 1);
    }
    removeNullElementsFromSpriteList(explosionSprites);
}


Sprite *
CosmoSmashEngine::createExplosionSprite(Couple posOfCenterOfExplosion,
						int timeToLive)
/*  
    Creates an explosion sprite centered at 'posOfCenterOfExplosion'.
    Sets the "time to live" value for the new sprite to 'timeToLive',
    which must be expressed in ticks.
    Inserts the address of this sprite in the list 'explosionSprites'.
    The sprite has no speed.
    Returns a pointer to the created sprite.
*/
{
    Couple size = explosionPA.getImageSize();
    Couple pos = posOfCenterOfExplosion - size / 2;
    Sprite *s = new Sprite(explosionPA, pos,
		    Couple(0, 0), Couple(0, 0),
		    Couple(2, 2), size - Couple(2, 2));
    s->setTimeToLive(timeToLive);
    explosionSprites.push_back(s);
    return s;
}


pair<Sprite *, Sprite *>
CosmoSmashEngine::splitBigRock(const Sprite &bigRock) const
/*  
    Returns two new rock sprites which have not been inserted in
    any sprite list.
*/
{
    const Couple bigRockPos = bigRock.getPos();
    const Couple bigRockSize = bigRock.getSize();

    const Couple smallRockSize = smallRockPA.getImageSize();
    int smallRockY = bigRockPos.y + bigRockSize.y / 2 - smallRockSize.y / 2;
    Couple pos1(bigRockPos.x + bigRockSize.x * 1 / 4, smallRockY);
    Couple pos2(bigRockPos.x + bigRockSize.x * 3 / 4, smallRockY);
    pos2.x -= smallRockSize.x;

    Couple speed1(rnd(-4, -2), bigRock.getSpeed().y);  // must go to the left
    Couple speed2(- speed1.x, speed1.y);  // must go to the right

    // Rotate the speeds of the small rocks some times:
    int dir = rand() % 3;
    for (int times = rand() % 2 + 1; times > 0; times--)
	switch (dir)
	{
	    case 0:
		speed1.x--;
		speed2.y++;
		break;
	    case 1:
		speed2.x++;
		speed1.y++;
		break;
	    default:
		;
	}

    speed1 = accelerate(speed1);
    speed2 = accelerate(speed2);
    Sprite *smallRock1 = new Sprite(smallRockPA,
			    pos1, speed1, Couple(0, 0), Couple(2, 2),
			    smallRockSize - Couple(2, 2));
    smallRock1->currentPixmapIndex = bigRock.currentPixmapIndex;
    Sprite *smallRock2 = new Sprite(smallRockPA,
			    pos2, speed2, Couple(0, 0), Couple(2, 2),
			    smallRockSize - Couple(2, 2));
    smallRock2->currentPixmapIndex = bigRock.currentPixmapIndex;

    //rockSprites.push_back(smallRock1);
    //rockSprites.push_back(smallRock2);
	/*  We do not insert the two new sprites in the rockSprites list
	    right now because this method is called during an iteration
	    over that list.  Insertions in the list at this point could
	    reallocate the internal data structure and thus disrupt
	    the iteration. That is why we return a pair of newly
	    created sprites which the caller can insert in the list
	    when appropriate.
	*/
    return pair<Sprite *, Sprite *>(smallRock1, smallRock2);
}


void
CosmoSmashEngine::makeSpriteExplode(const Sprite *target,
				    const Sprite *bullet,
				    long points)
/*  
    Makes the 'target' sprite explode.
    'bullet' must be the sprite that triggers the explosion; it is erased
    from the screen.
    The target is erased from the screen and an explosion is started
    at its position.
    'points' is the number of points to add to the player's score.
    The object designated by 'target' is not destroyed.
    'target' and 'bullet' must not be null.
*/
{
    assert(target != NULL);
    assert(bullet != NULL);

    playShortExplosionSound();

    Couple targetPos = target->getPos();

    Couple expPos = target->getCenterPos();
    createExplosionSprite(expPos, 6);

    addToScore(points);
}


bool
CosmoSmashEngine::detectCollisionsWithBullet(const Sprite &bullet,
					    SpriteList &slist,
					    int scoreOnCollision)
/*  
    Detects fatal collisions between a bullet and sprites in 'slist'.

    'slist' is allowed to contain null pointers; they are ignored.

    If a collision is detected between 'bullet' and an element
    of 'slist', this element is made null in 'slist' and the
    corresponding Sprite object is destroyed with operator delete;
    also, an explosion is created at the point of collision and the
    method returns true.

    The method returns false if no collision is detected.
*/
{
    bool collisionDetected = false;

    for (SpriteList::iterator its = slist.begin(); its != slist.end(); its++)
    {
	Sprite *aSprite = *its;
	if (aSprite == NULL)  // if already dead
	    continue;
	
	if (bullet.collidesWithSprite(*aSprite))
	{
	    makeSpriteExplode(aSprite, &bullet, scoreOnCollision);

	    delete aSprite;
	    *its = NULL;
	    collisionDetected = true;
	}
    }

    return collisionDetected;
}


bool
CosmoSmashEngine::detectCollisionsWithExplosions()
/*  
    Kills certain sprites if they touch an explosion sprite.
    Explosion sprites do not disappear because of a collision.
    Sprite lists containing killed sprites will contain null
    elements after this method finishes.  The caller is
    responsible for cleaning up those null elements.

    Returns true if the player dies, or false otherwise.
*/
{
    if (!playerIsActive())
	return false;

    /*  Make a local copy of the list of explosion sprites, and iterate
	on the copy.
	We cannot iterate over 'explosionSprites' because the calls
	to detectCollisionsWithBullet() will add elements to it;
	this could invalidate an iterator that would point into the list.

	This technique is not efficient in principle, but the number of
	explosions is normally low, so no visible performance hit should
	be observed.
    */
    SpriteList tempList(explosionSprites.size());
    copy(explosionSprites.begin(), explosionSprites.end(), tempList.begin());

    assert(tempList.size() == explosionSprites.size());

    SpriteList::const_iterator it;
    for (it = tempList.begin(); it != tempList.end(); it++)
    {
	const Sprite *anExplosion = *it;
	if (anExplosion == NULL)
	    continue;
	
	// Consider the explosion sprite as a bullet:
	(void) detectCollisionsWithBullet(*anExplosion, rockSprites, 0);
	(void) detectCollisionsWithBullet(*anExplosion, spinnerSprites, 0);
	(void) detectCollisionsWithBullet(*anExplosion, pulsarSprites, 0);
	(void) detectCollisionsWithBullet(*anExplosion, saucerSprites, 0);
	(void) detectCollisionsWithBullet(*anExplosion, saucerBulletSprites, 0);

	// NOTE: make the explosions kill the base also.
	if (anExplosion->collidesWithSprite(*playerSprite))
	{
	    killPlayerBase();  // this kills all meanies
	    return true;       // no need to continue
	}
    }

    return false;
}


void CosmoSmashEngine::detectCollisionsBetweenFallingObjects(SpriteList &slist)
/*
    Sprite lists containing killed sprites will contain null
    elements after this method finishes.  The caller is
    responsible for cleaning up those null elements.
*/
{
    SpriteList::iterator it;
    for (it = slist.begin(); it != slist.end(); it++)
    {
	const Sprite *s = *it;
	if (s == NULL)
	    continue;
	
	bool coll = false;
	if (!coll)
	    coll = detectCollisionsWithBullet(*s, rockSprites, 0);
	if (!coll)
	    coll = detectCollisionsWithBullet(*s, saucerSprites, 0);
	if (!coll)
	    coll = detectCollisionsWithBullet(*s, saucerBulletSprites, 0);
	if (coll)
	{
	    *it = NULL;
	    delete s;
	}
    }
}


void
CosmoSmashEngine::detectCollisions()
/*
    NOTE: Many collisions are detected here, but there are some others
    that are detected elsewhere, as in animateFallingObjects(), for example.
*/
{
    SpriteList newRocks;
    bool playerBoosted = false;

    /*  Detect collisions involving the player's bullets:
    */
    for (SpriteList::iterator itb = baseBulletSprites.begin();
				    itb != baseBulletSprites.end(); itb++)
    {
	Sprite *bullet = *itb;
	assert(bullet != NULL);

	for (SpriteList::iterator itr = rockSprites.begin();
					itr != rockSprites.end(); itr++)
	{
	    Sprite *rock = *itr;
	    if (rock == NULL)  // if already dead
		continue;  // next rock

	    assert(bullet != NULL);

	    if (bullet->collidesWithSprite(*rock))
	    {
		long points;
		bool explode = true;
		if (rock->getSize() == bigRockPA.getImageSize())
		{
		    points = SCORE_BIG_ROCK;

		    /*  Some big rocks break into two small rocks.
		    */
		    if (rand() % 3 == 0)
		    {
			playShortExplosionSound();

			pair<Sprite *, Sprite *> p = splitBigRock(*rock);
			newRocks.push_back(p.first);
			newRocks.push_back(p.second);


			/*  We don't create an explosion when a big rock splits
			    because an explosion would instantly destroy the
			    two new small rocks.  @sarrazip 20010530
			*/
			explode = false;
		    }
		}
		else
		    points = SCORE_SMALL_ROCK;

		if (explode)
		    makeSpriteExplode(rock, bullet, points);


		// don't destroy 'bullet' -- only mark it for deletion:
		*itb = NULL;
		delete rock;
		*itr = NULL;
		break;
	    }
	}


	assert(bullet != NULL);  // must still be alive for next for()


	SpriteList::iterator its;
	for (its = spinnerSprites.begin(); its != spinnerSprites.end(); its++)
	{
	    Sprite *spinner = *its;
	    if (spinner == NULL)  // if already dead
		continue;  // next spinner
	    
	    assert(bullet != NULL);
	    if (bullet->collidesWithSprite(*spinner))
	    {
		long points;
		if (spinner->getSize() == bigSpinnerPA.getImageSize())
		    points = SCORE_BIG_SPINNER;
		else
		    points = SCORE_SMALL_SPINNER;

		makeSpriteExplode(spinner, bullet, points);

		// don't destroy 'bullet' -- only mark it for deletion:
		*itb = NULL;
		delete spinner;
		*its = NULL;
		break;
	    }
	}


	assert(bullet != NULL);  // must still be alive for next for()


	if (detectCollisionsWithBullet(
				*bullet, pulsarSprites, SCORE_PULSAR))
	    *itb = NULL;  // this marks that the bullet hit something

	if (detectCollisionsWithBullet(
				*bullet, saucerSprites, SCORE_SAUCER))
	    *itb = NULL;

	if (detectCollisionsWithBullet(
			    *bullet, saucerBulletSprites, SCORE_SAUCER_BULLET))
	    *itb = NULL;

	if (detectCollisionsWithBullet(
			    *bullet, questionSprites, SCORE_QUESTION))
	{
	    *itb = NULL;
	    playerBoosted = true;
	}

	if (*itb == NULL)  // if bullet hit something
	    delete bullet;

    }   // for each bullet


    detectCollisionsBetweenFallingObjects(pulsarSprites);
    detectCollisionsBetweenFallingObjects(questionSprites);

    (void) detectCollisionsWithExplosions();

    removeNullElementsFromSpriteList(baseBulletSprites);
    removeNullElementsFromSpriteList(rockSprites);
    removeNullElementsFromSpriteList(spinnerSprites);
    removeNullElementsFromSpriteList(pulsarSprites);
    removeNullElementsFromSpriteList(saucerSprites);
    removeNullElementsFromSpriteList(saucerBulletSprites);
    removeNullElementsFromSpriteList(questionSprites);

    rockSprites.insert(rockSprites.end(), newRocks.begin(), newRocks.end());

    if (playerBoosted)
	boostPlayer();
}


void
CosmoSmashEngine::boostPlayer()
{
    playerBoostTicks = rnd(100, 150);

    switch (rnd(0, 1))
    {
	case 0: playerBoostType = TRIPLE_BULLETS; break;
	case 1: playerBoostType = CLOSER_BULLETS; break;
    }
}


///////////////////////////////////////////////////////////////////////////////


static void
putpixel(SDL_Surface *surface, int x, int y, Uint32 pixel)
{
    int bpp = surface->format->BytesPerPixel;
    Uint8 *p = (Uint8 *) surface->pixels + y * surface->pitch + x * bpp;

    switch (bpp)
    {
	case 1:
	    *p = Uint8(pixel);
	    break;

	case 2:
	    * (Uint16 *) p = Uint16(pixel);
	    break;

	case 3:
	    if (SDL_BYTEORDER == SDL_BIG_ENDIAN)
	    {
		p[0] = Uint8((pixel >> 16) & 0xFF);
		p[1] = Uint8((pixel >> 8) & 0xFF);
		p[2] = Uint8(pixel & 0xFF);
	    }
	    else
	    {
		p[0] = Uint8(pixel & 0xFF);
		p[1] = Uint8((pixel >> 8) & 0xFF);
		p[2] = Uint8((pixel >> 16) & 0xFF);
	    }
	    break;

	case 4:
	    * (Uint32 *) p = pixel;
	    break;
    }
}


static unsigned char gamma_table[256];
static bool gamma_table_initialized = false;


static void
wu_line(SDL_Surface *surface,
		       Uint32 x0, Uint32 y0, Uint32 x1, Uint32 y1,
		       Uint32 fgc, Uint32 bgc)
/* Source: http://mail.lokigames.com/ml/sdl/0288.html */
{
	const int nlevels = 256;
	const int nbits = 8;

  if (!gamma_table_initialized)
  {
    /* generate gamma correction table */
    const size_t numLevelsGammaTable = 256;
    const double Draw_gamma = 2.35;
    for (size_t i=0; i<numLevelsGammaTable; i++)
      gamma_table[i] = (unsigned char) ((numLevelsGammaTable - 1) * pow(
		    i / (double) (numLevelsGammaTable - 1), 1.0 / Draw_gamma));
    gamma_table_initialized = true;
  }

  Uint32 intshift, erracc,erradj;
  Uint32 erracctmp, wgt, wgtcompmask;
  int dx, dy, tmp, xdir, i;
  Uint32 colors[nlevels];
  SDL_Rect r;
  SDL_Color fg, bg;

  SDL_GetRGB(fgc, surface->format, &fg.r, &fg.g, &fg.b);
  SDL_GetRGB(bgc, surface->format, &bg.r, &bg.g, &bg.b);

  /* generate the colors by linear interpolation, applying gamma correction */
  for (i=0; i<nlevels; i++) {
    Uint8 r, g, b;

    r = gamma_table[fg.r - (i*(fg.r - bg.r))/(nlevels-1)];
    g = gamma_table[fg.g - (i*(fg.g - bg.g))/(nlevels-1)];
    b = gamma_table[fg.b - (i*(fg.b - bg.b))/(nlevels-1)];
    colors[i] = SDL_MapRGB(surface->format, r, g, b);
  }
  if (y0 > y1) {
    tmp = y0; y0 = y1; y1 = tmp;
    tmp = x0; x0 = x1; x1 = tmp;
  }
  /* draw the initial pixel in the foreground color */
  putpixel(surface, x0, y0, fgc);

  dx = x1 - x0;
  xdir = (dx >= 0) ? 1 : -1;
  dx = (dx >= 0) ? dx : -dx;

  /* special-case horizontal, vertical, and diagonal lines which need no
     weighting because they go right through the center of every pixel. */
  if ((dy = y1 - y0) == 0) {
    /* horizontal line */
    r.x = (x0 < x1) ? x0 : x1;
    r.y = y0;
    r.w = dx;
    r.h = 1;
    SDL_FillRect(surface, &r, fgc);
    return;
  }

  if (dx == 0) {
    /* vertical line */
    r.x = x0;
    r.y = y0;
    r.w = 1;
    r.h = dy;
    SDL_FillRect(surface, &r, fgc);
    return;
  }

  if (dx == dy) {
    for (; dy != 0; dy--) {
      x0 += xdir;
      y0++;
      putpixel(surface, x0, y0, fgc);
    }
    return;
  }
  /* line is not horizontal, vertical, or diagonal */
  erracc = 0;			/* err. acc. is initially zero */
  /* # of bits by which to shift erracc to get intensity level */
  intshift = 32 - nbits;
  /* mask used to flip all bits in an intensity weighting */
  wgtcompmask = nlevels - 1;
  /* x-major or y-major? */
  if (dy > dx) {
    /* y-major.  Calculate 16-bit fixed point fractional part of a pixel that
       X advances every time Y advances 1 pixel, truncating the result so that
       we won't overrun the endpoint along the X axis */
    erradj = ((Uint64)dx << 32) / (Uint64)dy;
    /* draw all pixels other than the first and last */
    while (--dy) {
      erracctmp = erracc;
      erracc += erradj;
      if (erracc <= erracctmp) {
	/* rollover in error accumulator, x coord advances */
	x0 += xdir;
      }
      y0++;			/* y-major so always advance Y */
      /* the nbits most significant bits of erracc give us the intensity
	 weighting for this pixel, and the complement of the weighting for
	 the paired pixel. */
      wgt = erracc >> intshift;
      putpixel(surface, x0, y0, colors[wgt]);
      putpixel(surface, x0+xdir, y0, colors[wgt^wgtcompmask]);
    }
    /* draw the final pixel, which is always exactly intersected by the line
       and so needs no weighting */
    putpixel(surface, x1, y1, fgc);
    return;
  }
  /* x-major line.  Calculate 16-bit fixed-point fractional part of a pixel
     that Y advances each time X advances 1 pixel, truncating the result so
     that we won't overrun the endpoint along the X axis. */
  erradj = ((Uint64)dy << 32) / (Uint64)dx;
  /* draw all pixels other than the first and last */
  while (--dx) {
    erracctmp = erracc;
    erracc += erradj;
    if (erracc <= erracctmp) {
      /* accumulator turned over, advance y */
      y0++;
    }
    x0 += xdir;			/* x-major so always advance X */
    /* the nbits most significant bits of erracc give us the intensity
       weighting for this pixel, and the complement of the weighting for
       the paired pixel. */
    wgt = erracc >> intshift;
    putpixel(surface, x0, y0, colors[wgt]);
    putpixel(surface, x0, y0+1, colors[wgt^wgtcompmask]);
  }
  /* draw final pixel, always exactly intersected by the line and doesn't
     need to be weighted. */
  putpixel(surface, x1, y1, fgc);
}


static void
line(SDL_Surface *surface,
		Uint32 x1, Uint32 y1, Uint32 x2, Uint32 y2, Uint32 color)
{
    wu_line(surface, x1, y1, x2, y2, color, 0);
}


///////////////////////////////////////////////////////////////////////////////


void
CosmoSmashEngine::restoreBackground()
{
    // Background:
    SDL_Rect rect0 = { 0, 0, theDrawingPixmapSize.x, theDrawingPixmapSize.y };
    (void) SDL_FillRect(theSDLScreen, &rect0, blackColor);

    // Ground:
    SDL_Rect rect1 = { 0, groundPos, theDrawingPixmapSize.x, groundHeight };
    (void) SDL_FillRect(theSDLScreen, &rect1, greenColor);

    // Stars:
    bool moveStars = (numLives > 0 && tickCount % 2 == 0);
    CoupleList::iterator it;
    for (it = starPositions.begin(); it != starPositions.end(); it++)
    {
	Couple &pos = *it;
	putpixel(theSDLScreen, pos.x, pos.y, whiteColor);

	if (moveStars)
	{
	    pos.y--;
	    if (pos.y < 0)
		pos.y = mountainTopPos - 1;
	}
    }

    // Mountain range:
    Couple previous(-1, -1);
    for (it = mountainRangePositions.begin();
    				it != mountainRangePositions.end(); it++)
    {
	if (previous.x != -1)  // do nothing the first time
	    line(theSDLScreen, previous.x, previous.y,
	    			(*it).x, (*it).y, greenColor);
	previous = Couple((*it).x, (*it).y);
    }
}


void
CosmoSmashEngine::drawSprites()
{
    if (playerIsActive())
    {
	int pixmapIndex = (playerBoostTicks > 0 ? ((tickCount & 4) != 0) : 0);
	copySpritePixmap(*playerSprite, pixmapIndex, playerSprite->getPos());
    }

    SpriteList::const_iterator it;
    for (it = rockSprites.begin(); it != rockSprites.end(); it++)
	copySpritePixmap(**it, (**it).currentPixmapIndex, (**it).getPos());

    if (useGameExtensions)
	for (it = questionSprites.begin(); it != questionSprites.end(); it++)
	{
	    size_t &index = (**it).currentPixmapIndex;
	    index = (tickCount / 8) % 2;
	    copySpritePixmap(**it, index, (**it).getPos());
	}

    for (it = spinnerSprites.begin(); it != spinnerSprites.end(); it++)
    {
	size_t &index = (**it).currentPixmapIndex;
	index = (tickCount / 4) % NUM_SPINNER_IMAGES;
	copySpritePixmap(**it, index, (**it).getPos());
    }

    if (!pulsarSprites.empty() && tickCount % (NUM_PULSAR_IMAGES * 4) == 0)
	playSoundEffect(pulsarBeepSound);
    for (it = pulsarSprites.begin(); it != pulsarSprites.end(); it++)
    {
	size_t &index = (**it).currentPixmapIndex;
	index = (tickCount / 2) % NUM_PULSAR_IMAGES;
	copySpritePixmap(**it, index, (**it).getPos());
    }

    for (it = saucerSprites.begin(); it != saucerSprites.end(); it++)
    {
	size_t &index = (**it).currentPixmapIndex;
	index = (tickCount / 4) % NUM_SAUCER_IMAGES;
	copySpritePixmap(**it, index, (**it).getPos());
    }

    for (it = saucerBulletSprites.begin(); it != saucerBulletSprites.end(); it++)
    {
	size_t &index = (**it).currentPixmapIndex;
	index = (tickCount / 2) % NUM_SAUCER_BULLET_IMAGES;
	copySpritePixmap(**it, index, (**it).getPos());
    }

    for (it = baseBulletSprites.begin(); it != baseBulletSprites.end(); it++)
	copySpritePixmap(**it, 0, (**it).getPos());

    for (it = explosionSprites.begin(); it != explosionSprites.end(); it++)
    {
	assert(*it != NULL);
	assert((**it).currentPixmapIndex <= 1);
	copySpritePixmap(**it, (**it).currentPixmapIndex, (**it).getPos());
    }

    {
	int ext = (useGameExtensions ? '*' : ' ');
	char s[64];
	snprintf(s, sizeof(s), "Score      :%8ld%c", theScore, ext);
	writeString(s, scoreAreaPos - Couple(0, 15));
	snprintf(s, sizeof(s), "Peak Score :%8ld%c", thePeakScore, ext);
	writeString(s, scoreAreaPos);
	updateScore = false;
    }

    {
	char s[64];
	snprintf(s, sizeof(s), "Lives:%3d", numLives);
	writeString(s, numLivesAreaPos);
	updateNumLives = false;
    }
}


void
CosmoSmashEngine::addToScore(long n)
{
    theScore += n * theLevel;
    if (theScore > thePeakScore)
    {
	/*  Maintain the peak score. Give the player a new life each time
	    the peak score crosses some boundary.
	*/
	const long newLifeScoreInterval =
		(useGameExtensions ? 10000 : 1000);
	long before = thePeakScore / newLifeScoreInterval;
	thePeakScore = theScore;
	long after = thePeakScore / newLifeScoreInterval;
	if (before != after)
	    addToNumLives(+1);
    }
    theLevel = getLevelNum(theScore);
    updateScore = true;
}


void
CosmoSmashEngine::addToNumLives(int n)
{
    numLives += n;
    updateNumLives = true;
}


Couple
CosmoSmashEngine::accelerate(Couple speed) const
{
    int coef = (theScore >= 100000 ? theScore - 100000 : 0) / 25000 + 1;
    if (speed.x < 0)
	speed.x -= coef;
    else if (speed.x > 0)
	speed.x += coef;
    if (speed.y < 0)
	speed.y -= coef;
    else if (speed.y > 0)
	speed.y += coef;
    return speed;
}


void
CosmoSmashEngine::displayPauseMessage(bool display)
/*  
    Displays the pause message if 'display' is true, or erases the
    corresponding region if 'display' is false.
*/
{
    if (display)
	displayMessage(0, "PAUSED -- press P to resume");
}


void
CosmoSmashEngine::displayStartMessage(bool display)
/*  
    Displays the start message if 'display' is true, or erases the
    corresponding region if 'display' is false.
*/
{
    if (display)
    {
	static const Couple fontdim = getFontDimensions();

	displayMessage(0, "Cosmosmash " VERSION " - by Pierre Sarrazin   ");
	displayMessage(1, "Press SPACE to start a standard game    ");
	displayMessage(2, "Press E to start a game with extensions ");

	displayMessage(4, "During game: move with left/right arrows");
	displayMessage(5, "and press down arrow for hyperspace;    ");
	displayMessage(6, "press P to pause the game.              ");

	if (useSound && theSoundMixer == NULL)
	    displayMessage(8, "*** Sound device not available ***      ");
    }
}


void
CosmoSmashEngine::displayMessage(int row, const char *msg)
{
    static const Couple fontdim = getFontDimensions();

    const size_t len = strlen(msg);
    int x = (theScreenSizeInPixels.x - len * fontdim.x) / 2;
    int y = theScreenSizeInPixels.y + (row - 26) * fontdim.y;

    writeString(msg, Couple(x, y));
}


void
CosmoSmashEngine::playShortExplosionSound()
{
    int soundNo = rand() % NUM_ROCK_HIT_SOUNDS;
    playSoundEffect(rockHitSounds[soundNo]);
}


void
CosmoSmashEngine::playSoundEffect(SoundMixer::Chunk &chunk)
{
    /*  'theSoundMixer' is still NULL if the sound effects were not wanted
	or if the sound device is not available.
	Also, there are no sounds during the demo mode.
    */
    if (theSoundMixer != NULL && numLives != 0)
    {
	try
	{
	    theSoundMixer->playChunk(chunk);
	}
	catch (const SoundMixer::Error &e)
	{
	    fprintf(stderr, "playSoundEffect: %s\n", e.what().c_str());
	}
    }
}
