/*  $Id$
    CosmoSmashEngine.h - A space rock shooting video game engine.

    cosmosmash - A space rock shooting video game.
    Copyright (C) 2000, 2001 Pierre Sarrazin <sarrazip@iname.com>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
    02111-1307, USA.
*/

#ifndef _H_CosmoSmashEngine
#define _H_CosmoSmashEngine


#include <gengameng/GameEngine.h>
#include <gengameng/Sprite.h>

#include <string>

using namespace std;


class CosmoSmashEngine : public GameEngine
/*  A space rock shooting video game engine.
*/
{
public:
    CosmoSmashEngine(long initialScore, bool mirrorHyperspace)
    							throw(int, string);
    /*  See base class.

	Throws a string or an integer code if an error occurs.

	If 'mirrorHyperspace' is true, then hyperspace pivots around
	the center of the ground.  If 'mirrorHyperspace' is false,
	then hyperspace is random.
    */

    virtual ~CosmoSmashEngine();
    /*  Nothing interesting.
    */

    virtual void processKey(SDLKey keysym, bool pressed);
    /*  Inherited.
    */

    virtual bool tick();
    /*  Inherited.
    */

protected:

    // Inherited:
    bool animatePlayer();
    void animateAutomaticCharacters();
    void detectCollisions();
    void restoreBackground();
    void drawSprites();

private:

    typedef vector<Couple> CoupleList;

    enum Key
    {
	NOKEY = 0,

	// The next four constants are guaranteed to have the values 1 to 4:
	UP_ARROW,
	DOWN_ARROW,
	LEFT_ARROW,
	RIGHT_ARROW,

	ENTER,
	SPACE,
	CTRL,
	ALT,
	ESC,

	NUM_SUPPORTED_KEYS
    };



    bool paused;
    unsigned long tickCount;

    // SDL color values (see SDL_MapRGB()):
    Uint32 blackColor;
    Uint32 whiteColor;
    Uint32 greenColor;

    bool mirrorHyperspace;

    PixmapArray playerPA;
    Sprite *playerSprite;

    PixmapArray baseBulletPA;
    SpriteList  baseBulletSprites;

    PixmapArray bigRockPA;
    PixmapArray smallRockPA;
    SpriteList rockSprites;  // contains big and small rock sprites

    PixmapArray bigSpinnerPA;
    PixmapArray smallSpinnerPA;
    SpriteList spinnerSprites;

    PixmapArray pulsarPA;
    SpriteList pulsarSprites;

    PixmapArray saucerPA;
    SpriteList saucerSprites;

    PixmapArray saucerBulletPA;
    SpriteList saucerBulletSprites;

    PixmapArray explosionPA;
    SpriteList  explosionSprites;

    int groundPos;     // y-position of highest pixel row of ground
    int groundHeight;  // height of ground in pixels

    bool leftArrowPressed, rightArrowPressed;
    bool escapePressed;
    bool spacePressed, spacePressedBefore;
    bool downArrowPressed, downArrowPressedBefore;
    bool letterPPressed, letterPPressedBefore;
    bool ctrlPressed;

    long   initScore;     // score at which a game starts (determines level)

    long   theScore;      // player's score in points
    long   thePeakScore;  // not displayed
    int    theLevel;      // level number (1--6) (depends on theScore)
    bool   updateScore;   // indicates if player's score needs to be rewritten
    Couple scoreAreaPos;

    int    numLives;  // number of player lives left
    bool   updateNumLives;
    Couple numLivesAreaPos;

    unsigned long timeBeforeNextPulsar;  // in ticks
    unsigned long timeBeforeNextSaucer;

    CoupleList starPositions;
    CoupleList mountainRangePositions;



    /*  Implementation functions:
    */
    void displayErrorMessage(const string &msg) throw();
    void initializeSprites() throw(PixmapLoadError);
    void initializeMisc() throw(string);
    void setTimeBeforeNextPulsar();
    void setTimeBeforeNextSaucer();
    int getSpeedUp() const;
    void readKeyboardCommands();
    Sprite *createExplosionSprite(Couple posOfCenterOfExplosion,
				    int timeToLive);
    pair<Sprite *, Sprite *> splitBigRock(const Sprite &bigRock) const;
    void makeSpriteExplode(const Sprite *target,
			    const Sprite *bullet,
			    long points);
    void addToScore(long n);
    void addToNumLives(int n);
    void killSpritesInList(SpriteList &sl);
    void centerPlayerSprite();
    void repositionPlayerSprite(Couple newpos);
    void killAllMeanies();
    void killPlayerBase();
    bool playerIsActive() const;
    pair<bool, size_t> animateFallingObjects(SpriteList &sprites);
    void animateSaucers();
    void eraseSprite(const Sprite &s);
    void displayPauseMessage(bool display);
    void displayStartMessage(bool display);
    void displayMessage(const char *msg);
    bool detectCollisionsWithBullet(const Sprite &bullet,
				    SpriteList &slist,
				    int scoreOnCollision);
    bool detectCollisionsWithExplosions();
    void detectCollisionsBetweenFallingObjects();
    Couple accelerate(Couple speed) const;


    /*	Forbidden operations:
    */
    CosmoSmashEngine(const CosmoSmashEngine &);
    CosmoSmashEngine &operator = (const CosmoSmashEngine &);
};


#endif  /* _H_CosmoSmashEngine */
