/*
 *	cook - file construction tool
 *	Copyright (C) 1991, 1992, 1993, 1994, 1995, 1996, 1997, 1998, 1999, 2000 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: functions to isolate operating system interface
 */

#include <ac/errno.h>
#include <ac/fcntl.h>
#include <ac/limits.h>
#include <signal.h>
#include <ac/stddef.h>
#include <ac/stdio.h>
#include <ac/stdlib.h>
#include <ac/string.h>
#include <ac/time.h>
#include <ac/unistd.h>

#include <archive.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <sys/wait.h>
#include <ac/utime.h>

#include <error_intl.h>
#include <home_directo.h>
#include <mem.h>
#include <option.h>
#include <os_interface.h>
#include <os/wait.h>
#include <stat.cache.h>
#include <str_list.h>
#include <trace.h>

#ifdef HAVE_GETRUSAGE
#include <sys/resource.h>
#else
#include <sys/times.h>
#endif


/*
 * NAME
 *	os_mtime - return the last-modified time of a file
 *
 * SYNOPSIS
 *	time_t os_mtime(string_ty *path);
 *
 * DESCRIPTION
 *	Os_mtime returns the time the named file was last modified.
 *	It returns 0 if the file does not exist.
 *
 * CAVEAT
 *	Assumes time will be the UNIX format.
 */

time_t
os_mtime_oldest(path)
	string_ty	*path;
{
	time_t		result;

	trace(("os_mtime_oldest(path = \"%s\")\n{\n", path->str_text));
	result = stat_cache_oldest(path, 1);
	trace(("return (%ld) %s", (long)result, ctime(&result)));
	trace(("}\n"));
	return result;
}

time_t
os_mtime_newest(path)
	string_ty	*path;
{
	time_t		result;

	trace(("os_mtime_newest(path = \"%s\")\n{\n", path->str_text));
	result = stat_cache_newest(path, 1);
	trace(("return (%ld) %s", (long)result, ctime(&result)));
	trace(("}\n"));
	return result;
}


static void adjust_message _((string_ty *, long));

static void
adjust_message(path, nsec)
	string_ty	*path;
	long		nsec;
{
	int		direction;
	string_ty	*buffer;
	sub_context_ty	*scp;

	/*
	 * work out the direction of adjustment
	 */
	trace(("adjust_message(nsec = %ld)\n{\n", nsec));
	if (nsec >= 0) 
		direction = 1;
	else
	{
		nsec = -nsec;
		direction = -1;
	}

	/*
	 * work out the units of adjustment
	 */
	scp = sub_context_new();
	if (nsec >= (365L * 24 * 60 * 60))
	{
		sub_var_set(scp, "Number", "%4.2f", nsec / (double)(365L * 24 * 60 * 60));
		buffer = subst_intl(scp, i18n("$number years"));
	}
	else if (nsec >= (24L * 60 * 60))
	{
		sub_var_set(scp, "Number1", "%ld", nsec / (24L * 60 * 60));
		sub_var_set(scp, "Number2", "%ld", (nsec / (60 * 60)) % 24);
		buffer = subst_intl(scp, i18n("$number1 days $number2 hours"));
	}
	else if (nsec >= (60 * 60))
	{
		sub_var_set(scp, "Number1", "%ld", nsec / (60 * 60));
		sub_var_set(scp, "Number2", "%ld", (nsec / 60) % 60);
		buffer = subst_intl(scp, i18n("$number1 hours $number2 minutes"));
	}
	else if (nsec >= 60)
	{
		sub_var_set(scp, "Number1", "%ld", nsec / 60);
		sub_var_set(scp, "Number2", "%ld", nsec % 60);
		buffer = subst_intl(scp, i18n("$number1 minutes $number2 seconds"));
	}
	else
	{
		sub_var_set(scp, "Number", "%ld", nsec);
		buffer = subst_intl(scp, i18n("$number seconds"));
	}

	/*
	 * print the message
	 * (re-use substitution context)
	 */
	trace(("mark\n"));
	sub_var_set(scp, "File_Name", "%S", path);
	sub_var_set(scp, "Number", "%S", buffer);
	if (direction < 0)
		error_intl(scp, i18n("adjusting \"$filename\" back $number"));
	else
		error_intl(scp, i18n("adjusting \"$filename\" forward $number"));
	str_free(buffer);
	sub_context_delete(scp);
	trace(("}\n"));
}


/*
 * NAME
 *	os_mtime_adjust - indicate change
 *
 * SYNOPSIS
 *	int os_mtime_adjust(string_ty *path);
 *
 * DESCRIPTION
 *	The os_mtime_adjust function is used to adjust the value in the stat
 *	cache to indicate that a recipe has constructed a file, and thus
 *	changed it last-modified time.  No change to the actual file system
 *	occurs.
 *
 * RETURNS
 *	int; -1 on error, 0 on success
 */

int
os_mtime_adjust(path, min_age)
	string_ty	*path;
	time_t		min_age;
{
	time_t		mtime;
	int		err;

	trace(("os_mtime_adjust(path = \"%s\")\n{\n", path->str_text));
	if (option_test(OPTION_UPDATE) && option_test(OPTION_ACTION))
	{
		stat_cache_clear(path);
		mtime = os_mtime_newest(path);
		if (mtime < 0)
		{
			trace(("return -1;\n"));
			trace(("}\n"));
			return -1;
		}
		if (mtime)
		{
			if
			(
				option_test(OPTION_UPDATE_MAX)
			?
				mtime != min_age
			:
				mtime < min_age
			)
			{
				struct utimbuf	ut;

				if (!option_test(OPTION_SILENT))
					adjust_message(path, (long)min_age - (long)mtime);
				ut.modtime = min_age;
				ut.actime = min_age;
				err = utime(path->str_text, &ut);
				if (err && errno == ENOENT)
					err = archive_utime(path, &ut);
				if (err && errno == EPERM)
				{
					sub_context_ty	*scp;

					/*
					 * The only way to get this
					 * error message is if the
					 * recipe body invoked sudo (or
					 * similar) to change the owner.
					 *
					 * They did it to themselves, so
					 * just give them a warning.
					 */
					scp = sub_context_new();
					sub_errno_set(scp);
					sub_var_set(scp, "File_Name", "%S", path);
					error_intl
					(
						scp,
			   i18n("warning: when adjusting \"$filename\": $errno")
					);
					trace(("return 0;\n"));
					trace(("}\n"));
					return 0;
				}
				if (err)
				{
					sub_context_ty	*scp;

					scp = sub_context_new();
					sub_errno_set(scp);
					sub_var_set(scp, "File_Name", "%S", path);
					error_intl
					(
						scp,
						i18n("utime $filename: $errno")
					);
					sub_context_delete(scp);
					option_set_errors();
					trace(("return -1;\n"));
					trace(("}\n"));
					return -1;
				}
#if defined(sun) || defined(__sun__)
				else
				{
					/*
					 * Solaris is brain dead.
					 * The inode times are not
					 * updated until the next sync.
					 * So we'll help it along.
					 */
					sync();
				}
#endif
				stat_cache_set(path, min_age, 1);
			}
		}
		else
		{
			/*
			 * file was deleted (or was a dummy)
			 * so pretend it was changed "now"
			 */
			time(&mtime);
			if (mtime > min_age)
				min_age = mtime;
			stat_cache_set(path, min_age, 0);
		}
	}
	else
	{
		time(&mtime);
		if (mtime > min_age)
			min_age = mtime;
		stat_cache_set(path, min_age, 0);
	}
	trace(("return 0;\n"));
	trace(("}\n"));
	return 0;
}


/*
 * NAME
 *	os_clear_stat - invalidate cache
 *
 * SYNOPSIS
 *	int os_clear_stat(string_ty *path);
 *
 * DESCRIPTION
 *	The os_clear_stat function is used to invalidate the the stat
 *	cache to indicate that a recipe has constructed a file, and thus
 *	changed it last-modified time.  No change to the actual file system
 *	occurs.
 *
 * RETURNS
 *	int; 0 on success, -1 on error
 *
 * CAVEAT
 *	This is used in situations where the recipe changes a file not named
 *	in the targets of the recipe.  This usually occurs around mkdir, rm
 *	and mv commands, used in conjunction with the [exists] builtin function.
 */

int
os_clear_stat(path)
	string_ty	*path;
{
	stat_cache_clear(path);
	return 0;
}


/*
 * NAME
 *	os_touch - update the modify time of the file
 *
 * SYNOPSIS
 *	int os_touch(string_ty *path);
 *
 * DESCRIPTION
 *	Os_touch updates the last-modified time of the file to the present.
 *	If the named file does not exist, then nothing is done.
 *
 * RETURNS
 *	int; 0 on success, -1 on error
 */

int
os_touch(path)
	string_ty	*path;
{
	struct utimbuf	ut;
	int		err;

	time(&ut.modtime);
	if (ut.modtime < 0)
		ut.modtime = 0;
	ut.actime = ut.modtime;
	err = utime(path->str_text, &ut);
	if (err && errno == ENOENT)
		err = archive_utime(path, &ut);
	if (err)
	{
		sub_context_ty	*scp;
		
		if (errno == ENOENT)
		{
			stat_cache_clear(path);
			return 0;
		}
		scp = sub_context_new();
		sub_errno_set(scp);
		sub_var_set(scp, "File_Name", "%S", path);
		error_intl(scp, i18n("utime $filename: $errno"));
		sub_context_delete(scp);
		option_set_errors();
		return -1;
	}
#if defined(sun) || defined(__sun__)
	else
	{
		/*
		 * Solaris is brain dead.  The inode times are not
		 * updated until the next sync.  So we'll help it along.
		 */
		sync();
	}
#endif
	stat_cache_set(path, ut.modtime, 1);
	return 0;
}


/*
 * NAME
 *	os_delete - delete a file
 *
 * SYNOPSIS
 *	int os_delete(string_ty *path);
 *
 * DESCRIPTION
 *	Os_delete deletes the named file.
 *	If it does not exist, no error is given.
 *
 * RETURNS
 *	int; -1 on error, 0 on success
 */

int
os_delete(path, echo)
	string_ty	*path;
	int		echo;
{
	if (unlink(path->str_text))
	{
		if (errno != ENOENT)
		{
			error_intl_unlink(path->str_text);
			option_set_errors();
			return -1;
		}
	}
	else if (echo)
	{
		sub_context_ty	*scp;

		scp = sub_context_new();
		sub_var_set(scp, "File_Name", "%S", path);
		error_intl(scp, i18n("rm $filename"));
		sub_context_delete(scp);
	}

	/*
	 * if the knew about the existence of the file before we deleted
	 * it, then we will have to adjust the stat cache.
	 */
	stat_cache_clear(path);
	return 0;
}


/*
 * NAME
 *	exit_status - pretty print
 *
 * SYNOPSIS
 *	int exit_status(char *cmd, int status, int errok);
 *
 * DESCRIPTION
 *	The exit_status function is used to pretty print the meaning of the
 *	exit status of the given command.  No output is generated for normal
 *	(0) termination.
 *
 * RETURNS
 *	int: zero if the command succeeded, 1 if it failed.
 *
 * CAVEAT
 *	This function should be static, but func_collect (builtin.c) uses it.
 */

int
exit_status(cmd, status, errok)
	char		*cmd;
	int		status;
	int		errok;
{
	int		a, b, c;
	sub_context_ty	*scp;

	a = (status >> 8) & 0xFF;
	b = (status >> 7) & 1;
	c = status & 0x7F;
	switch (c)
	{
	case 0x7F:
		/*
		 * process was stopped,
		 * since we didn't do it, treat it as an error
		 */
		scp = sub_context_new();
		sub_var_set(scp, "File_Name", "%s", cmd);
		error_intl(scp, i18n("command $filename: stopped"));
		sub_context_delete(scp);
		return 1;

	case 0:
		/* normal termination */
		if (a)
		{
			scp = sub_context_new();
			sub_var_set(scp, "File_Name", "%s", cmd);
			sub_var_set(scp, "Number", "%d", a);
			if (errok)
			{
				error_intl
				(
					scp,
			i18n("command $filename: exit status $number (ignored)")
				);
			}
			else
			{
				error_intl
				(
					scp,
				  i18n("command $filename: exit status $number")
				);
			}
			sub_context_delete(scp);
			if (errok)
				return 0;
			return 1;
		}
		return 0;

	default:
		/*
		 * process dies from unhandled condition
		 */
		scp = sub_context_new();
		sub_var_set(scp, "File_Name", "%s", cmd);
		sub_var_set(scp, "Name", "%s", strsignal(c));
		if (b)
		{
			error_intl
			(
				scp,
		    i18n("command $filename: terminated by $name (core dumped)")
			);
		}
		else
		{
			error_intl
			(
				scp,
				i18n("command $filename: terminated by $name")
			);
		}
		return 1;
	}
}


/*
 * NAME
 *	execute - do a command
 *
 * SYNOPSIS
 *	int execute(string_list_ty *cmd, int fd, int errok);
 *
 * DESCRIPTION
 *	The execute function is used to execute the command in the word list.
 *	If the file descriptor is >= 0, it indicates a file to use as stdin to
 *	the command.
 *
 * RETURNS
 *	int: zero if the commands succeeds, nonzero if it fails.
 *
 * SEE ALSO
 *	cook/opcode/command.c
 */

static int execute _((string_list_ty *, int, int));

static int
execute(cmd, fd, errok)
	string_list_ty	*cmd;
	int		fd;
	int		errok;
{
	int		child;
	int		pid;
	int		status;
	static char	**argv;
	static size_t	argvlen;
	int		j;
	sub_context_ty	*scp;

	if (!argv)
	{
		argvlen = cmd->nstrings + 1;
		argv = mem_alloc(argvlen * sizeof(char*));
	}
	else
	{
		if (argvlen < cmd->nstrings + 1)
		{
			argvlen = cmd->nstrings + 1;
			argv = mem_change_size(argv, argvlen * sizeof(char *));
		}
	}
	for (j = 0; j < cmd->nstrings; ++j)
		argv[j] = cmd->string[j]->str_text;
	argv[cmd->nstrings] = 0;
	switch (child = fork())
	{
	case -1:
		scp = sub_context_new();
		sub_errno_set(scp);
		error_intl(scp, i18n("fork(): $errno"));
		sub_context_delete(scp);
		return -1;

	case 0:
		if (fd >= 0)
		{
			if (close(0) && errno != EBADF)
			{
				string_ty	*fn0;
				int		err;

				err = errno;
				scp = sub_context_new();
				fn0 = subst_intl(scp, "standard input");
				/* re-use substitution context */
				sub_errno_setx(scp, err);
				sub_var_set(scp, "File_Name", "%S", fn0);
				fatal_intl(scp, i18n("close $filename: $errno"));
				sub_context_delete(scp);
				str_free(fn0);
			}
			if (dup(fd) < 0)
			{
				scp = sub_context_new();
				sub_errno_set(scp);
				fatal_intl(scp, i18n("dup(): $errno"));
				/* NOTREACHED */
			}
		}
		if (argv[0][0] == '/')
			execv(argv[0], argv);
		else
			execvp(argv[0], argv);
		scp = sub_context_new();
		sub_errno_set(scp);
		sub_var_set(scp, "File_Name", "%s", argv[0]);
		fatal_intl(scp, i18n("exec $filename: $errno"));
		/* NOTREACHED */

	default:
		for (;;)
		{
			pid = os_waitpid(child, &status);
			assert(pid == child || pid == -1);
			if (pid == child)
				return exit_status(argv[0], status, errok);
			if (pid < 0 && errno != EINTR)
			{
				scp = sub_context_new();
				sub_errno_set(scp);
				error_intl(scp, i18n("wait(): $errno"));
				sub_context_delete(scp);
				option_set_errors();
				return -1;
			}
		}
	}
}


/*
 * NAME
 *	os_execute - execute a command
 *
 * SYNOPSIS
 *	int os_execute(string_list_ty *args, string_ty *input, int errok);
 *
 * DESCRIPTION
 *	Os_execute performs the given command.
 *	If the command succeeds 0 is returned.
 *	If the command fails a diagnostic message is given
 *	and 1 is returned.
 *
 * SEE ALSO
 *	cook/opcode/command.c
 */

int
os_execute(args, input, errok)
	string_list_ty	*args;
	string_ty	*input;
	int		errok;
{
	int		j;
	int		fd;
	char		iname[L_tmpnam];
	int		retval;

	assert(args);
	assert(args->nstrings > 0);

	fd = -1;
	if (input)
	{
		/*
		 * He has given a string to be used as input to the command,
		 * so write it out to a file, and then redirect the input.
		 */
#ifdef __CYGWIN32__
		/*
		 * For some reason CygWin32's tmpnam() always produces
		 * filenames which give ``No such file or directory''
		 */
		{ static int nnn;
		sprintf(iname, "t%dp%d.tmp", ++nnn, getpid()); }
#else
		tmpnam(iname);
#endif
		fd = open(iname, O_RDWR | O_CREAT | O_TRUNC, 0666);
		if (fd < 0)
		{
			error_intl_open(iname);
			exec_fails:
			option_set_errors();
			retval = -1;
			goto ret;
		}
		if (unlink(iname))
		{
			error_intl_unlink(iname);
			goto exec_fails;
		}
		if
		(
			write(fd, input->str_text, input->str_length) < 0
		||
			lseek(fd, 0L, SEEK_SET)
		)
		{
			error_intl_write(iname);
			goto exec_fails;
		}
	}

	for (j = 1; j < args->nstrings; j++)
	{
		char	*s;

		s = args->string[j]->str_text;
		while (*s)
		{
			if (strchr("#|=^();&<>*?[]:$`'\"\\\n", *s))
			{
				string_ty	*str;
				string_list_ty	newcmd;
				char		*cp;

				cp = getenv("SHELL");
				if (!cp || !*cp)
					cp = CONF_SHELL;
				string_list_constructor(&newcmd);
				str = str_from_c(cp);
				string_list_append(&newcmd, str);
				str_free(str);
				if (option_test(OPTION_ERROK))
					str = str_from_c("-c");
				else
					str = str_from_c("-ce");
				string_list_append(&newcmd, str);
				str_free(str);
				str =
					wl2str
					(
						args,
						0,
						args->nstrings - 1,
						(char *)0
					);
				string_list_append(&newcmd, str);
				str_free(str);
				retval =
					execute
					(
						&newcmd,
						fd,
						errok
					);
				string_list_destructor(&newcmd);
				if (input)
					close(fd);
				return retval;
			}
			++s;
		}
	}
	retval = execute(args, fd, errok);
	if (input)
		close(fd);
	ret:
	return retval;
}


/*
 * NAME
 *	os_exists - tests for the existence of a file
 *
 * SYNOPSIS
 *	int os_exists(string_ty *path);
 *
 * DESCRIPTION
 *	The os_exists function is used to determine the existence of a file.
 *
 * RETURNS
 *	int; 1 if the file exists, 0 if it does not.  -1 on error
 */

int
os_exists(path)
	string_ty	*path;
{
	time_t		mtime;
	int		result;

	trace(("os_exists(path = \"%s\")\n{\n", path->str_text));
	mtime = stat_cache_newest(path, 1);
	if (mtime < 0)
		result = -1;
	else
		result = (mtime != 0);
	trace(("return %d\n", result));
	trace(("}\n"));
	return result;
}


int
os_exists_symlink(path)
	string_ty	*path;
{
	time_t		mtime;
	int		result;

	trace(("os_exists_symlink(path = \"%s\")\n{\n", path->str_text));
	mtime = stat_cache_newest(path, 0);
	if (mtime < 0)
		result = -1;
	else
		result = (mtime != 0);
	trace(("return %d\n", result));
	trace(("}\n"));
	return result;
}


int
os_exists_dir(path)
	string_ty	*path;
{
	struct stat	st;
	int		result;

	trace(("os_exists_symlink(path = \"%s\")\n{\n", path->str_text));
#ifdef S_ISLNK
	result = lstat(path->str_text, &st);
#else
	result = stat(path->str_text, &st);
#endif
	result = (result == 0) && S_ISDIR(st.st_mode);
	trace(("return %d\n", result));
	trace(("}\n"));
	return result;
}


/*
 * NAME
 *	os_accdir - return the directory path of the users account
 *
 * SYNOPSIS
 *	string_ty *os_accdir(void);
 *
 * DESCRIPTION
 *	The full pathname of the user's account is returned.
 *	The string will have been dynamically allocated.
 *
 * RETURNS
 *	A pointer to a string in dynamic memory is returned.
 *	A null pointer is returned on error.
 *
 * CAVEAT
 *	Use str_free() when you are done with the value returned.
 */

string_ty *
os_accdir()
{
	static string_ty *s;

	if (!s)
	{
		const char	*cp;

		cp = home_directory();
		assert(cp);
		s = str_from_c(cp);
	}
	return str_copy(s);
}


/*
 * NAME
 *	os_entryname - take path apart
 *
 * SYNOPSIS
 *	string_ty *os_entryname(string_ty *path);
 *
 * DESCRIPTION
 *	Os_entryname is used to extract the entry part
 *	from a pathname.
 *
 * RETURNS
 *	pointer to dynamically allocated string.
 *
 * CAVEAT
 *	Use str_free() when you are done with the return value.
 */

string_ty *
os_entryname(path)
	string_ty	*path;
{
	char		*cp;

	trace(("os_entryname(path = %08lX)\n{\n", path));
	trace_string(path->str_text);
	cp = strrchr(path->str_text, '/');
	if (cp)
		path = str_from_c(cp + 1);
	else
		path = str_copy(path);
	trace_string(path->str_text);
	trace(("return %08lX;\n", path));
	trace((/*{*/"}\n"));
	return path;
}


/*
 * NAME
 *	os_dirname - take path apart
 *
 * SYNOPSIS
 *	string_ty *os_dirname(string_ty *path);
 *
 * DESCRIPTION
 *	Os_dirname is used to extract the directory part
 *	of a pathname.
 *
 * RETURNS
 *	pointer to dynamically allocated string.
 *	A null pointer is returned on error.
 *
 * CAVEAT
 *	Use str_free() when you are done with the value returned.
 */

string_ty *
os_dirname(path)
	string_ty	*path;
{
	char		*cp;

	trace(("os_dirname(path = %08lX)\n{\n"/*}*/, path));
	trace_string(path->str_text);
	cp = strrchr(path->str_text, '/');
	if (cp)
	{
		if (cp > path->str_text)
			path = str_n_from_c(path->str_text, cp - path->str_text);
		else
			path = str_from_c("/");
	}
	else
		path = os_curdir();
	trace_string(path->str_text);
	trace(("return %08lX;\n", path));
	trace((/*{*/"}\n"));
	return path;
}


#ifdef HAVE_PATHCONF

static long pathconf_inner _((char *, int));

static long
pathconf_inner(path, arg)
	char		*path;
	int		arg;
{
	long		result;

	errno = EINVAL; /* IRIX 5.2 fails to set on errors */
	result = pathconf(path, arg);
	if (result < 0)
	{
		switch (errno)
		{
		case ENOSYS: /* lotsa systems say this for EINVAL */
#ifdef EOPNOTSUPP
		case EOPNOTSUPP: /* HPUX says this for EINVAL */
#endif
			errno = EINVAL;
			break;
		}
	}
	return result;
}


static long pathconf_wrapper _((char *, int, long));

static long
pathconf_wrapper(path, arg, default_value)
	char		*path;
	int		arg;
	long		default_value;
{
	long		result;

	result = pathconf_inner(path, arg);
	if (result < 0 && errno == EINVAL)
	{
		/*
		 * probably NFSv2 mounted
		 * assume is same as root
		 */
		result = pathconf_inner("/", arg);
		if (result < 0 && errno == EINVAL)
			result = default_value;
	}
	return result;
}

#endif


static int os_pathconf_path_max _((char *));

static int
os_pathconf_path_max(path)
	char	*path;
{
	long	result;

	result = 1024;
#ifdef HAVE_PATHCONF
	result = pathconf_wrapper(path, _PC_PATH_MAX, result);
	if (result < 0)
	{
		sub_context_ty	*scp;

		scp = sub_context_new();
		sub_errno_set(scp);
		sub_var_set(scp, "File_Name", "%s", path);
		fatal_intl
		(
			scp,
			i18n("pathconf(\"$filename\", {PATH_MAX}): $errno")
		);
		/* NOTREACHED */
	}
#endif
	return result;
}


static int os_pathconf_name_max _((char *));

static int
os_pathconf_name_max(path)
	char	*path;
{
	long	result;

#ifdef HAVE_LONG_FILE_NAMES
	result = 255;
#else
	result = 14;
#endif
#ifdef HAVE_PATHCONF
	result = pathconf_wrapper(path, _PC_NAME_MAX, result);
	if (result < 0)
	{
		sub_context_ty	*scp;

		scp = sub_context_new();
		sub_errno_set(scp);
		sub_var_set(scp, "File_Name", "%s", path);
		fatal_intl
		(
			scp,
			i18n("pathconf(\"$filename\", {NAME_MAX}): $errno")
		);
		/* NOTREACHED */
	}
#endif
	return result;
}


/*
 * NAME
 *	os_legal_path - test if path is legal
 *
 * SYNOPSIS
 *	int os_legal_path(string_ty *path);
 *
 * DESCRIPTION
 *	The os_legal_path function is used to test if each of the components of
 *	the given path are legal to the underlying operating system.
 *
 * RETURNS
 *	int: zero if it is an illegal path, nonzero it is a legal path.
 */

int
os_legal_path(str)
	string_ty	*str;
{
	char		*s;
	char		*ep;
	int		max;

	max = os_pathconf_path_max(".");
	if (str->str_length < 1 || str->str_length > max)
		return 0;
	max = os_pathconf_name_max(".");
	s = str->str_text;
	for (;;)
	{
		ep = strchr(s, '/');
		if (!ep)
			return (strlen(s) <= max);
		if (ep - s > max)
			return 0;
		s = ep + 1;
	}
}


int
os_mkdir(p, echo)
	string_ty	*p;
	int		echo;
{
	char		*cp;

	for (cp = p->str_text; ; ++cp)
	{
		if (cp > p->str_text && (*cp == '/' || *cp == 0))
		{
			string_ty	*tmp;

			tmp = str_n_from_c(p->str_text, cp - p->str_text);
			if (os_exists_dir(tmp))
			{
		        	/*
				 * Do nothing.
				 *
				 * SGI's IRIX 5.2 has a bug: it tests
				 * for permissions before it checks
				 * for existence.  Thus, you can get a
				 * permission denied error for a directory
				 * you couldn't possibly have tried to
				 * create, because it already existed.
				 *
				 * The Solaris automounter pulls a similar
				 * stunt: it says ENOSYS for indirect
				 * top-level directories without looking
				 * to see if the directory exists first.
				 * Amd does not seem to suffer from this.
				 */
			}
			else
			if (mkdir(tmp->str_text, 0777) < 0)
			{
				if (errno != EEXIST)
				{
					sub_context_ty	*scp;

					scp = sub_context_new();
					sub_errno_set(scp);
					sub_var_set(scp, "File_Name", "%S", tmp);
					error_intl
					(
						scp,
						i18n("mkdir $filename: $errno")
					);
					sub_context_delete(scp);
					str_free(tmp);
					option_set_errors();
					return -1;
				}
			}
			else if (echo)
			{
				sub_context_ty	*scp;

				scp = sub_context_new();
				sub_var_set(scp, "File_Name", "%S", tmp);
				error_intl(scp, i18n("mkdir $filename"));
				sub_context_delete(scp);
			}

			/*
			 * if the knew about the existence of the file
			 * before we deleted it, then we will have to
			 * adjust the stat cache.
			 */
			stat_cache_clear(tmp);
			str_free(tmp);
		}
		if (*cp == 0)
			return 0;
	}
}
