/*#####################################################################
#
# Copyright (C) 2000-2001 Peter J Jones (pjones@pmade.org)
# All Rights Reserved
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
#
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 3. Neither the name of the Author nor the names of its contributors
#    may be used to endorse or promote products derived from this software
#    without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND ANY
# EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
# OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
# DAMAGE.
#
#####################################################################*/
#include "StrUtil.hh"

namespace StrUtil {
	using namespace std;
	//# function rm_space_forward #################################################
	/** Remove space from a string
		
		This utility function will remove space from a
		string starting at the position pos until a non-space
		character. It uses isspace to work.

		@param s The string
		@param pos Where to start
		@author Peter Jones
	**/
	//#############################################################################
	void rm_space_forward(string &s, string::size_type pos) {
		while (pos < s.size()) {
			if (isspace(s[pos])) s.erase(pos, 1);
			else break; // no more space
		}
	}
	//# function rm_space_backward ################################################
	/** Remove space from a string backward
		
		This utility function will remove space from a
		string starting at pos and moveing backward until
		a non-space character is found.  It uses isspace
		to work.

		@param s The string
		@param pos Where to start
		@author Peter Jones
	**/
	//#############################################################################
	void rm_space_backward(string &s, string::size_type pos) {
		// safety check
		if (pos >= s.size() || pos < 0) return;

		for (;;) {
			if (isspace(s[pos])) s.erase(pos, 1);
			else break;

			if (pos == 0) break;
			else pos--;
		}
	}
	//# function rm_space_newline #################################################
	/** Remove space before and after a newline
		
		This utility function will remove space before and
		after each newline in the string. It also replaces
		the newline with a space. This function could
		produce a string that has two spaces next to each
		other. If this is a concern, also run the string
		through rm_space_dup.

		@param s a string
		@author Peter Jones
	**/
	//#############################################################################
	void rm_space_newline(string &s) {
		string::size_type pos = string::npos;

		while ((pos = s.find('\n')) != string::npos) {
			s.replace(pos, 1, 1, ' ');
			StrUtil::rm_space_forward(s, pos+1);
			StrUtil::rm_space_backward(s, pos-1);
		}

		// take off any space at the end of the string we may have caused
		StrUtil::rm_space_backward(s, s.size() - 1);
	}
	//# funtion rm_space_dup #####################################################
	/** Turn multiple spaces into one space
		
		This utility function will remove extra spaces in
		between words

		@param s the string
		@author Peter Jones
	**/
	//#############################################################################
	void rm_space_dup(string &s) {
		string::size_type pos = 0;

		// loop until we get to the last char
		while (pos < (s.size() - 1)) {
			if (isspace(s[pos]) && isspace(s[pos + 1]))
				s.erase(pos+1, 1);
			pos++;
		}
	}
	//# function quotemeta ########################################################
	/** Escape meta characters in a string
		
		This utility function will esacpe meta characters
		in a string so that it can be used inside C++
		source.

		@param s the string
		@author Peter Jones
	**/
	//#############################################################################
	void quotemeta(string &s) {
		string::size_type pos = 0;
		char *c;

		while (pos < s.size()) {
			if ((c = strchr("\"\\", s[pos])) != 0) {
				if (*c == '\\') {
					if (pos == (s.size()-1)){
						s.insert(s.end(), '\\');
						break;
					} else if (s[pos+1] != '\\') {
						s.insert(pos++, "\\");
					}
				} else {
					s.insert(pos++, "\\");
				}
			}
			pos++;
		}
	}
	//# function join #############################################################
	/** Join a vector of strings together
		
		This utility function will join a vector of strings
		together.  You can specify a string to "glue" the
		together. This string will only be present inbetween
		the strings, and not on the out sides

		@param sep The "glue" string
		@param list The vector of strings
		@return a string containing the joined strings
		@author Peter Jones
	**/
	//#############################################################################
	string join(string sep, vector<string> &list) {
		vector<string>::iterator it;
		string result;

		for (it=list.begin(); it!=list.end(); it++) {
			if (it != list.begin())
				result += sep;
			result += *it;
		}

		return result;
	}
} // end namespace
