/*
 * Copyright (C) 2000-2001 Peter J Jones (pjones@pmade.org)
 * All Rights Reserved
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Author nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * This file declares the Option class for managing
 * one command line option.
 */

#ifndef __Option_hh__
#define __Option_hh__

// Standard Includes
#include <vector>
#include <string>

// Local Includes
#include "StrUtil.hh"

// Import STD
using namespace std;

//# class Option
/** Class to handle one command line option

	This class contains all the attributes of one command line
	option.  Options can have various attributes, such as a
	long name, a short name, a default value and type.
**/
class Option {
  public:
	  enum Type {
		  type_flag,
		  type_count,
		  type_bool,
		  type_int,
		  type_double,
		  type_string
	  };

	  Option();

	  string			get_name_long() 	const;
	  char   			get_name_short()	const;
	  string::size_type	get_name_length();
	  string			get_argname();
	  string 			get_varname();
	  string 			get_default();
	  string 			get_description();
	  Type   			get_type();

	  bool is_list();
	  bool is_map();
	  bool is_mandatory();
	  bool has_long_name()
		{ return ! m_name_long.empty(); }
	  bool has_short_name()
		{ return m_name_short != '\0'; }

	  void set_name_long(string name);
	  void set_name_short(char name);
	  /// set the argname
	  void set_argname(string argname)
		{ m_argname = argname; }
	  void set_varname(string name);
	  void set_default(string default_value);
	  void set_description(string description);
	  void set_type(Type type);

	  void is_list(bool flag);
	  void is_map(bool flag);
	  void is_mandatory(bool flag);

	  /// struct for comparing Options
	  struct Cmp {
		  bool operator()(Option const &o1, Option const &o2) {
			  string s1, s2;

			  // prepare string one
			  if (o1.get_name_short() == '\0')
				  s1 = o1.get_name_long();
			  else 
				  s1 = o1.get_name_short();

			  // prepare string two
			  if (o2.get_name_short() == '\0')
				  s2 = o2.get_name_long();
			  else
				  s2 = o2.get_name_short();

			  return s1 < s2;
		  }
	  };
  private:
	  string	m_name_long;
	  char		m_name_short;
	  string	m_varname;
	  string	m_default;
	  string	m_description;
	  string	m_argname;
	  Type		m_type;
	  bool		m_is_list;
	  bool		m_is_map;
	  bool		m_is_mandatory;
	  bool		m_appended_desc;
};

#endif
