/*
 * Copyright (C) 2000-2001 Peter J Jones (pjones@pmade.org)
 * All Rights Reserved
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Author nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * This file defines the Generator class that 
 * generates a command line option parser.
 */

#ifndef __Generator_hh__
#define __Generator_hh__

// Standard Includes
#include <fstream>
#include <vector>
#include <algorithm>

// Local Includes
#include "StrUtil.hh"
#include "Option.hh"
#include "Command.hh"

// Import STD
using namespace std;

//# class Generator
/** Generate a C++ header file

	This class generates a C++ header file. The header file
	will be able to parse the argv argument to the mail function.
	It does this by looking for command line options. It knows
	which options to look for at generation time, so this class
	is a container for Option instances.
**/
class Generator {
  public:
	  Generator();
	  
	  /// set the name of the output file
	  void set_filename(string filename)
		{ m_filename = filename; }
	  /// set the name of the file to put the impl code into
	  void set_codefile(string codefile)
		{ m_codefile = codefile; }
	  /// set the name of the generated class
	  void set_classname(string classname)
		{ m_classname = classname; }
	  /// set the name of the generated namespace
	  void set_namespace(string namespacename)
		{ m_namespace = namespacename; }

	  /// set the subcommand_madatory flag
	  void subcommand_mandatory(bool flag)
		{ m_subcommand_mandatory = flag; }
	  // get the subcommand_mandatory flag
	  bool subcommand_mandatory()
		{ return m_subcommand_mandatory; }
	  /// set the autohelp flag
	  void autohelp(bool flag)
		{ m_autohelp = flag; }
	  /// get the autohelp flag
	  bool autohelp()
		{ return m_autohelp; }

	  /// set the autoversion flag
	  void autoversion(bool flag)
		{ m_autoversion = flag; }
	  /// get the autoversion flag
	  bool autoversion()
		{ return m_autoversion; }

	  /// set the nomix flag
	  void nomix(bool flag)
		{ m_nomix = flag; }
	  /// get the nomix flag
	  bool nomix()
		{ return m_nomix; }

	  void add_option(Option &option, bool really=true);
	  void add_subcommand(Command &commnd);
	  void generate();
  private:
	  struct TypeFlags {
		  bool	flag_exists;
		  bool	count_exists;
		  bool	bool_exists;
		  bool	int_exists;
		  bool	double_exists;
		  bool	string_exists;
	  };

	  struct TypeExtensionFlags {
		  bool	bool_map;
		  bool	bool_list;
		  bool	bool_norm;
		  bool	int_map;
		  bool	int_list;
		  bool	int_norm;
		  bool	double_map;
		  bool	double_list;
		  bool	double_norm;
		  bool	string_map;
		  bool	string_list;
		  bool	string_norm;
	  };

	  string::size_type		m_usage_width;
	  TypeFlags				m_types;
	  TypeExtensionFlags	m_type_extensions;
	  vector<Option>		m_options;
	  vector<Command>		m_subcommands;
	  vector<string>		m_varnames;
	  ofstream				m_output_file;
	  string				m_filename;
	  string				m_codefile;
	  string				m_classname;
	  string				m_namespace;
	  bool					m_autohelp;
	  bool					m_autoversion;
	  bool					m_subcommand_mandatory;
	  bool					m_include_map;
	  bool					m_nomix;
	  bool					m_bug_found;
	  bool					m_have_short_options;
	  bool					m_have_long_options;
	  int					m_dent;

	  void indent() { m_dent++; }
	  void outdent(){ m_dent--; }
	  void insert(string what);
	  bool varname_unique(Option &option);
	  void generate_usage(vector<Option> options, string varname);
	  void generate_subcommand_help();
	  void generate_get_function(Option::Type type, bool prototype=false);
	  void generate_parse_loop(Command *command=0);
	  void generate_long_option_test(Command *command=0);
	  void generate_short_option_test(Command *command=0);
	  void generate_value_test(Command *command=0);
	  void generate_mandatory_test(vector<Option> &options);
	  void generate_exception_class();
	  void generate_value_members(vector<Option> &options);
	  void generate_set_defaults();
	  void generate_map_split();
	  void generate_push_filenames();
	  void wrap_description(string &desc, string::size_type spaces);
};

#endif
