/*
 * Copyright (C) 2000-2001 Peter J Jones (pjones@pmade.org)
 * All Rights Reserved
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Author nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include "Generator.hh"
#include "function_constants.hh"

//# method Generator ##########################################################
/** Class constructor
	
	@author Peter Jones
**/
//#############################################################################
Generator::Generator() {
	m_types.flag_exists		= false;
	m_types.count_exists	= false;
	m_types.bool_exists		= false;
	m_types.int_exists		= false;
	m_types.double_exists	= false;
	m_types.string_exists	= false;

	m_type_extensions.bool_map		= false;
	m_type_extensions.bool_list 	= false;
	m_type_extensions.bool_norm		= false;
	m_type_extensions.int_map		= false;
	m_type_extensions.int_list		= false;
	m_type_extensions.int_norm		= false;
	m_type_extensions.double_map	= false;
	m_type_extensions.double_list	= false;
	m_type_extensions.double_norm	= false;
	m_type_extensions.string_map	= false;
	m_type_extensions.string_list	= false;
	m_type_extensions.string_norm	= false;
	
	m_filename = "Clo.hh";
	m_classname = "Parser";
	m_namespace = "Clo";

	m_autohelp = false;
	m_autoversion = false;
	m_subcommand_mandatory = false;
	m_dent = 0;
	m_usage_width = 78;
	m_include_map = false;
	m_nomix = false;
	m_bug_found = false;
	m_have_short_options = false;
	m_have_long_options = false;
}
//# method add_option #########################################################
/** Add a Option instance to the generator
	
	Since the Generator class is a container class for holding
	command line options (Option class), you will need to use
	this method to add command line options to the container.

	@param option An instance of Option
	@param really Really add this option or just process it
	@author Peter Jones
**/
//#############################################################################
void Generator::add_option(Option &option, bool really=true) {
	switch (option.get_type()) {
		case Option::type_flag:
			m_types.flag_exists = true;
			break;
		case Option::type_count:
			m_types.count_exists = true;
			break;
		case Option::type_bool:
			m_types.bool_exists = true;
			if (option.is_map()) m_type_extensions.bool_map = true;
			else if (option.is_list()) m_type_extensions.bool_list = true;
			else m_type_extensions.bool_norm = true;
			break;
		case Option::type_int:
			m_types.int_exists = true;
			if (option.is_map()) m_type_extensions.int_map = true;
			else if (option.is_list()) m_type_extensions.int_list = true;
			else m_type_extensions.int_norm = true;
			break;
		case Option::type_double:
			m_types.double_exists = true;
			if (option.is_map()) m_type_extensions.double_map = true;
			else if (option.is_list()) m_type_extensions.double_list = true;
			else m_type_extensions.double_norm = true;
			break;
		case Option::type_string:
			m_types.string_exists = true;
			if (option.is_map()) m_type_extensions.string_map = true;
			else if (option.is_list()) m_type_extensions.string_list = true;
			else m_type_extensions.string_norm = true;
			break;
	}

	if (!m_include_map && option.is_map()) m_include_map = true;

	// make sure that the varname is unique
	string varname;
	while (! varname_unique(option)) {
		varname = option.get_varname();
		varname += '_';
		option.set_varname(varname);
	}

	m_varnames.push_back(option.get_varname());

	// we only want to do some things if we are adding
	// a option to the vector
	if (really) {
		m_options.push_back(option);
	
		// see if this option has long and short names
		if (!m_have_short_options)
			m_have_short_options = option.has_short_name();
		if (!m_have_long_options)
			m_have_long_options  = option.has_long_name();
	}
}
//# method insert #############################################################
/** Insert a string into the file
	
	This method will insert a string into the file with proper indentation
	and terminate it with a new line.

	@param what A string to insert
	@author Peter Jones
**/
//#############################################################################
void Generator::insert(string what) {
	int tabs = m_dent;

	if (m_dent < 0) {
		if (! m_bug_found) {
			cerr << "clo++: bug found, please email your XML file and ";
			cerr << m_filename << " to clo++@pmade.org" << endl;
			m_bug_found = true;
		}

		m_output_file << "<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<" << endl;
	}

	while (tabs-- > 0) m_output_file << "\t";
	m_output_file << what << endl;
}
//# method varname_unique #####################################################
/** Make sure all options have unique variable names
	
	@param option The Option to check
	@return True if the name is unique, false otherwise
	@author Peter Jones
**/
//#############################################################################
bool Generator::varname_unique(Option &option) {
	vector<string>::iterator	i;

	for (i=m_varnames.begin(); i!=m_varnames.end(); i++) {
		if (option.get_varname() == *i) return false;
	}

	return true;
}
//# method add_subcommand #####################################################
/** Add a subcommand to this generator
	
	This method will add a subcommand to the generator. If
	subcommands are present, special code will be generated to
	handle subcommands.

	@param command The new subcommand
	@author Peter Jones
**/
//#############################################################################
void Generator::add_subcommand(Command &command) {
	vector<Option>::iterator i, end;

	i 	= command.get_options().begin();
	end = command.get_options().end();

	// process the options
	for (;i!=end; i++)
		add_option(*i, false);

	m_subcommands.push_back(command);
}
//# method generate ###########################################################
/** Generate the output file
	
	@author Peter Jones
**/
//#############################################################################
void Generator::generate() {
	vector<Option>::iterator	i, end=m_options.end();
	vector<string>::iterator	varname_i;
	vector<Command>::iterator	ci;

	m_output_file.open(m_filename.c_str());
	if (!m_output_file) {
		cerr << "can't open output file" << endl;
		exit(1);
	}

	insert("// This file generated by clo++, do not edit.");
	insert("#ifndef __" + m_namespace + "_" + m_classname + "__");
	insert("#define __" + m_namespace + "_" + m_classname + "__");
	insert("#include <exception>");
	insert("#include <vector>");
	insert("#include <string>");
	insert("#include <cstring>");

	if (m_include_map) insert("#include <map>");
	if (m_types.int_exists || m_types.double_exists || m_types.bool_exists)
		insert("#include <cctype>");

	if (m_types.int_exists || m_types.double_exists) {
		insert("#include <cstdlib>");
		insert("#include <climits>");
	}

	insert(""); // insert a blank line

	insert("// Use namespaces to keep things clean.");
	insert("namespace " + m_namespace + " {"); indent();
	insert("using namespace std;");

	// bool option types need the truth function for testing
	if (m_types.bool_exists) insert(k_truth_function);

	// usage constants
	generate_usage(m_options, "k_usage");
	generate_subcommand_help();
	for (ci=m_subcommands.begin(); ci!=m_subcommands.end(); ci++)
		generate_usage(ci->get_options(), "k_" + ci->get_varname() + "_usage");

	// class definition
	insert("class " + m_classname + " {"); indent();
	insert("public:"); indent();
	insert("void parse(int argc, char *argv[]);");
	if (m_subcommands.empty()) {
		insert("string usage() {return k_usage;}");
	} else {
		string selse;
		insert("string usage(string command=\"\") {"); indent();
		for (ci=m_subcommands.begin(); ci!=m_subcommands.end(); ci++) {
			if (selse.size()) outdent();
			insert(selse + "if (command == \"" + ci->get_name() + "\") {"); indent();
			insert("return k_" + ci->get_varname() + "_usage;");
			if (selse.empty()) selse = "} else ";
		}
		outdent(); insert("} else {"); indent();
		insert("return k_usage;");
		outdent(); insert("}"); // end else
		outdent(); insert("}"); // end usage method
	}

	insert("vector<string> &get_file_list() {return m_file_list;}");
	if (!m_subcommands.empty())
		insert("string get_command(){return m_command;}");

	// Generate Function Prototypes
	if (m_types.flag_exists) 	generate_get_function(Option::type_flag, true);
	if (m_types.count_exists)	generate_get_function(Option::type_count, true);
	if (m_types.bool_exists) 	generate_get_function(Option::type_bool, true);
	if (m_types.int_exists) 	generate_get_function(Option::type_int, true);
	if (m_types.double_exists) 	generate_get_function(Option::type_double, true);
	if (m_types.string_exists) 	generate_get_function(Option::type_string, true);

	outdent(); insert("private:"); indent();
	insert("struct Values {"); indent();
		generate_value_members(m_options);

		// don't forget the subcommands
		if (!m_subcommands.empty()) {
			vector<Command>::iterator c;
			for (c=m_subcommands.begin(); c!=m_subcommands.end(); c++)
				generate_value_members(c->get_options());
		}
	outdent(); insert("};"); insert("");
	
	// parser state enum
	insert("enum State {");
	indent();
		insert("state_option,");
		insert("state_value");
	outdent();
	insert("};"); insert("");

	// class member data
	insert("vector<string> m_file_list;");
	insert("Values m_values;");
	if (!m_subcommands.empty())
		insert("string m_command;");
	
	// end of class declaration
	outdent(); outdent(); // twice because private:
	insert("}; // End class"); insert("");

	// insert the exception class
	generate_exception_class();
	
	// close this file if codefile is set
	if (m_codefile.size()) {
		// stuff to put on the end of the file
		outdent(); insert("} // End namespace");
		insert("#endif");

		// okay, now open another file
		m_output_file.close();
		m_output_file.open(m_codefile.c_str());
		if (!m_output_file) {
			cerr << "can't open impl file (codefile)" << endl;
			exit(1);
		}

		// stuff for the top of the impl file
		insert("// This file generated by clo++, do not edit.");
		insert("#include \"" + m_filename + "\"");
		insert("// Use namespaces to keep things clean.");
		insert("namespace " + m_namespace + " {"); indent();
		insert("using namespace std;");
	}

	// now the parse method
	insert("void " + m_classname + "::parse(int argc, char *argv[]) {"); indent();

	// Declare some variables
	insert("State  state = state_option;");
	insert("int    option_c = 0;");
	insert("string option_v;");
	insert("string option_passon;");
	insert(""); // blank line

	// List of found options (for mandatory checking)
	insert("struct Found {"); indent();
	for (varname_i=m_varnames.begin(); varname_i!=m_varnames.end(); varname_i++)
		insert("bool " + *varname_i + ";");
	outdent();
	insert("} found_list;"); insert("");

	// initialize the found list to false
	for (varname_i=m_varnames.begin(); varname_i!=m_varnames.end(); varname_i++)
		insert("found_list." + *varname_i + " = false;");

	// set variables to their default value
	generate_set_defaults();

	// Start the parse loop
	insert(""); // blank line
	generate_parse_loop();
	insert(""); // blank line
	generate_mandatory_test(m_options);
	outdent(); insert("} // End parse()");
	insert(""); // blank line

	// now generate the various get_*_option functions
	if (m_types.flag_exists) 	generate_get_function(Option::type_flag);
	if (m_types.count_exists)	generate_get_function(Option::type_count);
	if (m_types.bool_exists) 	generate_get_function(Option::type_bool);
	if (m_types.int_exists)  	generate_get_function(Option::type_int);
	if (m_types.double_exists) 	generate_get_function(Option::type_double);
	if (m_types.string_exists) 	generate_get_function(Option::type_string);

	outdent(); insert("} // End namespace");
	if (m_codefile.empty()) insert("#endif");
	m_output_file.close();
}
//# method generate_usage #####################################################
/** Generate a const string that contains usage info for the given options
	
	@param options A vector of options to generate the string for
	@param varname The name of the string to put the usage info in
	@author Peter Jones
**/
//#############################################################################
void Generator::generate_usage(vector<Option> options, string varname) {
	string::size_type	max=0, current=0, j;
	vector<Option>::iterator i, end;
	string description, argname, name;
	char *c;

	// add fake options for autohelp and autoversion
	if (m_autohelp) {
		Option o;
		o.set_type(Option::type_flag);
		o.set_name_long("help");
		o.set_name_short('h');
		o.set_description("this message");
		options.push_back(o);
	}

	// insert the --help-commands and --help-aliases
	if (m_autohelp && varname == "k_usage" && !m_subcommands.empty()) {
		Option o;
		o.set_type(Option::type_flag);
		o.set_name_long("help-commands");
		o.set_description("print a list of commands and their descriptions");
		options.push_back(o);

		o.set_name_long("help-aliases");
		o.set_description("print a list of commands and their aliases");
		options.push_back(o);
	}

	if (m_autoversion && varname == "k_usage") {
		Option o;
		o.set_type(Option::type_flag);
		o.set_name_long("version");
		o.set_name_short('v');
		o.set_description("print version information");
		options.push_back(o);
	}
		
	// figure out what the longest name is
	for (i=options.begin(), end=options.end(); i!=end; i++) {
		current = i->get_name_length();
		if (current > max) max = current;
	}

	max += 2; // nice dividing space

	insert("const string " + varname + " ="); indent();

	// now sort these guys
	std::sort(options.begin(), options.end(), Option::Cmp());

	for (i=options.begin(), end=options.end(); i!=end; i++) {
		description = i->get_description();
		argname = i->get_argname();
		if (argname.size()) argname.insert(argname.begin(), ' ');

		// first add the correct amount of space
		description.insert(description.begin(), max - i->get_name_length(), ' ');

		// remove strange chars
		StrUtil::quotemeta(description);

		// wrap the description to fit on a term
		wrap_description(
			description, 
			i->get_name_length() + (max - i->get_name_length())
		);

		// add quotes for C++
		description.insert(description.begin(), '"');
		description.insert(description.end(), '\\');
		description.insert(description.end(), 'n');
		description.insert(description.end(), '"');
		
		// figure out what to use for the name
		name = i->get_name_long();

		if (name.size() == 1) {
			name.insert(name.begin(), '-');
		} else if (i->get_name_short() != '\0') {
			name.insert(static_cast<string::size_type>(0), ", --");
			name.insert(name.begin(), i->get_name_short());
			name.insert(name.begin(), '-');
		} else {
			name.insert(static_cast<string::size_type>(0), "--");
		}

		// now insert the data
		insert('"' + name + argname + '"');
		insert(description);
	}

	// put a place holder if there were no options
	if (! options.size()) insert("\"No options for this command\"");
	insert("; // end of " + varname); outdent(); insert("");
}
//# method generate_subcommand_help ###########################################
/** Generate the const strings that hold the list of subcommands and aliases

	@author Peter Jones
**/
//#############################################################################
void Generator::generate_subcommand_help() {
	vector<Command>::iterator	ci;
	string::size_type 			max=0, current=0;
	string 						description;

	// don't do anything unless we have subcommands
	if (m_subcommands.empty()) return;
	// don't continue unless autohelp is on
	if (!m_autohelp) return;
	
	// figure out what the longest command name is
	for (ci=m_subcommands.begin(); ci!=m_subcommands.end(); ci++) {
		current = ci->get_name().size();
		if (current > max) max = current;
	}

	max += 2; // divider

	// sort the list of subcommands so it looks nice
	std::sort(m_subcommands.begin(), m_subcommands.end(), Command::Cmp());

	// now generate the list of commands
	insert("const string k_command_help ="); indent();
	for (ci=m_subcommands.begin(); ci!=m_subcommands.end(); ci++) {
		description = ci->get_description();

		// first add the correct amount of space
		description.insert(description.begin(), max - ci->get_name().size(), ' ');

		// remove strange chars
		StrUtil::quotemeta(description);

		// also tack on the quote
		description.insert(description.begin(), '"');
		description.insert(description.end(), '\\');
		description.insert(description.end(), 'n');
		description.insert(description.end(), '"');
		
		// now insert the data
		insert('"' + ci->get_name() + '"');
		wrap_description(description, ci->get_name().size() + (max - ci->get_name().size()));
		insert(description);
	}

	insert("\"\\n(for help with a specific command, try '<command> --help')\\n\"");
	insert("\"(for a list of command aliases use the --help-aliases option)\\n\"");
	insert("; // end of k_command_help"); outdent(); insert("");

	// now generate the aliases help
	insert("const string k_command_aliases ="); indent();
	for (ci=m_subcommands.begin(); ci!=m_subcommands.end(); ci++) {
		description = StrUtil::join(" ", ci->get_aliases());

		// first add the correct amount of space
		description.insert(description.begin(), max - ci->get_name().size(), ' ');

		// remove strange chars
		StrUtil::quotemeta(description);

		// also tack on the quote
		description.insert(description.begin(), '"');
		description.insert(description.end(), '\\');
		description.insert(description.end(), 'n');
		description.insert(description.end(), '"');
		
		// now insert the data
		insert('"' + ci->get_name() + '"');
		wrap_description(description, ci->get_name().size() + (max - ci->get_name().size()));
		insert(description);
	}

	insert("\"\\n(for help with a specific command, try '<command> --help')\\n\"");
	insert("; // end of k_command_aliases"); outdent(); insert("");
}
//# method generate_get_function ##############################################
/** Generate the various get_*_* function's
	
	@param type A Option::Type
	@param prototype If true, the method will only generate a function prototype
	@author Peter Jones
**/
//#############################################################################
void Generator::generate_get_function(Option::Type type, bool prototype=false) {
	vector<Command>::iterator	ci;
	vector<Option>::iterator 	oi;
	vector<string>				function_names, return_types, extensions;
	string 						arguments, selse, c_selse;
	size_t						i;

	switch (type) {
		case Option::type_flag:
			function_names.push_back("get_flag_option");
			return_types.push_back("bool");
			extensions.push_back("norm");
			break;
		case Option::type_count:
			function_names.push_back("get_count_option");
			return_types.push_back("int");
			extensions.push_back("norm");
			break;
		case Option::type_bool:
			if (m_type_extensions.bool_norm) {
				function_names.push_back("get_bool_option");
				return_types.push_back("bool");
				extensions.push_back("norm");
			}
			if (m_type_extensions.bool_map) {
				function_names.push_back("get_bool_map");
				return_types.push_back("map<string, bool>&");
				extensions.push_back("map");
			}
			if (m_type_extensions.bool_list) {
				function_names.push_back("get_bool_list");
				return_types.push_back("vector<bool>&");
				extensions.push_back("list");
			}
			break;
		case Option::type_int:
			if (m_type_extensions.int_norm) {
				function_names.push_back("get_int_option");
				return_types.push_back("int");
				extensions.push_back("norm");
			}
			if (m_type_extensions.int_map) {
				function_names.push_back("get_int_map");
				return_types.push_back("map<string, int>&");
				extensions.push_back("map");
			}
			if (m_type_extensions.int_list) {
				function_names.push_back("get_int_list");
				return_types.push_back("vector<int>&");
				extensions.push_back("list");
			}
			break;
		case Option::type_double:
			if (m_type_extensions.double_norm) {
				function_names.push_back("get_double_option");
				return_types.push_back("double");
				extensions.push_back("norm");
			}
			if (m_type_extensions.double_map) {
				function_names.push_back("get_double_map");
				return_types.push_back("map<string, double>&");
				extensions.push_back("map");
			}
			if (m_type_extensions.double_list) {
				function_names.push_back("get_double_list");
				return_types.push_back("vector<double>&");
				extensions.push_back("list");
			}
			break;
		case Option::type_string:
			if (m_type_extensions.string_norm) {
				function_names.push_back("get_string_option");
				return_types.push_back("string");
				extensions.push_back("norm");
			}
			if (m_type_extensions.string_map) {
				function_names.push_back("get_string_map");
				return_types.push_back("map<string, string>&");
				extensions.push_back("map");
			}
			if (m_type_extensions.string_list) {
				function_names.push_back("get_string_list");
				return_types.push_back("vector<string>&");
				extensions.push_back("list");
			}
			break;
	}

	// set the arguments that we take
	if (m_subcommands.empty())
		arguments = "string option";
	else 
		arguments = "string option, string command=\"\"";

	// are we just generating a function prototype?
	if (prototype) {
		for (i=0; i<function_names.size(); i++) {
			insert(return_types[i] + " " + function_names[i] + "(" + arguments + ");");
		}
		return;
	}
	
	// function definitions
	for (i=0; i<function_names.size(); i++) {
		// set some stuff up
		c_selse.erase();
		selse.erase();

		// insert the function declaration
		insert(
			return_types[i] 
			+ " " 
			+ m_classname 
			+ "::" 
			+ function_names[i] 
			+ "(" 
			+ arguments 
			+ ") {"
		);

		indent(); // indent for function body

		// first check on subcommands
		if (!m_subcommands.empty()) {
			bool type_found, code_inserted = false;

			for (ci=m_subcommands.begin(); ci!=m_subcommands.end(); ci++) {
				type_found = false;
				for (oi=ci->get_options().begin(); oi!=ci->get_options().end(); oi++) {
					if (oi->get_type() == type){
						if (extensions[i] == "norm")
							type_found = true;
						else if (extensions[i] == "map" && oi->is_map())
							type_found = true;
						else if (extensions[i] == "list" && oi->is_list())
							type_found = true;
					}
				}
				if (!type_found) continue;
				code_inserted = true;
				if (c_selse.size()) outdent();
				insert(c_selse + "if (command == \"" + ci->get_name() + "\") {");
				selse.erase();
				indent();
				for (oi=ci->get_options().begin(); oi!=ci->get_options().end(); oi++) {
					// only check on the correct type and extension
					if (oi->get_type() != type) continue;
					if (extensions[i] == "map" && !oi->is_map()) continue;
					if (extensions[i] == "list" && !oi->is_list()) continue;
					if (extensions[i] == "norm" && (oi->is_map() || oi->is_list())) continue;

					if (selse.size()) outdent();
					insert(selse + "if (option == \"" + oi->get_name_long() + "\") {"); indent();
					insert("return m_values." + oi->get_varname() + ";");
					if (selse.empty()) selse = "} else ";
				}
				outdent(); insert("} else {"); indent();
				insert("throw Exception(\"option '\" + option + \"' is not a " + return_types[i] + " option\\n\");");
				outdent(); insert("}");
				if (c_selse.empty()) c_selse = "} else ";
			}

			if (code_inserted) {
				// clean up after the subcommands
				outdent(); insert("}");
				insert(""); // blank line
				selse.erase();
			}
		}
		

		bool code_inserted = false;
		for (oi=m_options.begin(); oi!=m_options.end(); oi++) {
			// only handle the correct type
			if (oi->get_type() != type) continue;
			if (extensions[i] == "map" && !oi->is_map()) continue;
			if (extensions[i] == "list" && !oi->is_list()) continue;
			if (extensions[i] == "norm" && (oi->is_map() || oi->is_list())) continue;
			
			code_inserted = true;
			if (selse.size()) outdent();
			insert(selse + "if (option == \"" + oi->get_name_long() + "\") {"); indent();
			insert("return m_values." + oi->get_varname() + ";");
			if (selse.empty()) selse = "} else ";
		}

		// FIXME need to throw exception when only
		// subcommands have this option and no subcommand
		// matched the list above
		if (code_inserted) {
			outdent(); insert("} else {"); indent();
				insert("throw Exception(\"option '\" + option + \"' is not a " + return_types[i] + " option\\n\");");
			outdent(); insert("}");
		}

		outdent(); insert("} // End " + function_names[i] + "()");
	}
}
//# method generate_long_option_test ##########################################
/** Generate a if/elseif test for all options
	
	This method will generate a if and else if tree for testing
	to see if the current option matches one from the Command
	Object. If the command object is 0, it will use options from
	the m_options vector.
  
	@param command A pointer to a Command Object
	@author Peter Jones
**/
//#############################################################################
void Generator::generate_long_option_test(Command *command=0) {
	vector<Option> options;
	vector<Option>::iterator oi;
	string selse, command_name;

	insert("// some temp variables");
	insert("string::size_type eq = option_v.find('=');");
	insert("string            tmp_value;"); insert("");

	insert("if (eq != string::npos) {");
	indent();
		insert("tmp_value = option_v.substr(eq + 1);");
		insert("option_v.erase(eq);");
	outdent(); insert("}"); insert("");

	// use the right options vector
	if (command) {
		options = command->get_options();
		command_name = " for the " + command->get_name() + " command";
	} else {
		options = m_options;
	}

	// let the user know about --help
	if (m_autohelp)
		command_name += "\\n(for usage information try the --help option)";

	// look for all the options
	for (oi=options.begin(); oi!=options.end(); oi++) {
		if (oi->get_name_long().size() == 1) continue;
		if (selse.size()) outdent();
		insert(selse + "if (option_v == \"" + oi->get_name_long() + "\") {");
		indent();
		if (oi->get_type() == Option::type_flag) {
			insert("if (tmp_value.size()) throw Exception(argc, argv, option_c, \"--" + oi->get_name_long() + " option does not take an argument\\n\");");
			insert("m_values." + oi->get_varname() + " = true;");
			insert("option_v.erase(); continue;");
		} else if (oi->get_type() == Option::type_count) {
			insert("if (tmp_value.size()) throw Exception(argc, argv, option_c, \"--" + oi->get_name_long() + " option does not take an argument\\n\");");
			insert("m_values." + oi->get_varname() + "++;");
			insert("option_v.erase(); continue;");
		} else {
			insert("if (tmp_value.size()) { option_c--; option_v = tmp_value; }");
			insert("else option_v.erase();");
			insert("state = state_value;");
			insert("option_passon = \"" + oi->get_name_long() + "\";");
		}
		insert("found_list." + oi->get_varname() + " = true;");
		insert("continue;");
		if (selse.empty()) selse = "} else ";
	}

	if (m_autohelp) {
		string varname = command ? "k_" + command->get_varname() + "_usage" : "k_usage";
		string extra = command ? "\"command: " + command->get_name() + "\\n\"+" : "";
		if (selse.size()) outdent();
		insert(selse + "if (option_v == \"help\") {"); indent();
		insert("Exception e(" + extra + varname + "); e.autohelp=true; throw e;");
		if (selse.empty()) selse = "} else ";
	}

	if (m_autohelp && !command && !m_subcommands.empty()) {
		if (selse.size()) outdent();
		insert(selse + "if (option_v == \"help-commands\") {"); indent();
		insert("Exception e(k_command_help); e.autohelp=true; throw e;");
		if (selse.empty()) selse = "} else ";
		if (selse.size()) outdent(); // redundant
		insert(selse + "if (option_v == \"help-aliases\") {"); indent();
		insert("Exception e(k_command_aliases); e.autohelp=true; throw e;");
	}


	if (m_autoversion && !command) {
		if (selse.size()) outdent();
		insert(selse + "if (option_v == \"version\") {"); indent();
		insert("Exception e(\"\"); e.autoversion=true; throw e;");
		if (selse.empty()) selse = "} else ";
	}

	// unknown option gets taken care of here
	outdent(); insert("} else {"); indent();
		insert("throw Exception(argc, argv, option_c, \"option --\" + option_v + \" is not valid" + command_name + "\\n\");");
	outdent(); insert("}");
}	
//# method generate_short_option_test #########################################
/** Generate a if/else tree for short options, see generate_long_option_test

	@param command A pointer to a Command Object
	@author Peter Jones
**/
//#############################################################################
void Generator::generate_short_option_test(Command *command=0) {
	vector<Option> options;
	vector<Option>::iterator oi;
	string selse, command_name;

	// pick the right options
	if (command) {
		options = command->get_options();
		command_name = " for the " + command->get_name() + " command";
	} else {
		options = m_options;
	}

	insert("while (option_v.size()) {");
	indent();
		for (oi=options.begin(); oi!=options.end(); oi++) {
			if (oi->get_name_short() == '\0') continue;
			if (selse.size()) outdent();
			insert(selse + "if (option_v[0] == '" + oi->get_name_short() + "') {");
			indent();
			insert("found_list." + oi->get_varname() + " = true;");
			insert("option_v.erase(0, 1);");
			if (oi->get_type() == Option::type_flag) {
				insert("m_values." + oi->get_varname() + " = true;");
			} else if (oi->get_type() == Option::type_count) {
				insert("m_values." + oi->get_varname() + "++;");
			} else {
				insert("if (option_v.size()) option_c--;");
				insert("state = state_value;");
				insert("option_passon = \"" + oi->get_name_long() + "\";");
				insert("break;");
			}
			if (selse.empty()) selse = "} else ";
		}
		
		if (m_autohelp) {
			string varname = command ? "k_" + command->get_varname() + "_usage" : "k_usage";
			string extra = command ? "\"command: " + command->get_name() + "\\n\"+" : "";
			if (selse.size()) outdent();
			insert(selse + "if (option_v[0] == 'h') {"); indent();
			insert("Exception e(" + extra + varname + "); e.autohelp=true; throw e;");
			if (selse.empty()) selse = "} else ";
		}

		if (m_autoversion && !command) {
			if (selse.size()) outdent();
			insert(selse + "if (option_v[0] == 'v') {"); indent();
			insert("Exception e(\"\"); e.autoversion=true; throw e;");
			if (selse.empty()) selse = "} else ";
		}

		outdent(); insert("} else {"); indent();
			insert("option_v.erase(1);"); // FIXME
			insert("throw Exception(argc, argv, option_c, \"option -\" + option_v + \" is not valid" + command_name + "\\n\");");
		outdent(); insert("}");
	outdent(); insert("}");
}
//# method generate_value_test ################################################
/** Generate code to set the value of a command line option

	Generates a if/else tree to assign values to the correct
	option variable. Checks for options in the Command object
	unless it is 0. If it is 0 then checks for options in
	m_options.

	@param command A pointer to a Command Object to get options from
	@author Peter Jones
**/
//#############################################################################
void Generator::generate_value_test(Command *command=0) {
	vector<Option> options;
	vector<Option>::iterator oi;
	string selse, varname;

	// a coule of constants to help typing
	string const bool_function	= "truth_function(option_v)";
	string const int_function	= "static_cast<int>(strtol(option_v.c_str(), 0, 0))";
	string const double_function= "strtod(option_v.c_str(), 0)";
	string const string_function= "option_v";

	// use the correct options
	if (command)
		options = command->get_options();
	else
		options = m_options;

	for (oi=options.begin(); oi!=options.end(); oi++) {
		if (oi->get_type() == Option::type_flag) continue;
		if (oi->get_type() == Option::type_count) continue;
		if (selse.size()) outdent();
		varname = "m_values." + oi->get_varname();

		insert(selse + "if (option_passon == \"" + oi->get_name_long() + "\") {");
		indent();

		switch (oi->get_type()) {
			case Option::type_bool:
				if (oi->is_map()) {
					generate_map_split();
					insert(varname + "[tmp_var] = " + bool_function + ";");
				}
				else if (oi->is_list()) insert(varname + ".push_back(" + bool_function + ");");
				else insert(varname + " = " + bool_function + ";");
				break;
			case Option::type_int:
				if (oi->is_map()) {
					generate_map_split();
					insert(varname + "[tmp_var] = " + int_function + ";");
				}
				else if (oi->is_list()) insert(varname + ".push_back(" + int_function + ");");
				else insert(varname + " = " + int_function + ";");
				break;
			case Option::type_double:
				if (oi->is_map()) {
					generate_map_split();
					insert(varname + "[tmp_var] = " + double_function + ";");
				}
				else if (oi->is_list()) insert(varname + ".push_back(" + double_function + ");");
				else insert(varname + " = " + double_function + ";");
				break;
			case Option:: type_string:
				if (oi->is_map()) {
					generate_map_split();
					insert(varname + "[tmp_var] = " + string_function + ";");
				}
				else if (oi->is_list()) insert(varname + ".push_back(" + string_function + ");");
				else insert(varname + " = " + string_function + ";");
				break;
		}

		if (selse.empty()) selse = "} else ";
	}

	if (selse.size()) { outdent(); insert("}"); } // all options could be flags
	insert("state = state_option;");
	insert("option_v.erase();");
}
//# method generate_mandatory_test ############################################
/** Generate a test to check for mandatory options at the end of the loop
	
	@param options A vector of options to check for mandatory setting
	@author Peter Jones
**/
//#############################################################################
void Generator::generate_mandatory_test(vector<Option> &options) {
	vector<Option>::iterator oi;
	string additional;

	if (m_autohelp)
		additional = "\\n(for usage information give the --help option)";

	// make sure mandatory functions were found
	for (oi=options.begin(); oi!=options.end(); oi++) {
		if (!oi->is_mandatory()) continue;
		insert("if (!found_list." + oi->get_varname() + ") {");
		indent();
			insert("throw Exception(\"" + oi->get_name_long() + " is a mandatory option" + additional + "\\n\");");
		outdent(); insert("}"); insert("");
	}
}
//# method generate_exception_class ###########################################
/** Generate the class for throwing exceptions
	
	@author Peter Jones
**/
//#############################################################################
void Generator::generate_exception_class() {
	// Generate a class for the Clo::Exceptions
	insert("class Exception : public exception {"); indent();
	insert("public:"); indent();
	insert("Exception(int argc, char *argv[], int which, string why) {");
	indent();
		insert("int count=-1; size_t offset=0, length=0; m_why = why;");
		insert("while (++count < argc) {");
		indent();
			insert("if (which == count) length = strlen(argv[count]);");
			insert("if (!length) offset += strlen(argv[count]) + 1;");
			insert("m_fancy += argv[count]; m_fancy += ' ';");
		outdent(); insert("}");
		insert("m_fancy += '\\n';");
		insert("for (count=0; count < offset; count++) m_fancy += ' ';");
		insert("for (count=0; count < length; count++) m_fancy += '^';");
		insert("m_fancy += '\\n'; m_fancy += why;");
		insert("autohelp = autoversion = false;");
	outdent(); insert("} // end constructor");
	insert("Exception(string why) {m_why = why; autohelp=autoversion=false;}");
	insert("const char *what() {return m_why.c_str();}");
	insert("const char *fancy() {if(m_fancy.empty()) return m_why.c_str(); else return m_fancy.c_str();}");
	insert("bool autohelp, autoversion;");
	outdent(); insert("private:"); indent();
	insert("string m_why, m_fancy;");
	outdent(); // end private;
	outdent(); insert("}; // end Exception");
	insert(""); // insert a blank line
}
//# method generate_value_members #############################################
/** Generate the members of the Values struct
	
	@param options A vector of options to add to the struct
	@author Peter Jones
**/
//#############################################################################
void Generator::generate_value_members(vector<Option> &options) {
	vector<Option>::iterator i, end=options.end();

	for(i=options.begin(); i!=end; i++) {
		switch (i->get_type()) {
			case Option::type_count:
				insert("int " + i->get_varname() + ";");
				break;
			case Option::type_flag:
				insert("bool " + i->get_varname() + ";");
				break;
			case Option::type_bool:
				if (i->is_map())  insert("map<string, bool> " + i->get_varname() + ";");
				else if (i->is_list()) insert("vector<bool> " + i->get_varname() + ";");
				else insert("bool " + i->get_varname() + ";");
				break;
			case Option::type_int:
				if (i->is_map())  insert("map<string, int> " + i->get_varname() + ";");
				else if (i->is_list()) insert("vector<int> " + i->get_varname() + ";");
				else insert("int " + i->get_varname() + ";");
				break;
			case Option::type_double:
				if (i->is_map())  insert("map<string, double>" + i->get_varname() + ";");
				else if (i->is_list()) insert("vector<double>" + i->get_varname() + ";");
				else insert("double " + i->get_varname() + ";");
				break;
			case Option::type_string:
				if (i->is_map())  insert("map<string, string> " + i->get_varname() + ";");
				else if (i->is_list()) insert("vector<string> " + i->get_varname() + ";");
				else insert("string " + i->get_varname() + ";");
				break;
		}
	}
}
//# method generate_set_defaults ##############################################
/** Generate code to set the options to their default values
	
	@author Peter Jones
**/
//#############################################################################
void Generator::generate_set_defaults() {
	vector<Command>::iterator	ci;
	vector<Option>::iterator	oi;
	
	// set the values to their defaults
	insert(""); insert("// setup defaults");
	for (oi=m_options.begin(); oi!=m_options.end(); oi++) {
		if (oi->is_list() || oi->is_map()) continue;
		insert("m_values." + oi->get_varname() + " = " + oi->get_default() + ";");
	}

	// and the sub commands too
	for (ci=m_subcommands.begin(); ci!=m_subcommands.end(); ci++)
		for (oi=ci->get_options().begin(); oi!=ci->get_options().end(); oi++) {
			if (oi->is_list() || oi->is_map()) continue;
			insert("m_values." + oi->get_varname() + " = " + oi->get_default() + ";");
		}
}
//# method generate_parse_loop ################################################
/** Generate code to parse the command line for the main program or a command
	
	@param command If non-zero, generates code for this command object
	@author Peter Jones
**/
//#############################################################################
void Generator::generate_parse_loop(Command *command=0) {
	bool have_long_options, have_short_options;
	string selse;

	// set the long and short option bool flags
	if (command) {
		have_long_options = command->has_long_options();
		have_short_options= command->has_short_options();
	} else {
		have_long_options = m_have_long_options;
		have_short_options= m_have_short_options;
	}

	if (m_autohelp || m_autoversion) {
		have_long_options = true;
		have_short_options = true;
	}

	insert("while (++option_c < argc) {"); 
	indent();
		// only accept the option if we don't allready have one
		// loaded. This is so that we can send control back to the
		// top of the loop and have it send control to the part
		// of code that handles setting variables.
		insert("if (option_v.empty()) option_v = argv[option_c];");
		insert("// a double dash will end command line processing");
		insert("// and place remaining command line options in the filename vector");
		insert("if (option_v == \"--\") {");
		indent();
			// only accept a double dash when the state says we can
			insert("if (state == state_option) {");
			indent();
				insert("while (++option_c < argc) m_file_list.push_back(argv[option_c]);");
				insert("break; // leave main while loop");
			outdent(); insert("} else {"); indent();
				insert("throw Exception(\"missing value for '\" + option_passon + \"'\\n\");");
			outdent(); insert("}");
		outdent(); insert("}");
		if (have_long_options || have_short_options) {
			// look for a option if state says so
			insert("if (state == state_option) {");
			indent(); // indent the if state is state_option
			if (have_long_options) {
				insert("if (option_v.size() > 1 && option_v.substr(0, 2) == \"--\") {");
				indent();
					insert("option_v.erase(0, 2);");
					generate_long_option_test(command);
				outdent();
				if (selse.empty()) selse = "} else "; // mark that we were here
			}

			if (have_short_options) {
				insert(selse + "if (option_v.size() > 1 && option_v[0] == '-') {");
				indent();
					insert("option_v.erase(0, 1);");
					generate_short_option_test(command);
				outdent();
				if (selse.empty()) selse = "} else "; // mark that we were here
			}
		} // end if we have long or short options

		if (selse.size()) {
			insert("} else {");
			indent();
		}

		// look for subsommands if we have some and
		// this is not a command run
		if (command == 0 && !m_subcommands.empty()) {
			vector<Command>::iterator ci; string selse;
			vector<string> aliases_v; string aliases_s;
			vector<string>::iterator ai;
			insert("if (m_command.empty() && m_file_list.empty()) {"); indent();
			for (ci=m_subcommands.begin(); ci!=m_subcommands.end(); ci++) {
				aliases_s.erase(); aliases_v = ci->get_aliases();
				for (ai=aliases_v.begin(); ai!=aliases_v.end(); ai++) {
					StrUtil::quotemeta(*ai);
					ai->insert(0, "option_v == \"");
					ai->insert(ai->end(), '"');
				}
				aliases_s = StrUtil::join(" || ", aliases_v);
				if (aliases_s.empty())
					aliases_s = "\"" + ci->get_name() + "\"";
				else
					aliases_s.insert(0, "\"" + ci->get_name() + "\" || ");

				if (selse.size()) outdent();
				insert(selse + "if (option_v == " + aliases_s + ") {"); indent();
				insert("m_command = \"" + ci->get_name() +"\";");
				insert("option_v.erase();");
				generate_parse_loop(ci);
				generate_mandatory_test(ci->get_options());
				if (selse.empty()) selse = "} else ";
			}
			outdent(); insert("}");
			outdent(); insert("}");// end the main if m_command.empty...
		} else {
			if (m_nomix)
				generate_push_filenames();
			else {
				insert("m_file_list.push_back(option_v);");
				insert("option_v.erase();");
			}
		}

		if (selse.size()) {
			outdent();
			insert("}");
		}

		if (have_long_options || have_short_options) {
			outdent();
			insert("} else if (state == state_value) {");
			indent();
				generate_value_test(command);
			outdent();
			insert("}");
		}

	outdent(); insert("} // end parse() while loop");
	insert("if (state == state_value) {");
	indent();
		insert("throw Exception(\"missing value for '\" + option_passon + \"'\\n\");");
	outdent();
	insert("}");

	// check for mandatory subcommand
	if (!m_subcommands.empty() && !command && m_subcommand_mandatory) {
		if (m_autohelp)
			insert("if (m_command.empty()) { throw Exception(\"missing command, try --help-commands option\\n\"); }");
		else
			insert("if (m_command.empty()) { throw Exception(\"missing expected sub-command.\\n\"); }");
	}
}
//# method generate_map_split #################################################
/** Generate code to split a option value into key=value pair for maps
	
	@author Peter Jones
**/
//#############################################################################
void Generator::generate_map_split() {
	insert("string::size_type eq_pos = option_v.find('=');");
	insert("if (eq_pos == string::npos) {"); indent();
	insert("throw Exception(argc, argv, option_c, \"the \" + option_passon + \" option needs a key=value\");");
	outdent(); insert("}");
	insert("string tmp_var = option_v.substr(0, eq_pos);");
	insert("option_v.erase(0, eq_pos + 1);");
}
//# method generate_push_filenames ############################################
/** Generate code to put all remaning options into the file_list vector
	
	@author Peter Jones
**/
//#############################################################################
void Generator::generate_push_filenames() {
	insert("while (option_c < argc) {"); indent();
	insert("if (strcmp(argv[option_c], \"--\") != 0) {"); indent();
	insert("m_file_list.push_back(argv[option_c]);");
	outdent(); insert("}");
	insert("option_c++;");
	outdent(); insert("}");
}
//# method wrap_description ###################################################
/** Wrap a line for use in a usage string
	
	@param desc The string to wrap
	@param spaces The number of spaces to put at the begining of each line
	@author Peter Jones
**/
//#############################################################################
void Generator::wrap_description(string &desc, string::size_type spaces) {
	string::size_type i, last_space = 0, current = spaces;

	i=0; while (isspace(desc[i++]) && i<desc.size());

	for (; i<desc.size(); i++, current++) {
		if (isspace(desc[i])) last_space = i; // save the pos of the last space
		if (current > m_usage_width
			&& last_space != 0
			&& (i+1) != desc.size())
		{
			i = last_space;
			last_space = 0;
			desc.erase(i, 1);
			desc.insert(i, spaces, ' ');
			desc.insert(i, "\\n");
			i += spaces + 2; // the 2 if for the \\n we added
			current = spaces;
		}
	}
}
