/*
 *
 * CLEX File Manager
 *
 * Copyright (C) 2001-2005 Vlado Potisk <vlado_potisk@clex.sk>
 *
 * CLEX is free software without warranty of any kind; see the
 * GNU General Public License as set out in the "COPYING" document
 * which accompanies the CLEX File Manager package.
 *
 * CLEX can be downloaded from http://www.clex.sk
 *
 */

#include <config.h>

#include <sys/types.h>		/* clex.h */
#include <stdlib.h>			/* free() */
#include <string.h>			/* strcmp() */

#include "clex.h"
#include "history.h"

#include "cfg.h"			/* config_num() */
#include "edit.h"			/* edit_update() */
#include "exec.h"			/* execute_cmd() */
#include "inout.h"			/* win_remark() */
#include "panel.h"			/* pan_adjust() */
#include "undo.h"			/* undo_reset() */
#include "util.h"			/* emalloc() */
#include "ustring.h"		/* us_copy() */

#define HS_LIST (panel_hist.hist)
static int hs_alloc = 0;	/* number of allocated entries */
static int hs_cnt;			/* entries in use */
static int pn_index;		/* index for previous/next cmd */
static USTRING save_line = { 0,0 };
						/* for temporary saving of the command line */

void
hist_reconfig(void)
{
	int i;
	static HIST_ENTRY *storage;

	if (hs_alloc) {
		free(storage);
		free(HS_LIST);
	}

	hs_cnt = 0;
	hs_alloc = config_num(CFG_H_SIZE);
	storage = emalloc(hs_alloc * sizeof(HIST_ENTRY));
	HS_LIST = emalloc(hs_alloc * sizeof(HIST_ENTRY *));
	for (i = 0; i < hs_alloc; i++) {
		HS_LIST[i] = storage + i;
		US_INIT(HS_LIST[i]->cmd);
	}

	hist_reset_index();
}

void
hist_prepare(void)
{
	panel_hist.pd->top = -1;
	panel_hist.pd->cnt = hs_cnt;
	panel_hist.pd->curs = pn_index >= 0 ? pn_index : 0;
	pan_adjust(panel_hist.pd);

	panel = panel_hist.pd;
	textline = &line_cmd;
}

void
hist_reset_index(void)
{
	pn_index = -1;
}

/*
 * hist_save() puts the command 'cmd' on the top
 * of the command history list.
 */
void
hist_save(const char *cmd, int failed)
{
	int i;
	FLAG new;
	HIST_ENTRY *x, *top;

	hist_reset_index();

	new = 1;
	for (top = HS_LIST[0], i = 0; i < hs_alloc; i++) {
		x = HS_LIST[i];
		HS_LIST[i] = top;
		top = x;
		if (i == hs_cnt) {
			hs_cnt++;
			break;
		}
		if (strcmp(USTR(top->cmd),cmd) == 0) {
			/* avoid duplicates */
			new = 0;
			break;
		}
	}
	if (new)
		us_copy(&top->cmd,cmd);
	top->failed = failed;
	
	HS_LIST[0] = top;
}

static void
warn_fail(int n)
{
	if (HS_LIST[n]->failed)
		win_remark("this command failed last time");
}

/* copy next (i.e. more recent) command from the history list */
void
cx_hist_next(void)
{
	if (pn_index == -1) {
		win_remark("top of the history list");
		return;
	}

	if (pn_index-- == 0)
		edit_putstr(USTR(save_line));
	else {
		edit_putstr(USTR(HS_LIST[pn_index]->cmd));
		warn_fail(pn_index);
	}
}

/* copy previous (i.e. older) command from the history list */
void
cx_hist_prev(void)
{
	if (pn_index >= hs_cnt - 1) {
		win_remark("bottom of the history list");
		return;
	}

	if (++pn_index == 0)
		us_xchg(&save_line,&line_cmd.line);
	edit_putstr(USTR(HS_LIST[pn_index]->cmd));
	warn_fail(pn_index);
}

void
cx_hist_complete(void)
{
	int i;

	for (i = 0; i < hs_cnt; i++)
		if (strncmp(USTR(HS_LIST[i]->cmd),USTR(line_cmd.line),
		  line_cmd.size) == 0) {
			textline->curs = textline->size;
			edit_insertstr(USTR(HS_LIST[i]->cmd) + textline->size,0);
			warn_fail(i);
			return;
		}
	win_remark("cannot complete this command (no match found)");
}

void
cx_hist_paste(void)
{
	if (panel_hist.pd->cnt && panel_hist.pd->curs >= 0)
		edit_insertstr(USTR(HS_LIST[panel_hist.pd->curs]->cmd),0);
}

void
cx_hist_enter(void)
{
	const char *cmd;

	next_mode = MODE_SPECIAL_RETURN;
	if (line_cmd.size)
		cmd = USTR(line_cmd.line);
	else if (panel_hist.pd->cnt && panel_hist.pd->curs >= 0)
		cmd = USTR(HS_LIST[panel_hist.pd->curs]->cmd);
	else
		return;

	/*
	 * Right after the command execution CLEX will return
	 * from the history mode back to the file mode. This
	 * little hack prevents one needless screen redrawing.
	 */
	panel = ppanel_file->pd;

	if (execute_cmd(cmd)) {
		cx_edit_kill();
		undo_reset();
	}
}

void
cx_hist_del(void)
{
	int i, del;
	HIST_ENTRY *x;

	if ( (del = panel_hist.pd->curs) < 0) {
		win_remark("cannot delete");
		return;
	}

	panel_hist.pd->cnt = --hs_cnt;
	if (panel_hist.pd->curs == panel_hist.pd->cnt) {
		panel_hist.pd->curs--;
		pan_adjust(panel_hist.pd);
	}

	x = HS_LIST[del];
	for (i = del; i < hs_cnt; i++)
		HS_LIST[i] = HS_LIST[i + 1];
	HS_LIST[hs_cnt] = x;

	if (pn_index > del)
		pn_index--;
	else if (pn_index == del)
		hist_reset_index();

	win_panel();
}
