/*
 * Copyright (c) 1996-1998
 *	John-Mark Gurney.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by John-Mark Gurney
 * 4. Neither the name of the author nor the names of any co-contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY John-Mark Gurney AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

char *rcsid = "$Id: cgi.c,v 1.31 1998/03/01 20:13:45 jmg Exp $";

#include "cgi.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <err.h>

#define mymalloc(var,size)	((var) = malloc(size), (!(var)) ? (fprintf(stderr, "%s:%d: out of memory on allocating %d bytes for %s\n", __FILE__, __LINE__, (size), #var), exit(1), (void *)1) : (var) )
#define mycalloc(var,size)	((var) = calloc(1, (size)), (!(var))? (fprintf(stderr, "%s:%d: out of memory on callocating %d bytes for %s\n", __FILE__, __LINE__, (size), #var), exit(1), (void *)1) : (var) )

struct cgivalues {
	char	*name,			/* name is the register name */
		*value;			/* value is the value of the reg */
	struct	cgivalues *next;	/* next is the next in the linklist */
};

struct cgivalueslist {
	struct	cgivalues *head;
	struct	cgivalueslist *next;
};

static struct cgivalues *start = NULL;	/* this is global for the main linklist
					   all fuctions operate on this */
static struct cgivalueslist *holding = NULL; /* this is used to push new
					   entries onto a stack used for
					   saving context */
#ifdef EXIT_REG
static int registered = 0;		/* used to tell if it has resistered
					   with atexit */
#endif

/*	internal function prototypes	*/
static struct cgivalues **cgigetlastinline(void);
					/* returns pointer to NULL in list */
static void cgidealloccgival(void);	/* deallocate start */
static void cgiparsequerystr(char *a);	/* used to parse the query string */
static int cgiinit(void);		/* starts cgi, for first time, auto
					   decodes incoming data */
static char x2c(char *what);
static void unescape_url(char *url);
static char *cwsc(char **a, char b);	/* copies word to a new str through
					    char */


void
cgidealloccgival(void)
{
	struct cgivalues *a, *b;

	a = start;
	while (a) {
		free(a->name);
		free(a->value);
		b = a->next;
		free(a);
		a = b;
	}
	start = 0;
}

int
cgiinit(void)
{
        char *a;
#ifdef EXIT_REG
	if (!registered) {
		registered=1;		/* mark registered true */
		atexit(cgiclose);	/* and register for cleanup when done */
	}
#endif
	if (!(a = getenv("REQUEST_METHOD")))
		return -1;

	if (!strcmp(a, "GET"))
		if ((a = getenv("QUERY_STRING")))
					/* calls parsestr if QUERY_STRING
					   environment variable exists */
			cgiparsequerystr(a);
		else
			return -1;
	else if (!strcmp(a, "POST")) {
		int b;

		if (!(a = getenv("CONTENT_TYPE")) || (a &&
		    strcasecmp("application/x-www-form-urlencoded", a)))
			return -1;

		if (!(a = getenv("CONTENT_LENGTH")))
			return -1;
		b = atoi(a);
		mymalloc(a, sizeof(char)*(b+1));
		b = read(STDIN_FILENO, (void *)a, sizeof(char) * b);
		a[b] = 0;
		cgiparsequerystr(a);
		free(a);
	} else
		return -1;
	return 0;
}

void
cginewquerystr(char *a)
{
	cgidealloccgival();		/* make sure we have a clean start */
	cgiparsequerystr(a);		/* parse the user supplied string */
}

const char *
cgigetvalue(const char *a)
{
	struct cgivalues *b;

	if (start == 0) {		/* only NULL when first time call
					    or when it was deallocated */
		if (cgiinit())		/* start it off */
			return 0;
	}
	b = start;
	while (b) {
		if (!strcmp(b->name, a))
			return b->value;
		b = b->next;
	}
	return 0;
}

void
cgiclose(void)
{
	cgidealloccgival();
	cgiclearstack();
}

void
cgiclear(void)
{
	cgidealloccgival();
}

/*
 * The following two fuctions are:
 * Copyright (c) 1995 The Apache Group.  All rights reserved.
 */

char x2c(char *what)
{
	register char digit;

	digit = (what[0] >= 'A' ? ((what[0] & 0xdf) - 'A')+10 : (what[0] - '0'));
	digit *= 16;
	digit += (what[1] >= 'A' ? ((what[1] & 0xdf) - 'A')+10 : (what[1] - '0'));
	return(digit);
}

void unescape_url(char *url)
{
	register int x,y;

	for(x=0,y=0;url[y];++x,++y)
	{
		if((url[x] = url[y]) == '+')
			url[x] = ' ';
		else				/* added this if to replace the
						   +'s with spaces as per spec
						*/
		if((url[x] = url[y]) == '%')
		{
			url[x] = x2c(&url[y+1]);
			y+=2;
		}
	}
	url[x] = '\0';
}

struct cgivalues **
cgigetlastinline(void)
{
	struct cgivalues **a;

	a = &start;
	while (a[0])
		a = &(a[0]->next);
	return a;
}

void
cgiparsequerystr(char *a)
{
	struct cgivalues **d;
	struct cgivalues *e;
	char *f;

	f = strdup(a);

	d = cgigetlastinline();

	while (*f != '\0') {
		*d = (mycalloc(e, sizeof(struct cgivalues)));
		
		e->name = cwsc(&f, '=');
		unescape_url(e->name);
		
		e->value = cwsc(&f, '&');
		unescape_url(e->value);
		
		d = &(e->next);
	}
	free(f);
}

char *
cwsc(char **a, char b)
{
	int c;
	char *d, *e, *pos;

	e = *a;
	if (strchr(e, b))
		pos = strchr(e, b);
	else
		pos = strchr(e, 0);

	c = (unsigned int)(pos - e);

	mymalloc(d, sizeof(char) * (c + 1));

	strncpy(d, e, c);
	d[c] = 0;

	if (*pos != '\0')	/* see if we are ending at a null */
		*a = pos + 1;	/* if not, point at next char */
	else
		*a = pos;	/* if so, then point at null */

	return d;
}

int
cgidumpentry(FILE *a, char **entries)
{
	int ts, ret;
	char *c, *d, **b;

	ts = 0;
	b = entries;
	while (*b) {
		if (cgigetvalue(*b))
			ts += strlen(cgigetvalue(*b))+1;
		else
			ts += 1;
		b++;
	}

	d = mymalloc(c, sizeof(char) * ts);
	b = entries;

	while (*b) {
		if (cgigetvalue(*b)) {
			strcpy(c, cgigetvalue(*b));
			c[strlen(c)] = 0;
			c += strlen(cgigetvalue(*b)) + 1;
		} else {
			*c = 0;
			c += 1;
		}
		b++;
	}

	ret = fwrite((void *)d, ts, 1, a);

	free(d);

	if (ret != 1)
		return -1;
	return 0;
}

char **
cgigetentry(FILE *a, char **b)
{
	int num, buf_size, pos, tnum;
	char *buf, **ret, **c;

	c = b;
	num = 0;

	if (ungetc(fgetc(a), a) == EOF)		/* used to make sure that it
						   isn't at EOF */
		return 0;

	while (*c++)
		num++;

	mycalloc(ret, (num+1) * sizeof(char *));
	
	tnum = num;
	c = ret;

	for(; num > 0; c++, num--) {
		pos = 0;
		mymalloc(buf, sizeof(char) * (buf_size = 4096));
		while ((buf[pos++] = fgetc(a)) != 0 && !feof(a) && !ferror(a))
		/*while((pos+=fread(buf+pos, 1, buf_size-pos, a)) && !feof(a) && !ferror(a)) {*/
						/* ferror makes sure we don't
						   get into a loop when fgetc
						   returns EOF and it isn't
						   the end of the file */
			if (pos == buf_size)
				buf = realloc(buf, buf_size += 4096);
		
		if (ungetc(fgetc(a), a) == EOF)	/* used to make sure that it
						   isn't at EOF */
			pos--;			/* and if it is, adjust pos */

		buf[pos] = 0;			/* terminate string */
		
		mymalloc(*c, strlen(buf) + 1);
		strcpy(*c, buf);
		free(buf);
	}
	return ret;
}

void
cgifreeentry(char **a)
{
	char **b;

	b = a;
	while (*b) {
		free(*b);
		b++;
	}
	free(a);
	return;
}

int
fhtmlheader(FILE *a)
{
	return fputs("Content-type: text/html\n\n", a);
}

int
htmlheader(void)
{
	return fhtmlheader(stdout);
}

int
mail(char *from, char *to, char *subject, char *body)
{
	FILE *fd;
	struct cgivalues *a;

	a = start;

	if ((fd = popen("/bin/sh", "w")) == 0)
		err(1, "popen /bin/sh");

	while (a) {
		char *z, *y;

		z = y = strdup(a->value);
		while (*y) {
			if (*y == '\r') {
				*(y++) = '\\';
				*y = '\n';
			}
			y++;
		}

		fprintf(fd, "%s=\"%s\"\n", a->name, z);
		a = a->next;
		free(z);
	}
	
	fputs(
"/usr/sbin/sendmail -t << HOPEFULLY_THIS_WILL_NEVER_BE_IN_A_MESSAGE\n", fd);

	if (from)
		fprintf(fd, "From: %s\n", from);

	if (to)
		fprintf(fd, "To: %s\n", to);

	if (subject)
		fprintf(fd, "Subject: %s\n", subject);

	if (body)
		fprintf(fd, "\n%s\n.\n", body);

	fputs("HOPEFULLY_THIS_WILL_NEVER_BE_IN_A_MESSAGE\n", fd);

	fputs("exit\n", fd);

	return pclose(fd);
}

void
cgiaddpair(char *name, char *value)
{
	struct cgivalues **a;
	a = cgigetlastinline();

	mycalloc(a[0], sizeof(struct cgivalues));
	a[0]->name = strdup(name);
	a[0]->value = strdup(value);
}

void
cgipushcurrent(void)
{
	struct cgivalueslist *a;

	mymalloc(a, sizeof(struct cgivalueslist));

	a->head = start;
	a->next = holding;
	holding = a;

	start = 0;
}

int
cgipopcurrent(void)
{
	struct cgivalueslist *a;

	if (start)
		cgiclear();

	if (holding) {
		start = holding->head;
		a = holding;
		holding = holding->next;
		free(a);
		return 1;
	} else {
		start = 0;
		return 0;
	}
}

void
cgirecreate(char **fields, char **data)
{
	if (start)
		cgiclear();

	while (*fields) {
		cgiaddpair(*fields, *data);
		fields++;
		data++;
	}
}

int
cgireplace(char *name, char *value)
{
	struct cgivalues *a;

	a = start;

	while (a && strcmp(name, a->name))
		a = a->next;

	if (a) {
		free(a->value);
		a->value = strdup(value);
		return 0;
	} else
		return 1;
}

void
cgiclearstack(void)
{
	struct cgivalues *a;
	a = start;
	start = NULL;
	while (cgipopcurrent());
	start = a;
}
