/*
*
* $Id: tcphandle.c,v 1.3 2001/06/02 07:10:45 konst Exp $
*
* Copyright (C) 2000 Denis V. Dmitrienko <denis@null.net>
* Copyright (C) 2000 Bill Soudan <soudan@kde.org>
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or (at
* your option) any later version.
*
* This program is distributed in the hope that it will be useful, but
* WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
* USA
*
*/

#include <time.h>

#ifndef _WIN32
#include <unistd.h>
#endif

#include "icqtypes.h"
#include "icq.h"
#include "icqlib.h"

#include "tcp.h"
#include "stdpackets.h"
#include "tcplink.h"

void icq_TCPOnMessageReceived(ICQLINK *link, DWORD uin, const char *message, DWORD id, icq_TCPLink *plink);
void icq_TCPOnURLReceived(ICQLINK *link, DWORD uin, const char *message, DWORD id);
void icq_TCPOnContactReceived(ICQLINK *link, DWORD uin, const char *message, DWORD id, icq_TCPLink *plink);
void icq_TCPOnChatReqReceived(ICQLINK *link, DWORD uin, const char *message, DWORD id);
void icq_TCPOnFileReqReceived(ICQLINK *link, DWORD uin, const char *message, 
   const char *filename, unsigned long filesize, DWORD id);
void icq_TCPProcessAck(ICQLINK *link, icq_Packet *p);
void icq_HandleChatAck(icq_TCPLink *plink, icq_Packet *p, int port);
void icq_HandleChatHello(icq_TCPLink *plink);
void icq_HandleFileHello(icq_TCPLink *plink);
void icq_HandleFileAck(icq_TCPLink *plink, icq_Packet *p, int port);

void icq_TCPProcessPacket(icq_Packet *p, icq_TCPLink *plink)
{
  DWORD uin;
  WORD version;
  WORD command;
  WORD type;
  WORD status;
  DWORD command_type;
  DWORD filesize = 0;
  DWORD port = 0;
  
  const char *message;
  const char *filename = 0;

  icq_PacketBegin(p);
  (void)icq_PacketRead32(p);
  version=icq_PacketRead16(p);
  command=icq_PacketRead16(p);
  (void)icq_PacketRead16(p);

  uin=icq_PacketRead32(p);
  type=icq_PacketRead16(p);
  message=icq_PacketReadString(p);
  (void)icq_PacketRead32(p);
  (void)icq_PacketRead32(p);
  (void)icq_PacketRead32(p);
  (void)icq_PacketRead8(p);
  status=icq_PacketRead16(p);
  command_type=icq_PacketRead16(p);

  switch(type)
  {
    case ICQ_TCP_MSG_MSG:
    case ICQ_TCP_MSG_URL:
    case ICQ_TCP_MSG_MSG_MASS:
    case ICQ_TCP_MSG_URL_MASS:
      p->id=icq_PacketRead32(p);
      break;
    case ICQ_TCP_MSG_CONTACT:
      p->id=icq_PacketRead32(p);
      break;

    case ICQ_TCP_MSG_CHAT:
      (void)icq_PacketReadString(p);
      (void)icq_PacketRead16(p);
      (void)icq_PacketRead16(p);
      port=icq_PacketRead32(p);
      p->id=icq_PacketRead32(p);
      break;

    case ICQ_TCP_MSG_FILE:
      (void)icq_PacketRead16(p);
      (void)icq_PacketRead16(p);
      filename=icq_PacketReadString(p);
      filesize=icq_PacketRead32(p);
      port=icq_PacketRead32(p);
      p->id=icq_PacketRead32(p);
      break;

    default:
      icq_FmtLog(plink->icqlink, ICQ_LOG_WARNING, "unknown message packet, type %x\n", type);
  }

#ifdef TCP_PROCESS_TRACE
  printf("packet processed from uin: %lu:\n", uin);
  printf("   command: %x\ttype: %x\n", command, type);
  printf("   status: %x\tcommand_type: %x\n", status, (int)command_type);
  printf("   message %s\n", message);
  printf("   id: %x\n", (int)p->id);
#endif

  switch(command)
  {
    case ICQ_TCP_MESSAGE:
      switch(type)
      {
	case ICQ_TCP_MSG_MSG:
	case ICQ_TCP_MSG_MSG_MASS:
	  icq_TCPOnMessageReceived(plink->icqlink, uin, message, p->id, plink);
	  break;

	case ICQ_TCP_MSG_URL:
	case ICQ_TCP_MSG_URL_MASS:
	  icq_TCPOnURLReceived(plink->icqlink, uin, message, p->id);
	  break;

	case ICQ_TCP_MSG_CONTACT:
	  icq_TCPOnContactReceived(plink->icqlink, uin, message, p->id, plink);
	  break;

	case ICQ_TCP_MSG_CHAT:
	  icq_TCPOnChatReqReceived(plink->icqlink, uin, message, p->id);
	  break;

	case ICQ_TCP_MSG_FILE:
	  icq_TCPOnFileReqReceived(plink->icqlink, uin, message, filename, filesize, p->id);
	  break;

	default:
	  icq_FmtLog(plink->icqlink, ICQ_LOG_WARNING, "unknown message type %d!\n", type);
	  break;
      }
      break;

    case ICQ_TCP_ACK:
      switch(type) {
	case ICQ_TCP_MSG_CHAT:
	  icq_HandleChatAck(plink, p, port);
	  break;

	case ICQ_TCP_MSG_FILE:
	  icq_HandleFileAck(plink, p, port);
	  break;

	case ICQ_TCP_MSG_MSG:
	case ICQ_TCP_MSG_URL:
	  if(plink->icqlink->icq_RequestNotify) {
	    icq_FmtLog(plink->icqlink, ICQ_LOG_MESSAGE, "received ack %d\n", p->id);
	    (*plink->icqlink->icq_RequestNotify)(plink->icqlink, p->id, ICQ_NOTIFY_ACK, status,
					       (void *)message);
	    (*plink->icqlink->icq_RequestNotify)(plink->icqlink, p->id, ICQ_NOTIFY_SUCCESS, 0, 0);
	  }
	  break;
      }        
      break;

    case ICQ_TCP_CANCEL:
      /* icq_TCPProcessCancel(p); */
      break;

    default:
      icq_FmtLog(plink->icqlink, ICQ_LOG_WARNING, 
			  "unknown packet command %d!\n", command);
  }
}

void icq_TCPProcessCancel(icq_Packet *p)
{
  (void)p;

/*
  find packet in queue
  call notification function
  remove packet from queue
*/
}

int icq_TCPProcessHello(icq_Packet *p, icq_TCPLink *plink)
{       
  /* TCP Hello packet */
  BYTE code;                /* 0xFF - init packet code */
  DWORD version;            /* tcp version */
  DWORD remote_port;        /* remote message listen port */
  DWORD remote_uin;         /* remote uin */
  DWORD remote_ip;          /* remote IP as seen by ICQ server */
  DWORD remote_real_ip;     /* remote IP as seen by client */
  BYTE flags;               /* tcp flags */
  DWORD remote_other_port;  /* remote chat or file listen port */

  icq_PacketBegin(p);
  
  code=icq_PacketRead8(p);
  version=icq_PacketRead32(p);

  if (!(p->length>=26 && code==ICQ_TCP_HELLO))
  {
    icq_FmtLog(plink->icqlink, ICQ_LOG_WARNING, 
      "malformed hello packet received from %s:%d, closing link\n",
      inet_ntoa(*((struct in_addr *)(&(plink->remote_address.sin_addr)))),
      ntohs(plink->remote_address.sin_port));

    icq_TCPLinkClose(plink);
    return 0;
  }
  remote_port=icq_PacketRead32(p);
  remote_uin=icq_PacketRead32(p);
  remote_ip=icq_PacketRead32(p);
  remote_real_ip=icq_PacketRead32(p);
  flags=icq_PacketRead8(p);
  remote_other_port=icq_PacketRead32(p);

  icq_FmtLog(plink->icqlink, ICQ_LOG_MESSAGE, 
    "hello packet received from %lu { version=%d }\n", remote_uin, version);

  plink->remote_version=version;
  plink->remote_uin=remote_uin;
  plink->flags=flags;
  plink->mode&=~TCP_LINK_MODE_HELLOWAIT;

  /* file and chat sessions require additional handling */
  if(plink->type==TCP_LINK_CHAT) icq_HandleChatHello(plink);
  if(plink->type==TCP_LINK_FILE) icq_HandleFileHello(plink);

  return 1;
}

void icq_TCPOnMessageReceived(ICQLINK *link, DWORD uin, const char *message, DWORD id, icq_TCPLink *plink)
{
  char *data;
#ifdef TCP_PACKET_TRACE
  printf("tcp message packet received from %lu { sequence=%x }\n",
	 uin, (int)id);
#endif

  if(link->icq_RecvMessage)
  {
    /* use the current system time for time received */
    time_t t=time(0);
    struct tm *ptime=localtime(&t);
    icq_Packet *pack;
    icq_TCPLink *preallink=icq_FindTCPLink(link, uin, TCP_LINK_MESSAGE);

    data = strdup(message);
    icq_RusConv("wk", data);

    (*link->icq_RecvMessage)(link, uin, ptime->tm_hour, ptime->tm_min,
      ptime->tm_mday, ptime->tm_mon+1, ptime->tm_year+1900, data);

    free(data);

    if(plink != preallink)
    {
/*      if(icq_SpoofedMessage)
	(*icq_SpoofedMessage(uin, ...));*/
    }

    if(plink)
    {
      /* send an acknowledgement to the remote client */
      pack=icq_TCPCreateMessageAck(plink,0);
      icq_PacketAppend32(pack, id);
      icq_PacketSend(pack, plink->socket);
#ifdef TCP_PACKET_TRACE
      printf("tcp message ack sent to uin %lu { sequence=%lx }\n", uin, id);
#endif
      icq_PacketDelete(pack);
    }
  }
}

void icq_TCPOnContactReceived(ICQLINK *link, DWORD uin, const char *message, DWORD id, icq_TCPLink *plink)
{
  int firstcontact, i, x;
  char *tmp, *data = (char *) message;
  icqcontactmsg *ctitem, *contacts, *citem;

  if(link->icq_RecvContact)
  {
    /* use the current system time for time received */
    time_t t=time(0);
    struct tm *ptime=localtime(&t);
    icq_Packet *pack;
    icq_TCPLink *preallink=icq_FindTCPLink(link, uin, TCP_LINK_MESSAGE);

    firstcontact = 1;
    if(tmp = strchr(data, '\xFE')) {
      *tmp = 0;  
      i = atoi(data);
      tmp++;
    }
      
    for(x = 0; i > x; x++) {
      ctitem = (icqcontactmsg *) malloc(sizeof(icqcontactmsg));
      ctitem->next = NULL;
      data = tmp;
      if(tmp = strchr(tmp, '\xFE')) *tmp = 0;
      ctitem->uin = atol(data);
      tmp++; data = tmp;
      if(tmp = strchr(tmp, '\xFE')) *tmp = 0;
      strncpy(ctitem->nick, data, 10);
      ctitem->nick[10] = 0;
      tmp++;

      if(firstcontact) {
	contacts = ctitem;
	citem = ctitem;
	firstcontact = 0;
      } else {
	  citem->next = ctitem;
	  citem = ctitem;
      }
    }

    (*link->icq_RecvContact)(link, uin, ptime->tm_hour, ptime->tm_min,
      ptime->tm_mday, ptime->tm_mon+1, ptime->tm_year+1900, contacts);

    if(plink != preallink)
    {
/*      if(icq_SpoofedMessage)
	(*icq_SpoofedMessage(uin, ...));*/
    }

    if(plink)
    {
      /* send an acknowledgement to the remote client */
      pack=icq_TCPCreateMessageAck(plink,0);
      icq_PacketAppend32(pack, id);
      icq_PacketSend(pack, plink->socket);
      icq_PacketDelete(pack);
    }
  }
}

void icq_TCPOnURLReceived(ICQLINK *link, DWORD uin, const char *message, DWORD id)
{
#ifdef TCP_PACKET_TRACE
  printf("tcp url packet received from %lu { sequence=%lx }\n",
     uin, id);
#endif /*TCP_PACKET_TRACE*/

  if(link->icq_RecvURL) {

    /* use the current system time for time received */
    time_t t=time(0);
    struct tm *ptime=localtime(&t);
    icq_Packet *pack;
    char *pfe, *data;
    icq_TCPLink *plink=icq_FindTCPLink(link, uin, TCP_LINK_MESSAGE);

    /* the URL is split from the description by 0xFE */
    pfe = strchr(message, '\xFE');
    *pfe = 0;

    data = strdup(message);
    icq_RusConv("wk", data);

    (*link->icq_RecvURL)(link, uin, ptime->tm_hour, ptime->tm_min,
       ptime->tm_mday, ptime->tm_mon+1, ptime->tm_year+1900, pfe+1, data);

    free(data);
    /* send an acknowledgement to the remote client */
    pack=icq_TCPCreateURLAck(plink,0);
    icq_PacketAppend32(pack, id);
    icq_PacketSend(pack, plink->socket);
#ifdef TCP_PACKET_TRACE
    printf("tcp message ack sent to %lu { sequence=%lx }\n", uin, id);
#endif
    icq_PacketDelete(pack);
  }
}


