(* camlp4r pa_extend.cmo q_MLast.cmo *)
(***********************************************************************)
(*                                                                     *)
(*                             Camlp4                                  *)
(*                                                                     *)
(*        Daniel de Rauglaudre, projet Cristal, INRIA Rocquencourt     *)
(*                                                                     *)
(*  Copyright 1998 Institut National de Recherche en Informatique et   *)
(*  Automatique.  Distributed only by permission.                      *)
(*                                                                     *)
(***********************************************************************)

(* $Id: pa_o.ml,v 2.14 2000/04/20 10:05:38 ddr Exp $ *)

open Stdpp;
open Pcaml;

Pcaml.no_constructors_arity.val := True;

do Grammar.Unsafe.reinit_gram gram (Plexer.make ());
   Grammar.Unsafe.clear_entry interf;
   Grammar.Unsafe.clear_entry implem;
   Grammar.Unsafe.clear_entry top_phrase;
   Grammar.Unsafe.clear_entry use_file;
   Grammar.Unsafe.clear_entry module_type;
   Grammar.Unsafe.clear_entry module_expr;
   Grammar.Unsafe.clear_entry sig_item;
   Grammar.Unsafe.clear_entry str_item;
   Grammar.Unsafe.clear_entry expr;
   Grammar.Unsafe.clear_entry patt;
   Grammar.Unsafe.clear_entry ctyp;
   Grammar.Unsafe.clear_entry let_binding;
   Grammar.Unsafe.clear_entry class_type;
   Grammar.Unsafe.clear_entry class_expr;
   Grammar.Unsafe.clear_entry class_sig_item;
   Grammar.Unsafe.clear_entry class_str_item;
return ();

value o2b =
  fun
  [ Some _ -> True
  | None -> False ]
;

value mkumin loc f arg =
  match arg with
  [ <:expr< $int:n$ >> when int_of_string n > 0 ->
      let n = "-" ^ n in <:expr< $int:n$ >>
  | <:expr< $flo:n$ >> when float_of_string n > 0.0 ->
      let n = "-" ^ n in <:expr< $flo:n$ >>
  | _ ->
      let f = "~" ^ f in
      <:expr< $lid:f$ $arg$ >> ]
;

external loc_of_node : 'a -> (int * int) = "%field0";

value mklistexp loc last =
  loop True where rec loop top =
    fun
    [ [] ->
        match last with
        [ Some e -> e
        | None -> <:expr< [] >> ]
    | [e1 :: el] ->
        let loc = if top then loc else (fst (loc_of_node e1), snd loc) in
        <:expr< [$e1$ :: $loop False el$] >> ]
;

value mklistpat loc last =
  loop True where rec loop top =
    fun
    [ [] ->
        match last with
        [ Some p -> p
        | None -> <:patt< [] >> ]
    | [p1 :: pl] ->
        let loc = if top then loc else (fst (loc_of_node p1), snd loc) in
        <:patt< [$p1$ :: $loop False pl$] >> ]
;

value neg s = string_of_int (- int_of_string s);

value is_operator =
  let ht = Hashtbl.create 73 in
  let ct = Hashtbl.create 73 in
  do List.iter (fun x -> Hashtbl.add ht x True)
       ["asr"; "land"; "lor"; "lsl"; "lsr"; "lxor"; "mod"; "or"];
     List.iter (fun x -> Hashtbl.add ct x True)
       ['!'; '&'; '*'; '+'; '-'; '/'; ':'; '<'; '='; '>'; '@'; '^'; '|';
        '~'; '?'; '%'; '.'];
  return
  fun x -> try Hashtbl.find ht x with
  [ Not_found -> try Hashtbl.find ct x.[0] with [ _ -> False ] ]
;

(*
value p_operator strm =
  match Stream.peek strm with
  [ Some (Token.Tterm "(") ->
      match Stream.npeek 3 strm with
      [ [_; Token.Tterm x; Token.Tterm ")"] when is_operator x ->
          do Stream.junk strm; Stream.junk strm; Stream.junk strm; return x
      | _ -> raise Stream.Failure ]
  | _ -> raise Stream.Failure ]
;

value operator = Grammar.Entry.of_parser gram "operator" p_operator;
*)

value operator =
  Grammar.Entry.of_parser gram "operator"
    (parser [: `("", x) when is_operator x :] -> x)
;

value symbolchar =
  let list =
    ['!'; '$'; '%'; '&'; '*'; '+'; '-'; '.'; '/'; ':'; '<'; '='; '>'; '?';
     '@'; '^'; '|'; '~']
  in
  loop where rec loop s i =
    if i == String.length s then True
    else if List.mem s.[i] list then loop s (i + 1)
    else False
;

value prefixop =
  let list = ['!'; '?'; '~'] in
  let excl = ["!="] in
  Grammar.Entry.of_parser gram "prefixop"
    (parser
       [: `("", x) when not (List.mem x excl)
          && String.length x >= 2 && List.mem x.[0] list
          && symbolchar x 1 :] -> x)
;

value infixop0 =
  let list = ['='; '<'; '>'; '|'; '&'; '$'] in
  let excl = ["<-"; "||"; "&&"] in
  Grammar.Entry.of_parser gram "infixop0"
    (parser
       [: `("", x) when not (List.mem x excl)
          && String.length x >= 2 && List.mem x.[0] list
          && symbolchar x 1 :] -> x)
;

value infixop1 =
  let list = ['@'; '^'] in
  Grammar.Entry.of_parser gram "infixop1"
    (parser
       [: `("", x)
          when String.length x >= 2 && List.mem x.[0] list
          && symbolchar x 1 :] -> x)
;

value infixop2 =
  let list = ['+'; '-'] in
  Grammar.Entry.of_parser gram "infixop2"
    (parser
       [: `("", x)
          when x <> "->" && String.length x >= 2 && List.mem x.[0] list
          && symbolchar x 1 :] -> x)
;

value infixop3 =
  let list = ['*'; '/'; '%'] in
  Grammar.Entry.of_parser gram "infixop3"
    (parser
       [: `("", x)
          when String.length x >= 2 && List.mem x.[0] list
          && symbolchar x 1 :] -> x)
;

value infixop4 =
  Grammar.Entry.of_parser gram "infixop4"
    (parser
        [: `("", x)
           when String.length x >= 3 && x.[0] == '*' && x.[1] == '*'
           && symbolchar x 2 :] -> x)
;

value test_constr_decl =
  Grammar.Entry.of_parser gram "test_constr_decl"
    (fun strm ->
       match Stream.npeek 1 strm with
       [ [("UIDENT", _)] ->
           match Stream.npeek 2 strm with
           [ [_; ("", ".")] -> raise Stream.Failure
           | [_; ("", "(")] -> raise Stream.Failure
           | [_ :: _] -> ()
           | _ -> raise Stream.Failure ]
       | [("", "|")] -> ()
       | _ -> raise Stream.Failure ])
;

value stream_peek_nth n strm =
  loop n (Stream.npeek n strm) where rec loop n =
    fun
    [ [] -> None
    | [x] -> if n == 1 then Some x else None
    | [_ :: l] -> loop (n - 1) l ]
;

value test_label_eq =
  let rec test lev strm =
    match stream_peek_nth lev strm with
    [ Some (("UIDENT", _) | ("LIDENT", _) | ("", ".")) ->
        test (lev + 1) strm
    | Some ("", "=") -> ()
    | _ -> raise Stream.Failure ]
  in
  Grammar.Entry.of_parser gram "test_label_eq" (test 1)
;

value constr_arity = ref [("Some", 1); ("Match_Failure", 1)];

value rec constr_expr_arity =
  fun
  [ <:expr< $uid:c$ >> ->
      try List.assoc c constr_arity.val with [ Not_found -> 0 ]
  | <:expr< $uid:_$.$e$ >> -> constr_expr_arity e
  | _ -> 1 ]
;

value rec constr_patt_arity =
  fun
  [ <:patt< $uid:c$ >> ->
      try List.assoc c constr_arity.val with [ Not_found -> 0 ]
  | <:patt< $uid:_$.$p$ >> -> constr_patt_arity p
  | _ -> 1 ]
;

value rec get_seq =
  fun
  [ <:expr< do $list:el$ return $e$ >> ->
      let (el2, e) = get_seq e in
      let el =
        List.fold_right
          (fun e el2 -> let (el, e) = get_seq e in el @ [e :: el2]) el el2
      in
      (el, e)
  | e -> ([], e) ]
;

value choose_tvar tpl =
  let rec find_alpha v =
    let s = String.make 1 v in
    if List.mem s tpl then
      if v = 'z' then None else find_alpha (Char.chr (Char.code v + 1))
    else Some (String.make 1 v)
  in
  let rec make_n n =
    let v = "a" ^ string_of_int n in
    if List.mem v tpl then make_n (succ n) else v
  in
  match find_alpha 'a' with
  [ Some x -> x
  | None -> make_n 1 ]
;

value rec patt_lid =
  fun
  [ <:patt< $lid:i$ $p$ >> -> Some (i, [p])
  | <:patt< $p1$ $p2$ >> ->
      match patt_lid p1 with
      [ Some (i, pl) -> Some (i, [p2 :: pl])
      | None -> None ]
  | _ -> None ]
;

value type_parameter = Grammar.Entry.create gram "type_parameter";
value fun_def = Grammar.Entry.create gram "fun_def";
value fun_binding = Grammar.Entry.create gram "fun_binding";

EXTEND
  GLOBAL: interf implem top_phrase use_file sig_item str_item ctyp patt expr
    module_type module_expr let_binding
    type_parameter fun_def fun_binding;

  (* Main entry points *)

  interf:
    [ [ st = LIST0 [ s = sig_item; OPT ";;" -> (s, loc) ]; EOI -> st ] ]
  ;
  implem:
    [ [ st = LIST0 [ s = str_item; OPT ";;" -> (s, loc) ]; EOI -> st ] ]
  ;
  top_phrase:
    [ [ ph = phrase; ";;" -> Some ph
      | EOI -> None ] ]
  ;
  use_file:
    [ [ l = LIST0 [ ph = phrase; OPT ";;" -> ph ]; EOI -> l ] ]
  ;

  phrase:
    [ [ sti = str_item -> MLast.PhStr loc sti
      | "#"; n = LIDENT; dp = dir_param -> MLast.PhDir loc n dp ] ]
  ;
  dir_param:
    [ [ -> MLast.DpNon
      | s = STRING -> MLast.DpStr s
      | i = INT -> MLast.DpInt i
      | "false" -> MLast.DpIde ["False"]
      | "true" -> MLast.DpIde ["True"]
      | sl = mod_ident -> MLast.DpIde sl ] ]
  ;

  (* Module expressions *)

  module_expr:
    [ [ "functor"; "("; i = UIDENT; ":"; t = module_type; ")"; "->";
        me = module_expr ->
          <:module_expr< functor ( $i$ : $t$ ) -> $me$ >>
      | "struct"; st = LIST0 [ s = str_item; OPT ";;" -> s ]; "end" ->
          <:module_expr< struct $list:st$ end >> ]
    | [ me1 = module_expr; me2 = module_expr ->
          <:module_expr< $me1$ $me2$ >> ]
    | [ i = mod_expr_ident -> i
      | "("; me = module_expr; ":"; mt = module_type;")" ->
          <:module_expr< ( $me$ : $mt$ ) >>
      | "("; me = module_expr; ")" ->
          <:module_expr< $me$ >> ] ]
  ;
  mod_expr_ident:
    [ LEFTA
      [ m1 = mod_expr_ident; "."; m2 = mod_expr_ident ->
          <:module_expr< $m1$ . $m2$ >> ]
    | [ m = UIDENT -> <:module_expr< $uid:m$ >> ] ]
  ;
  str_item:
    [ "top"
      [ "exception"; (c, tl) = constructor_declaration ->
          <:str_item< exception $c$ of $list:tl$ >>
      | "external"; i = LIDENT; ":"; t = ctyp; "="; pd = LIST1 STRING ->
          <:str_item< external $i$ : $t$ = $list:pd$ >>
      | "external"; "("; i = operator; ")"; ":"; t = ctyp; "=";
        pd = LIST1 STRING ->
          <:str_item< external $i$ : $t$ = $list:pd$ >>
      | "module"; i = UIDENT; mb = module_binding ->
          <:str_item< module $i$ = $mb$ >>
      | "module"; "type"; i = UIDENT; "="; mt = module_type ->
          <:str_item< module type $i$ = $mt$ >>
      | "open"; i = mod_ident ->
          <:str_item< open $i$ >>
      | "type"; tdl = LIST1 type_declaration SEP "and" ->
          <:str_item< type $list:tdl$ >>
      | "let"; r = OPT "rec"; l = LIST1 let_binding SEP "and"; "in";
        x = expr ->
          let e = <:expr< let $rec:o2b r$ $list:l$ in $x$ >> in
          <:str_item< $exp:e$ >>
      | "let"; r = OPT "rec"; l = LIST1 let_binding SEP "and" ->
          match l with
          [ [(<:patt< _ >>, e)] -> <:str_item< $exp:e$ >>
          | _ -> <:str_item< value $rec:o2b r$ $list:l$ >> ]
      | "let"; "module"; m = UIDENT; mb = module_binding; "in"; e = expr ->
          <:str_item< let module $m$ = $mb$ in $e$ >>
      | e = expr ->
          <:str_item< $exp:e$ >> ]]
  ;
  module_binding:
    [ RIGHTA
      [ "("; m = UIDENT; ":"; mt = module_type; ")"; mb = module_binding ->
          <:module_expr< functor ( $m$ : $mt$ ) -> $mb$ >>
      | ":"; mt = module_type; "="; me = module_expr ->
          <:module_expr< ( $me$ : $mt$ ) >>
      | "="; me = module_expr ->
          <:module_expr< $me$ >> ] ]
  ;

  (* Module types *)

  module_type:
    [ [ "functor"; "("; i = UIDENT; ":"; t = module_type; ")"; "->";
        mt = module_type ->
          <:module_type< functor ( $i$ : $t$ ) -> $mt$ >> ]
    | [ mt = module_type; "with"; wcl = LIST1 with_constr SEP "and" ->
          <:module_type< $mt$ with $list:wcl$ >> ]
    | [ "sig"; sg = LIST0 [ s = sig_item; OPT ";;" -> s ]; "end" ->
          <:module_type< sig $list:sg$ end >>
      | i = mod_type_ident -> i
      | "("; mt = module_type; ")" ->
          <:module_type< $mt$ >> ] ]
  ;
  mod_type_ident:
    [ LEFTA
      [ m1 = mod_type_ident; "."; m2 = mod_type_ident ->
          <:module_type< $m1$ . $m2$ >>
      | m1 = mod_type_ident; "("; m2 = mod_type_ident; ")" ->
          <:module_type< $m1$ $m2$ >> ]
    | [ m = UIDENT -> <:module_type< $uid:m$ >>
      | m = LIDENT -> <:module_type< $lid:m$ >> ] ]
  ;
  sig_item:
    [ "top"
      [ "exception"; (c, tl) = constructor_declaration ->
          <:sig_item< exception $c$ of $list:tl$ >>
      | "external"; i = LIDENT; ":"; t = ctyp; "="; pd = LIST1 STRING ->
          <:sig_item< external $i$ : $t$ = $list:pd$ >>
      | "external"; "("; i = operator; ")"; ":"; t = ctyp; "=";
        pd = LIST1 STRING ->
          <:sig_item< external $i$ : $t$ = $list:pd$ >>
      | "include"; mt = module_type ->
          <:sig_item< include $mt$ >>
      | "module"; i = UIDENT; mt = module_declaration ->
          <:sig_item< module $i$ : $mt$ >>
      | "module"; "type"; i = UIDENT; "="; mt = module_type ->
          <:sig_item< module type $i$ = $mt$ >>
      | "open"; i = mod_ident ->
          <:sig_item< open $i$ >>
      | "type"; tdl = LIST1 type_declaration SEP "and" ->
          <:sig_item< type $list:tdl$ >>
      | "val"; i = LIDENT; ":"; t = ctyp ->
          <:sig_item< value $i$ : $t$ >>
      | "val"; "("; i = operator; ")"; ":"; t = ctyp ->
          <:sig_item< value $i$ : $t$ >> ] ]
  ;
  module_declaration:
    [ RIGHTA
      [ ":"; mt = module_type ->
          <:module_type< $mt$ >>
      | "("; i = UIDENT; ":"; t = module_type; ")"; mt = module_declaration ->
          <:module_type< functor ( $i$ : $t$ ) -> $mt$ >> ] ]
  ;

  (* "with" constraints (additional type equations over signature
     components) *)

  with_constr:
    [ [ "type"; tp = type_parameters; i = mod_ident; "="; t = ctyp ->
          MLast.WcTyp loc i tp t
      | "module"; i = mod_ident; "="; mt = module_type ->
          MLast.WcMod loc i mt ] ]
  ;

  (* Core expressions *)

  expr:
    [ "top" LEFTA
      [ e1 = expr; ";"; e2 = expr ->
          let (el, e) = get_seq <:expr< do $e1$; return $e2$ >> in
          <:expr< do $list:el$ return $e$ >>
      | e1 = expr; ";" -> e1 ]
    | "expr1"
      [ "let"; o = OPT "rec"; l = LIST1 let_binding SEP "and"; "in";
        x = expr LEVEL "top" ->
          <:expr< let $rec:o2b o$ $list:l$ in $x$ >>
      | "let"; "module"; m = UIDENT; mb = module_binding; "in";
         e = expr LEVEL "top" ->
          <:expr< let module $m$ = $mb$ in $e$ >>
      | "function"; OPT "|"; l = LIST1 match_case SEP "|" ->
          <:expr< fun [ $list:l$ ] >>
      | "fun"; p = patt LEVEL "simple"; e = fun_def ->
          <:expr< fun [$p$ -> $e$] >>
      | "match"; x = expr; "with"; OPT "|"; l = LIST1 match_case SEP "|" ->
          <:expr< match $x$ with [ $list:l$ ] >>
      | "try"; x = expr; "with"; OPT "|"; l = LIST1 match_case SEP "|" ->
          <:expr< try $x$ with [ $list:l$ ] >>
      | "if"; e1 = expr; "then"; e2 = expr LEVEL "expr1";
        e3 = [ "else"; e = expr LEVEL "expr1" -> e | -> <:expr< () >> ] ->
          <:expr< if $e1$ then $e2$ else $e3$ >>
      | "for"; i = LIDENT; "="; e1 = expr; df = direction_flag; e2 = expr;
        "do"; e = expr; "done" ->
          let (el, e) = get_seq e in
          let el = el @ [e] in
          <:expr< for $i$ = $e1$ $to:df$ $e2$ do $list:el$ done >>
      | "while"; e1 = expr; "do"; e2 = expr; "done" ->
          let (el, e) = get_seq e2 in
          let el = el @ [e] in
          <:expr< while $e1$ do $list:el$ done >> ]
    | [ e = expr; ","; el = LIST1 NEXT SEP "," ->
          <:expr< ( $list:[e :: el]$ ) >> ]
    | ":=" NONA
      [ e1 = expr; ":="; e2 = expr LEVEL "expr1" -> <:expr< $e1$.val := $e2$ >>
      | e1 = expr; "<-"; e2 = expr LEVEL "expr1" -> <:expr< $e1$ := $e2$ >> ]
    | "||" RIGHTA
      [ e1 = expr; f = [ op = "or" -> op | op = "||" -> op ]; e2 = expr ->
          <:expr< $lid:f$ $e1$ $e2$ >> ]
    | "&&" RIGHTA
      [ e1 = expr; f = [ op = "&" -> op | op = "&&" -> op ]; e2 = expr ->
          <:expr< $lid:f$ $e1$ $e2$ >> ]
    | "<" LEFTA
      [ e1 = expr;
        f = [ op = "<" -> op
            | op = ">" -> op
            | op = "<=" -> op
            | op = ">=" -> op
            | op = "=" -> op
            | op = "<>" -> op
            | op = "==" -> op
            | op = "!=" -> op
            | op = infixop0 -> op ];
        e2 = expr ->
          <:expr< $lid:f$ $e1$ $e2$ >> ]
    | "^" RIGHTA
      [ e1 = expr;
        f = [ op = "^" -> op
            | op = "@" -> op
            | op = infixop1 -> op ];
        e2 = expr ->
          <:expr< $lid:f$ $e1$ $e2$ >> ]
    | RIGHTA
      [ e1 = expr; "::"; e2 = expr -> <:expr< [$e1$ :: $e2$] >> ]
    | "+" LEFTA
      [ e1 = expr;
        f = [ op = "+" -> op
            | op = "-" -> op
            | op = "+." -> op
            | op = "-." -> op
            | op = infixop2 -> op ];
        e2 = expr ->
          <:expr< $lid:f$ $e1$ $e2$ >> ]
    | "*" LEFTA
      [ e1 = expr;
        f = [ op = "*" -> op
            | op = "/" -> op
            | op = "*." -> op
            | op = "/." -> op
            | op = "land" -> op
            | op = "lor" -> op
            | op = "lxor" -> op
            | op = "mod" -> op
            | op = infixop3 -> op ];
        e2 = expr ->
          <:expr< $lid:f$ $e1$ $e2$ >> ]
    | "**" RIGHTA
      [ e1 = expr;
        f = [ op = "**" -> op
            | op = "asr" -> op
            | op = "lsl" -> op
            | op = "lsr" -> op
            | op = infixop4 -> op ];
        e2 = expr ->
          <:expr< $lid:f$ $e1$ $e2$ >> ]
    | "unary minus" NONA
      [ f = [ op = "-" -> op
            | op = "-." -> op ];
        e = expr ->
          <:expr< $mkumin loc f e$ >> ]
    | "apply" LEFTA
      [ e1 = expr; e2 = expr ->
          match constr_expr_arity e1 with
          [ 1 -> <:expr< $e1$ $e2$ >>
          | _ ->
              match e2 with
              [ <:expr< ( $list:el$ ) >> ->
                  List.fold_left (fun e1 e2 -> <:expr< $e1$ $e2$ >>) e1 el
              | _ -> <:expr< $e1$ $e2$ >> ] ]
      | "assert"; e = expr LEVEL "simple" ->
          let f = <:expr< $str:input_file.val$ >> in
          let bp = <:expr< $int:string_of_int (fst loc)$ >> in
          let ep = <:expr< $int:string_of_int (snd loc)$ >> in
          let raiser =
            <:expr< raise (Pervasives.Assert_failure ($f$, $bp$, $ep$)) >>
          in
          match e with
          [ <:expr< False >> -> raiser
          | _ ->
              if no_assert.val then <:expr< () >>
              else <:expr< if $e$ then () else $raiser$ >> ]
      | "lazy"; e = expr ->
          <:expr< Pervasives.ref (Lazy.Delayed (fun () -> $e$)) >> ]
    | "simple" LEFTA
      [ e1 = expr; "."; "("; e2 = expr; ")" -> <:expr< $e1$ .( $e2$ ) >>
      | e1 = expr; "."; "["; e2 = expr; "]" -> <:expr< $e1$ .[ $e2$ ] >>
      | e1 = expr; "."; e2 = expr -> <:expr< $e1$ . $e2$ >>
      | "!"; e = expr -> <:expr< $e$ . val>>
      | f = [ op = "~-" -> op
            | op = "~-." -> op
            | op = prefixop -> op ];
        e = expr ->
          <:expr< $lid:f$ $e$ >>
      | s = INT -> <:expr< $int:s$ >>
      | s = FLOAT -> <:expr< $flo:s$ >>
      | s = STRING -> <:expr< $str:s$ >>
      | c = CHAR -> <:expr< $chr:c.[0]$ >>
      | i = expr_ident -> i
      | s = "false" -> <:expr< False >>
      | s = "true" -> <:expr< True >>
      | "["; "]" -> <:expr< [] >>
      | "["; el = expr1_semi_list; "]" -> <:expr< $mklistexp loc None el$ >>
      | "[|"; "|]" -> <:expr< [| |] >>
      | "[|"; el = expr1_semi_list; "|]" -> <:expr< [| $list:el$ |] >>
      | "{"; test_label_eq; lel = lbl_expr_list; "}" ->
          <:expr< { $list:lel$ } >>
      | "{"; e = expr LEVEL "simple"; "with"; lel = lbl_expr_list; "}" ->
          <:expr< { ($e$) with $list:lel$ } >>
      | "("; ")" -> <:expr< () >>
      | "("; e = expr; ":"; t = ctyp; ")" -> <:expr< ($e$ : $t$) >>
      | "("; e = expr; ")" -> <:expr< $e$ >>
      | "("; "-"; ")" -> <:expr< $lid:"-"$ >>
      | "("; "-."; ")" -> <:expr< $lid:"-."$ >>
      | "("; op = operator; ")" -> <:expr< $lid:op$ >>
      | "begin"; e = expr; "end" -> <:expr< $e$ >>
      | x = LOCATE ->
          let x =
            try
              let i = String.index x ':' in
              (int_of_string (String.sub x 0 i),
               String.sub x (i + 1) (String.length x - i - 1))
            with
            [ Not_found | Failure _ -> (0, x) ]
          in
          Pcaml.handle_expr_locate loc x
      | x = QUOTATION ->
          let x =
            try
              let i = String.index x ':' in
              (String.sub x 0 i,
               String.sub x (i + 1) (String.length x - i - 1))
            with
            [ Not_found -> ("", x) ]
          in
          Pcaml.handle_expr_quotation loc x ] ]
  ;
  let_binding:
    [ [ p = patt; e = fun_binding ->
          match patt_lid p with
          [ Some (i, pl) ->
              let e =
                List.fold_left (fun e p -> <:expr< fun $p$ -> $e$ >>) e pl
              in
              (<:patt< $lid:i$ >>, e)
          | None -> (p, e) ] ] ]
  ;
  fun_binding:
    [ RIGHTA
      [ p = patt LEVEL "simple"; e = fun_binding -> <:expr< fun $p$ -> $e$ >>
      | "="; e = expr -> <:expr< $e$ >>
      | ":"; t = ctyp; "="; e = expr -> <:expr< ($e$ : $t$) >> ] ]
  ;
  match_case:
    [ [ x1 = patt; w = OPT [ "when"; e = expr -> e ]; "->"; x2 = expr ->
          (x1, w, x2) ] ]
  ;
  lbl_expr_list:
    [ [ le = lbl_expr; ";"; lel = lbl_expr_list -> [le :: lel]
      | le = lbl_expr; ";" -> [le]
      | le = lbl_expr -> [le] ] ]
  ;
  lbl_expr:
    [ [ i = expr_ident; "="; e = (expr LEVEL "expr1") -> (i, e) ] ]
  ;
  expr1_semi_list:
    [ [ e = expr LEVEL "expr1"; ";"; el = expr1_semi_list -> [e :: el]
      | e = expr LEVEL "expr1"; ";" -> [e]
      | e = expr LEVEL "expr1" -> [e] ] ]
  ;
  fun_def:
    [ RIGHTA
      [ p = patt LEVEL "simple"; e = fun_def -> <:expr< fun $p$ -> $e$ >>
      | "->"; e = expr -> <:expr< $e$ >> ] ]
  ;
  expr_ident:
    [ RIGHTA
      [ i = LIDENT -> <:expr< $lid:i$ >>
      | i = UIDENT -> <:expr< $uid:i$ >>
      | m = UIDENT; "."; i = expr_ident ->
          loop <:expr< $uid:m$ >> i where rec loop m =
            fun
            [ <:expr< $x$ . $y$ >> -> loop <:expr< $m$ . $x$ >> y
            | e -> <:expr< $m$ . $e$ >> ]
      | m = UIDENT; "."; "("; i = operator; ")" ->
          <:expr< $uid:m$ . $lid:i$ >> ] ]
  ;

  (* Patterns *)

  patt:
    [ LEFTA
      [ p1 = patt; "as"; i = LIDENT -> <:patt< ($p1$ as $lid:i$) >> ]
    | LEFTA
      [ p1 = patt; "|"; p2 = patt -> <:patt< $p1$ | $p2$ >> ]
    | [ p = patt; ","; pl = LIST1 NEXT SEP "," ->
          <:patt< ( $list:[p :: pl]$) >> ]
    | NONA
      [ p1 = patt; ".."; p2 = patt -> <:patt< $p1$ .. $p2$ >> ]
    | RIGHTA
      [ p1 = patt; "::"; p2 = patt -> <:patt< [$p1$ :: $p2$] >> ]
    | LEFTA
      [ p1 = patt; p2 = patt ->
          match constr_patt_arity p1 with
          [ 1 -> <:patt< $p1$ $p2$ >>
          | n ->
              let p2 =
                match p2 with
                [ <:patt< _ >> when n > 1 ->
                    let pl =
                      loop n where rec loop n =
                        if n = 0 then [] else [ <:patt< _ >> :: loop (n - 1) ]
                    in
                    <:patt< ( $list:pl$ ) >>
                | _ -> p2 ]
              in
              match p2 with
              [ <:patt< ( $list:pl$ ) >> ->
                  List.fold_left (fun p1 p2 -> <:patt< $p1$ $p2$ >>) p1 pl
              | _ -> <:patt< $p1$ $p2$ >> ] ] ]
    | LEFTA
      [ p1 = patt; "."; p2 = patt -> <:patt< $p1$ . $p2$ >> ]
    | "simple"
      [ s = LIDENT -> <:patt< $lid:s$ >>
      | s = UIDENT -> <:patt< $uid:s$ >>
      | s = INT -> <:patt< $int:s$ >>
      | "-"; s = INT -> <:patt< $int:neg s$ >>
      | s = STRING -> <:patt< $str:s$ >>
      | s = CHAR -> <:patt< $chr:s.[0]$ >>
      | s = "false" -> <:patt< False >>
      | s = "true" -> <:patt< True >>
      | "["; "]" -> <:patt< [] >>
      | "["; pl = patt_semi_list; "]" -> <:patt< $mklistpat loc None pl$ >>
      | "[|"; "|]" -> <:patt< [| |] >>
      | "[|"; pl = patt_semi_list; "|]" -> <:patt< [| $list:pl$ |] >>
      | "{"; lpl = lbl_patt_list; "}" -> <:patt< { $list:lpl$ } >>
      | "("; ")" -> <:patt< () >>
      | "("; p = patt; ":"; t = ctyp; ")" -> <:patt< ($p$ : $t$) >>
      | "("; p = patt; ")" -> <:patt< $p$ >>
      | "("; "-"; ")" -> <:patt< $lid:"-"$ >>
      | "("; op = operator; ")" -> <:patt< $lid:op$ >>
      | "_" -> <:patt< _ >>
      | x = LOCATE ->
          let x =
            try
              let i = String.index x ':' in
              (int_of_string (String.sub x 0 i),
               String.sub x (i + 1) (String.length x - i - 1))
            with
            [ Not_found | Failure _ -> (0, x) ]
          in
          Pcaml.handle_patt_locate loc x
      | x = QUOTATION ->
          let x =
            try
              let i = String.index x ':' in
              (String.sub x 0 i,
               String.sub x (i + 1) (String.length x - i - 1))
            with
            [ Not_found -> ("", x) ]
          in
          Pcaml.handle_patt_quotation loc x ] ]
  ;
  patt_semi_list:
    [ [ p = patt; ";"; pl = patt_semi_list -> [p :: pl]
      | p = patt; ";" -> [p]
      | p = patt -> [p] ] ]
  ;
  lbl_patt_list:
    [ [ le = lbl_patt; ";"; lel = lbl_patt_list -> [le :: lel]
      | le = lbl_patt; ";" -> [le]
      | le = lbl_patt -> [le] ] ]
  ;
  lbl_patt:
    [ [ i = patt_label_ident; "="; p = patt -> (i, p) ] ]
  ;
  patt_label_ident:
    [ RIGHTA
      [ i = UIDENT -> <:patt< $uid:i$ >>
      | i = LIDENT -> <:patt< $lid:i$ >>
      | m = UIDENT; "."; i = patt_label_ident -> <:patt< $uid:m$ . $i$ >> ] ]
  ;
  
  (* Type declaration *)

  type_declaration:
    [ [ tpl = type_parameters; n = LIDENT; "="; tk = type_kind;
        cl = LIST0 constrain ->
          (n, tpl, tk, cl)
      | tpl = type_parameters; n = LIDENT; cl = LIST0 constrain ->
          (n, tpl, <:ctyp< '$choose_tvar tpl$ >>, cl) ] ]
  ;
  constrain:
    [ [ "constraint"; t1 = ctyp; "="; t2 = ctyp -> (t1, t2) ] ]
  ;
  type_kind:
    [ [ test_constr_decl; OPT "|";
        cdl = LIST1 constructor_declaration SEP "|" ->
          <:ctyp< [ $list:cdl$ ] >>
      | t = ctyp -> <:ctyp< $t$ >>
      | t = ctyp; "="; "{"; ldl = label_declarations; "}" ->
          <:ctyp< $t$ == { $list:ldl$ } >>
      | t = ctyp; "="; OPT "|"; cdl = LIST1 constructor_declaration SEP "|" ->
          <:ctyp< $t$ == [ $list:cdl$ ] >>
      | "{"; ldl = label_declarations; "}" -> <:ctyp< { $list:ldl$ } >> ] ]
  ;
  type_parameters:
    [ [ (* empty *) -> []
      | tp = type_parameter -> [tp]
      | "("; tpl = LIST1 type_parameter SEP ","; ")" -> tpl ] ]
  ;
  type_parameter:
    [ [ "'"; i = ident -> i ] ]
  ;
  constructor_declaration:
    [ [ ci = UIDENT; "of"; cal = LIST1 (ctyp LEVEL "ctyp1") SEP "*" ->
          (ci, cal)
      | ci = UIDENT ->
          (ci, []) ] ]
  ;
  label_declarations:
    [ [ ld = label_declaration; ";"; ldl = label_declarations -> [ld :: ldl]
      | ld = label_declaration; ";" -> [ld]
      | ld = label_declaration -> [ld] ] ]
  ;
  label_declaration:
    [ [ i = LIDENT; ":"; t = ctyp -> (i, False, t)
      | "mutable"; i = LIDENT; ":"; t = ctyp -> (i, True, t) ] ]
  ;

  (* Core types *)

  ctyp:
    [ [ t1 = ctyp; "as"; i = type_parameter -> <:ctyp< $t1$ as '$i$ >> ]
    | "arrow" RIGHTA
      [ t1 = ctyp; "->"; t2 = ctyp -> <:ctyp< $t1$ -> $t2$ >> ]
    | [ t = ctyp; "*"; tl = LIST1 (ctyp LEVEL "ctyp1") SEP "*" ->
          <:ctyp< ( $list:[t :: tl]$ ) >> ]
    | "ctyp1"
      [ t1 = ctyp; t2 = ctyp -> <:ctyp< $t2$ $t1$ >> ]
    | "ctyp2"
      [ t1 = ctyp; "."; t2 = ctyp -> <:ctyp< $t1$ . $t2$ >>
      | t1 = ctyp; "("; t2 = ctyp; ")" -> <:ctyp< $t1$ $t2$ >> ]
    | "simple"
      [ "'"; i = ident -> <:ctyp< '$i$ >>
      | "_" -> <:ctyp< _ >>
      | i = LIDENT -> <:ctyp< $lid:i$ >>
      | i = UIDENT -> <:ctyp< $uid:i$ >>
      | "("; t = ctyp; ","; tl = LIST1 ctyp SEP ","; ")";
        i = ctyp LEVEL "ctyp2"->
          List.fold_left (fun c a -> <:ctyp< $c$ $a$ >>) i [t :: tl]
      | "("; t = ctyp; ")" -> <:ctyp< $t$ >> ] ]
  ;

  (* Identifiers *)

  ident:
    [ [ i = LIDENT -> i
      | i = UIDENT -> i ] ]
  ;
  mod_ident:
    [ RIGHTA
      [ i = UIDENT -> [i]
      | i = LIDENT -> [i]
      | m = UIDENT; "."; i = mod_ident -> [m :: i] ] ]
  ;

  (* Miscellaneous *)

  direction_flag:
    [ [ "to" -> True
      | "downto" -> False ] ]  
  ;
END;

(* Objects and Classes *)

value rec class_type_of_ctyp loc t =
  match t with
  [ <:ctyp< $lid:i$ >> -> <:class_type< $list:[i]$ >>
  | <:ctyp< $uid:m$.$t$ >> -> <:class_type< $list:[m :: type_id_list t]$ >>
  | _ -> raise_with_loc loc (Stream.Error "lowercase identifier expected") ]
and type_id_list =
  fun
  [ <:ctyp< $uid:m$.$t$ >> -> [m :: type_id_list t]
  | <:ctyp< $lid:i$ >> -> [i]
  | t ->
      raise_with_loc (loc_of_node t)
        (Stream.Error "lowercase identifier expected") ]
;

value class_fun_binding = Grammar.Entry.create gram "class_fun_binding";

EXTEND
  GLOBAL: str_item sig_item expr ctyp
    class_sig_item class_str_item class_type class_expr
    class_fun_binding;

  str_item:
    [ [ "class"; cd = LIST1 class_declaration SEP "and" ->
          <:str_item< class $list:cd$ >>
      | "class"; "type"; ctd = LIST1 class_type_declaration SEP "and" ->
          <:str_item< class type $list:ctd$ >> ] ]
  ;
  sig_item:
    [ [ "class"; cd = LIST1 class_description SEP "and" ->
          <:sig_item< class $list:cd$ >>
      | "class"; "type"; ctd = LIST1 class_type_declaration SEP "and" ->
          <:sig_item< class type $list:ctd$ >> ] ]
  ;

  (* Class expressions *)

  class_declaration:
    [ [ vf = OPT "virtual"; ctp = class_type_parameters; i = LIDENT;
        cfb = class_fun_binding ->
          {MLast.ciLoc = loc; MLast.ciVir = o2b vf; MLast.ciPrm = ctp;
           MLast.ciNam = i; MLast.ciExp = cfb} ] ]
  ;
  class_fun_binding:
    [ [ "="; ce = class_expr -> ce
      | ":"; ct = class_type; "="; ce = class_expr ->
          <:class_expr< ($ce$ : $ct$) >>
      | p = patt LEVEL "simple"; cfb = class_fun_binding ->
          <:class_expr< fun $p$ -> $cfb$ >> ] ]
  ;
  class_type_parameters:
    [ [ -> (loc, [])
      | "["; tpl = LIST1 type_parameter SEP ","; "]" -> (loc, tpl) ] ]
  ;
  class_fun_def:
    [ [ p = patt LEVEL "simple"; "->"; ce = class_expr ->
          <:class_expr< fun $p$ -> $ce$ >>
      | p = patt LEVEL "simple"; cfd = class_fun_def ->
          <:class_expr< fun $p$ -> $cfd$ >> ] ]
  ;      
  class_expr:
    [ "top"
      [ "fun"; cfd = class_fun_def -> cfd
      | "let"; rf = OPT "rec"; lb = LIST1 let_binding SEP "and"; "in";
        ce = class_expr ->
          <:class_expr< let $rec:o2b rf$ $list:lb$ in $ce$ >> ]
    | "apply" NONA
      [ ce = class_expr; sel = LIST1 (expr LEVEL "label") ->
          <:class_expr< $ce$ $list:sel$ >> ]
    | "simple"
      [ "["; ct = ctyp; ","; ctcl = LIST1 ctyp SEP ","; "]";
        ci = class_longident ->
          <:class_expr< $list:ci$ [ $ct$ , $list:ctcl$ ] >>
      | "["; ct = ctyp; "]"; ci = class_longident ->
          <:class_expr< $list:ci$ [ $ct$ ] >>
      | ci = class_longident ->
          <:class_expr< $list:ci$ >>
      | "object"; cspo = OPT class_self_patt; cf = class_structure; "end" ->
          <:class_expr< object $cspo$ $list:cf$ end >>
      | "("; ce = class_expr; ":"; ct = class_type; ")" ->
          <:class_expr< ($ce$ : $ct$) >>
      | "("; ce = class_expr; ")" ->
          ce ] ]
  ;
  class_structure:
    [ [ cf = LIST0 class_str_item -> cf ] ]
  ;
  class_self_patt:
    [ [ "("; p = patt; ")" -> p
      | "("; p = patt; ":"; t = ctyp; ")" -> <:patt< ($p$ : $t$) >> ] ]
  ;
  class_str_item:
    [ [ "inherit"; ce = class_expr; pb = OPT [ "as"; i = LIDENT -> i ] ->
          <:class_str_item< inherit $ce$ $as:pb$ >>
      | "val"; (lab, mf, e)  = cvalue ->
          <:class_str_item< value $mut:mf$ $lab$ = $e$ >>
      | "method"; "private"; "virtual"; l = label; ":"; t = ctyp ->
          <:class_str_item< method private virtual $l$ : $t$ >>
      | "method"; "virtual"; pf = OPT "private"; l = label; ":"; t = ctyp ->
          <:class_str_item< method virtual $priv:o2b pf$ $l$ : $t$ >>
      | "method"; "private"; l = label; fb = fun_binding ->
          <:class_str_item< method private $l$ = $fb$ >>
      | "method"; l = label; fb = fun_binding ->
          <:class_str_item< method $l$ = $fb$ >>
      | "constraint"; t1 = ctyp; "="; t2 = ctyp ->
          <:class_str_item< type $t1$ = $t2$ >>
      | "initializer"; se = expr ->
          <:class_str_item< initializer $se$ >> ] ]
  ;
  cvalue:
    [ [ mf = OPT "mutable"; l = label; "="; e = expr ->
          (l, o2b mf, e)
      | mf = OPT "mutable"; l = label; ":"; t = ctyp; "="; e = expr ->
          (l, o2b mf, <:expr< ($e$ : $t$) >>)
      | mf = OPT "mutable"; l = label; ":"; t1 = ctyp; ":>"; t2 = ctyp; "=";
        e = expr ->
          (l, o2b mf, <:expr< ($e$ : $t1$ :> $t2$) >>)
      | mf = OPT "mutable"; l = label; ":>"; t = ctyp; "="; e = expr ->
          (l, o2b mf, <:expr< ($e$ :> $t$) >>) ] ]
  ;
  label:
    [ [ i = LIDENT -> i ] ]
  ;

  (* Class types *)

  class_type:
    [ [ t = ctyp LEVEL "ctyp1" ->
          class_type_of_ctyp loc t
      | t = ctyp LEVEL "ctyp1"; "->"; ct = class_type ->
          <:class_type< [ $t$ ] -> $ct$ >>
      | t = ctyp LEVEL "ctyp1"; "*"; tl = LIST1 (ctyp LEVEL "simple") SEP "*";
        "->"; ct = class_type ->
          <:class_type< [ ($t$ * $list:tl$) ] -> $ct$ >>
      | cs = class_signature ->
          cs ] ]
  ;
  class_signature:
    [ [ "["; t = ctyp; ","; tl = LIST1 ctyp SEP ","; "]";
        id = clty_longident ->
          <:class_type< [ $t$ , $list:tl$ ] $list:id$ >>
      | "["; t = ctyp; "]"; id = clty_longident ->
          <:class_type< [ $t$ ] $list:id$ >>
      | id = clty_longident ->
          <:class_type< $list:id$ >>
      | "object"; cst = OPT class_self_type; csf = LIST0 class_sig_item;
        "end" ->
          <:class_type< object $cst$ $list:csf$ end >> ] ]
  ;
  class_self_type:
    [ [ "("; t = ctyp; ")" -> t ] ]
  ;
  class_sig_item:
    [ [ "inherit"; cs = class_signature ->
          <:class_sig_item< inherit $cs$ >>
      | "val"; mf = OPT "mutable"; l = label; ":"; t = ctyp ->
          <:class_sig_item< value $mut:o2b mf$ $l$ : $t$ >>
      | "method"; "private"; "virtual"; l = label; ":"; t = ctyp ->
          <:class_sig_item< method private virtual $l$ : $t$ >>
      | "method"; "virtual"; pf = OPT "private"; l = label; ":"; t = ctyp ->
          <:class_sig_item< method virtual $priv:o2b pf$ $l$ : $t$ >>
      | "method"; "private"; l = label; ":"; t = ctyp ->
          <:class_sig_item< method private $l$ : $t$ >>
      | "method"; l = label; ":"; t = ctyp ->
          <:class_sig_item< method $l$ : $t$ >>
      | "constraint"; t1 = ctyp; "="; t2 = ctyp ->
          <:class_sig_item< type $t1$ = $t2$ >> ] ]
  ;
  class_description:
    [ [ vf = OPT "virtual"; ctp = class_type_parameters; n = LIDENT; ":";
        ct = class_type ->
          {MLast.ciLoc = loc; MLast.ciVir = o2b vf; MLast.ciPrm = ctp;
           MLast.ciNam = n; MLast.ciExp = ct} ] ]
  ;
  class_type_declaration:
    [ [ vf = OPT "virtual"; ctp = class_type_parameters; n = LIDENT; "=";
        cs = class_signature ->
          {MLast.ciLoc = loc; MLast.ciVir = o2b vf; MLast.ciPrm = ctp;
           MLast.ciNam = n; MLast.ciExp = cs} ] ]
  ;

  (* Expressions *)

  expr: LEVEL "apply"
    [ LEFTA
      [ "new"; i = class_longident -> <:expr< new $list:i$ >> ] ]
  ;
  expr: LEVEL "simple"
    [ [ e = expr; "#"; lab = label -> <:expr< $e$ # $lab$ >> ] ]
  ;
  expr: LEVEL "simple"
    [ [ "("; e = expr; ":"; t1 = ctyp; ":>"; t2 = ctyp; ")" ->
          <:expr< ($e$ : $t1$ :> $t2$) >>
      | "("; e = expr; ":>"; t = ctyp; ")" ->
          <:expr< ($e$ :> $t$) >>
      | "{<"; ">}" ->
          <:expr< {< >} >>
      | "{<"; fel = field_expr_list; ">}" ->
          <:expr< {< $list:fel$ >} >> ] ]
  ;
  field_expr_list:
    [ [ l = label; "="; e = expr LEVEL "expr1"; ";";
        fel = field_expr_list ->
          [(l, e) :: fel]
      | l = label; "="; e = expr LEVEL "expr1"; ";" -> [(l, e)]
      | l = label; "="; e = expr LEVEL "expr1" -> [(l, e)] ] ]
  ;

  (* Core types *)

  ctyp: LEVEL "simple"
    [ [ "#"; id = class_longident -> <:ctyp< # $list:id$ >>
      | "<"; (ml, v) = meth_list; ">" -> <:ctyp< < $list:ml$ $v$ > >>
      | "<"; ">" -> <:ctyp< < > >> ] ]
  ;
  meth_list:
    [ [ f = field; ";"; (ml, v) = meth_list -> ([f :: ml], v)
      | f = field; ";" -> ([f], False)
      | f = field -> ([f], False)
      | ".." -> ([], True) ] ]
  ;
  field:
    [ [ lab = LIDENT; ":"; t = ctyp -> (lab, t) ] ]
  ;

  (* Identifiers *)

  clty_longident:
    [ [ m = UIDENT; "."; l = clty_longident -> [m :: l]
      | i = LIDENT -> [i] ] ]
  ;
  class_longident:
    [ [ m = UIDENT; "."; l = class_longident -> [m :: l]
      | i = LIDENT -> [i] ] ]
  ;
END;

(* Labels *)

EXTEND
  GLOBAL: ctyp expr patt fun_def fun_binding class_type class_fun_binding;
  ctyp: AFTER "arrow"
    [ NONA
      [ i = LIDENT; ":"; t = ctyp -> <:ctyp< ~ $i$ : $t$ >>
      | i = QUESTIONIDENTCOLON; t = ctyp -> <:ctyp< ? $i$ : $t$ >>  ] ]
  ;
  ctyp: LEVEL "simple"
    [ [ "["; OPT "|"; rfl = LIST0 row_field SEP "|"; "]" ->
          <:ctyp< [| $list:rfl$ |] >>
      | "["; ">"; OPT "|"; rfl = LIST1 row_field SEP "|"; "]" ->
          <:ctyp< [| > $list:rfl$ |] >>
      | "[<"; OPT "|"; (rfl, clos) = row_field_list_dd; "]" ->
          <:ctyp< [| < $list:rfl$ $dd:clos$ |] >> ] ]
  ;
  row_field:
    [ [ "`"; i = ident -> (i, False, [])
      | "`"; i = ident; "of"; ao = OPT "&"; l = LIST1 ctyp SEP "&" ->
          (i, o2b ao, l) ] ]
  ;
  row_field_list_dd:
    [ [ rf = row_field -> ([rf], True)
      | rf = row_field; "|"; ".." -> ([rf], False)
      | rf = row_field; "|"; (rfl, dd) = row_field_list_dd ->
          ([rf :: rfl], dd) ] ]
  ;
  expr: LEVEL "expr1"
    [ [ "fun"; p = labeled_patt; e = fun_def ->
          <:expr< fun $p$ -> $e$ >> ] ]
  ;
  expr: AFTER "apply"
    [ "label"
      [ i = TILDEIDENTCOLON; e = expr -> <:expr< ~ $i$ : $e$ >>
      | i = TILDEIDENT -> <:expr< ~ $i$ >>
      | i = QUESTIONIDENTCOLON; e = expr -> <:expr< ? $i$ : $e$ >>
      | i = QUESTIONIDENT -> <:expr< ? $i$ >> ] ]
  ;
  expr: LEVEL "simple"
    [ [ "`"; s = ident -> <:expr< ` $s$ >> ] ]
  ;
  fun_def:
    [ [ p = labeled_patt; e = fun_def ->
          <:expr< fun $p$ -> $e$ >> ] ]
  ;
  fun_binding:
    [ [ p = labeled_patt; e = fun_binding -> <:expr< fun $p$ -> $e$ >> ] ]
  ;
  patt: LEVEL "simple"
    [ [ "`"; s = ident -> <:patt< ` $s$ >> ] ]
  ;
  labeled_patt:
    [ [ i = TILDEIDENTCOLON; p = patt LEVEL "simple" ->
          <:patt< ~ $i$ : $p$ >>
      | i = TILDEIDENT ->
          <:patt< ~ $i$ >>
      | i = QUESTIONIDENTCOLON; j = LIDENT ->
          <:patt< ? $i$ : $lid:j$ >>
      | i = QUESTIONIDENTCOLON; "("; j = LIDENT; "="; e = expr; ")" ->
          <:patt< ? $i$ : ( $lid:j$ = $e$ ) >>
      | i = QUESTIONIDENT ->
          <:patt< ? $i$ : $lid:i$ >>
      | "?"; "("; i = LIDENT; "="; e = expr; ")" ->
          <:patt< ? ( $i$ = $e$ ) >> ] ]
  ;
  class_type:
    [ [ i = LIDENT; ":"; t = ctyp LEVEL "ctyp1"; "->"; ct = class_type ->
          <:class_type< [ ~ $i$ : $t$ ] -> $ct$ >>
      | i = QUESTIONIDENTCOLON; t = ctyp LEVEL "ctyp1"; "->";
        ct = class_type ->
          <:class_type< [ ? $i$ : $t$ ] -> $ct$ >> ] ]
  ;
  class_fun_binding:
    [ [ p = labeled_patt; cfb = class_fun_binding ->
          <:class_expr< fun $p$ -> $cfb$ >> ] ]
  ;
  ident:
    [ [ i = LIDENT -> i
      | i = UIDENT -> i ] ]
  ;
END;
