/*
 * calmwm - the calm window manager
 *
 * Copyright (c) 2004 Marius Aamodt Eriksen <marius@monkey.org>
 * All rights reserved.
 *
 * $Id: cycle.c,v 1.11 2004/07/13 06:50:00 marius Exp $
 */

#include "headers.h"
#include "calmwm.h"

struct cycle *
cycle_new(void)
{
	struct cycle *cy;

	XCALLOC(cy);
	TAILQ_INIT(&cy->cycleq);
	TAILQ_INIT(&cy->disabledq);

	return (cy);
}

void
cycle_free(struct cycle *cy)
{
	struct cycle_entry *cye;

	while ((cye = TAILQ_FIRST(&cy->cycleq)) != NULL) {
		TAILQ_REMOVE(&cy->cycleq, cye, entry);
		xfree(cye);
	}

	xfree(cy);
}

void
cycle_add(struct cycle *cy, void *ctx)
{
	struct cycle_entry *cye;

	XMALLOC(cye);
	cye->ctx = ctx;
	TAILQ_INSERT_TAIL(&cy->cycleq, cye, entry);

	if (cy->current == NULL)
		cy->current = cye;
}

/*
 * XXX this is ugly.  We should either hash ctx -> entries, or make
 * the user of the API keep entries.
 *
 * XXX; we require that the ctx is disabled.
 */
void
cycle_remove(struct cycle *cy, void *ctx)
{
	struct cycle_entry *cye, *nextcye;

	for (cye = TAILQ_FIRST(&cy->disabledq);
	     cye != NULL; cye = nextcye) {
		nextcye = TAILQ_NEXT(cye, entry);

		if (cye->ctx == ctx) {
			TAILQ_REMOVE(&cy->disabledq, cye, entry);

			if (cy->begin == cye)
				cy->begin = NULL;
			if (cy->current == cye)
				cy->current = NULL;

			xfree(cye);
		}
	}
}

void
cycle_disable(struct cycle *cy, void *ctx)
{
	struct cycle_entry *cye, *nextcye, *prevcye;

	for (prevcye = cye = TAILQ_FIRST(&cy->cycleq);
	     cye != NULL; cye = nextcye) {
		nextcye = cye;
		do { 
			nextcye = TAILQ_NEXT(nextcye, entry);
		} while (nextcye == prevcye);

		if (nextcye != NULL && nextcye->nhits > cye->nhits) {
			TAILQ_REMOVE(&cy->cycleq, nextcye, entry);
			TAILQ_INSERT_BEFORE(cye, nextcye, entry);
		}

		if (cye->ctx == ctx) {
			TAILQ_REMOVE(&cy->cycleq, cye, entry);
			TAILQ_INSERT_TAIL(&cy->disabledq, cye, entry);
		}

		if (cy->current == cye)
			cy->current = NULL;
		if (cy->begin == cye)
			cy->begin = NULL;
	}

}

void
cycle_enable(struct cycle *cy, void *ctx)
{
	struct cycle_entry *cye, *cye1, *nextcye;

	for (cye = TAILQ_FIRST(&cy->disabledq); cye != NULL; cye = nextcye) {
		nextcye = TAILQ_NEXT(cye, entry);

		if (cye->ctx == ctx) {
			TAILQ_REMOVE(&cy->disabledq, cye, entry);
			break;
		}
	}

	if (cye != NULL) {
		TAILQ_FOREACH(cye1, &cy->cycleq, entry) {
			if (cye1->nhits < cye->nhits) {
				TAILQ_INSERT_BEFORE(cye1, cye, entry);
				break;
			}
		}
		if (cye1 == NULL)
			TAILQ_INSERT_TAIL(&cy->cycleq, cye, entry);
	}
}

void *
cycle_next(struct cycle *cy, int backward, int begin)
{
	struct cycle_entry *cye;

	if (begin) {
		cye = cy->begin;
		cy->begin = cy->current;
		cy->current = cye;
	} else {
		if (cy->current != NULL) {
			cye = cy->current;
			cy->current = backward ?
			    TAILQ_PREV(cye, cycle_entry_q, entry) :
			    TAILQ_NEXT(cye, entry);
		}
	}

	if (cy->current == NULL)
		cy->current = TAILQ_FIRST(&cy->cycleq);

	/* XXX */
	if (cy->begin == NULL && cy->current != NULL)
		cy->begin = TAILQ_NEXT(cy->current, entry);

	if (cy->current != NULL)
		cy->current->nhits++;

	return (cy->current != NULL ? cy->current->ctx : NULL);
}

void
cycle_current(struct cycle *cy, void *ctx)
{
	struct cycle_entry *cye;

	if (cy->current != NULL && cy->current->ctx == ctx)
		return;

	TAILQ_FOREACH(cye, &cy->cycleq, entry)
		if (cye->ctx == ctx)
			break;

	if (cye == NULL)
		return;

	cye->nhits++;
	cy->begin = cy->current;
	cy->current = cye;
}
