/*
 * Calcoo: body_display.c
 *
 * Copyright (C) 2001 Alexei Kaminski
 *
 * display creation and
 * functions that facilitate display functionality
 *
 */

#include <string.h>
#include <stdlib.h>
#include <gtk/gtk.h>
#include <gdk/gdk.h>

#include "codes.h"
#include "body.h"
#include "body_headers.h"
#include "aux_headers.h"
#include "gtkaux_headers.h"

void malloc_displays(void)
{
	int i;

	for (i = 0; i < NUMBER_OF_DISPLAYS; i++){
		body->display[i] = 
			(t_body_display*) malloc (sizeof(t_body_display));
		if(body->display[i] == NULL)
			error_occured("malloc failed", TRUE);
	}
	for (i = 0; i < NUMBER_OF_REG_DISPLAYS; i++) {
		body->display[MEM_D_N_SHIFT + i] = 
			(t_body_display*) malloc (sizeof(t_body_display));
		if(body->display[MEM_D_N_SHIFT + i] == NULL)
			error_occured("malloc failed", TRUE);
	}
	for (i = 0; i < NUMBER_OF_SIMPLE_DISPLAYS; i++){
		body->simple_display[i] = 
			(t_body_simple_display*) 
			malloc (sizeof(t_body_simple_display));
		if(body->display[i] == NULL)
			error_occured("malloc failed", TRUE);
	}
}

/*-------- Displays for numbers ----------------*/

void create_display(int display_number,
		    int x, int y, int w, int h,
		    int cell_width,
		    int dot_offset_vert,
		    int dot_width,
		    int margin_vert,
		    int margin_horiz,
		    int error_indent,
		    int tick_offset_vert,
		    int tick_offset_horiz,
		    int show_frame, 
		    char ***digit_xpm) 
{
	int i, n, x1, y1;
	int display_exp_indent;
	t_body_display *display;

        /* for the sake of brevity: */
	display = body->display[display_number];

	/* Frame */
	if (show_frame) {
		display->frame = gtk_frame_new(NULL);
		gtk_frame_set_shadow_type(GTK_FRAME(display->frame), 
					  GTK_SHADOW_IN);
		gtk_fixed_put(GTK_FIXED(body->fixer), display->frame, x, y);
		gtk_widget_set_usize(display->frame, w, h);
		gtk_widget_show(display->frame);
	}

	/* Offset for the digits */
	x1 = x + margin_horiz;
	y1 = y + margin_vert;

	/* Things to show */

	for (i = 0; i < INPUT_LENGTH; i++) 
		for (n = 0; n < BASE; n++) 
			create_n_put_pixmap(&display->int_digits[i][n],
					    x1 + cell_width * (i + 1), y1,
					    digit_xpm[n]);

	for (i = 0; i < INPUT_LENGTH; i++) 
		for (n = 0; n < BASE; n++) 
			create_n_put_pixmap(&display->frac_digits[i][n],
					    x1 + dot_width 
					       + cell_width * (i + 1), y1,
					    digit_xpm[n]);

	display_exp_indent = cell_width * (INPUT_LENGTH + 1) + dot_width;

	for (i = 0; i < EXP_INPUT_LENGTH; i++) 
		for (n = 0; n < BASE; n++) 
			create_n_put_pixmap(&display->exp_digits[i][n],
					    x1 + display_exp_indent
			  		       + cell_width * (i + 2), y1,
					    digit_xpm[n]);

	for (i = 0; i < INPUT_LENGTH; i++) 
		create_n_put_pixmap(&display->dot[i],
				    x1 + cell_width * (i + 2), 
				    y1 + dot_offset_vert,
				    digit_xpm[D_DOT]);

	for (i = 0; i < INPUT_LENGTH; i++) 
		create_n_put_pixmap(&display->minus_sign[i],
				    x1 + cell_width * i, y1,
				    digit_xpm[D_MINUS]);

	for (i = 0; i < INPUT_LENGTH; i++) 
		create_n_put_pixmap(&display->int_ticks[i],
				    x1 + cell_width * i + tick_offset_horiz,
				    y1 + tick_offset_vert,
				    digit_xpm[D_TICK]);

	for (i = 0; i < INPUT_LENGTH; i++) 
		create_n_put_pixmap(&display->frac_ticks[i],
				    x1 + cell_width * i + dot_width 
				    + tick_offset_horiz,
				    y1 + tick_offset_vert,
				    digit_xpm[D_TICK]);


	create_n_put_pixmap(&display->exp_minus_sign, 
			    x1 + display_exp_indent + cell_width, y1,
			    digit_xpm[D_MINUS]);
		       
	create_n_put_pixmap(&display->exp_plus_sign, 
			    x1 + display_exp_indent + cell_width, y1,
			    digit_xpm[D_PLUS]);

	create_n_put_pixmap(&display->E, 
			    x1 + display_exp_indent, y1,
			    digit_xpm[D_E]);

	create_n_put_pixmap(&display->error_message, 
			    x1 + error_indent, y1,
			    digit_xpm[D_OVERFLOW]);

	clear_display(display_number);
}

void clear_display(int display_number)
{
	int i,n;
	t_body_display *display;

        /* for the sake of brevity: */
	display = body->display[display_number];

	for (i = 0; i < INPUT_LENGTH; i++) 
		for (n = 0; n < BASE; n++) 
			gtk_widget_hide(display->int_digits[i][n]);

	for (i = 0; i < INPUT_LENGTH; i++) 
		for (n = 0; n < BASE; n++) 
			gtk_widget_hide(display->frac_digits[i][n]);

	for (i = 0; i < EXP_INPUT_LENGTH; i++) 
		for(n = 0; n < BASE; n++) 
			gtk_widget_hide(display->exp_digits[i][n]);


	for(i = 0; i < INPUT_LENGTH; i++) 
		gtk_widget_hide(display->dot[i]);

	for(i = 0; i < INPUT_LENGTH; i++) 
		gtk_widget_hide(display->minus_sign[i]);

	for(i = 0; i < INPUT_LENGTH; i++) 
		gtk_widget_hide(display->int_ticks[i]);

	for(i = 0; i < INPUT_LENGTH; i++) 
		gtk_widget_hide(display->frac_ticks[i]);

	gtk_widget_hide(display->exp_minus_sign);
	gtk_widget_hide(display->exp_plus_sign);
	gtk_widget_hide(display->E);
	gtk_widget_hide(display->error_message);
}

void show_int_digit(int d, int i, int n)
{
	gtk_widget_show(body->display[d]->int_digits[i][code_to_digit(n)]);
}

void show_frac_digit(int d, int i, int n)
{
	gtk_widget_show(body->display[d]->frac_digits[i][code_to_digit(n)]);
}

void show_exp_digit(int d, int i, int n)
{
	gtk_widget_show(body->display[d]->exp_digits[i][code_to_digit(n)]);
}

void show_minus(int d, int i)
{
	gtk_widget_show(body->display[d]->minus_sign[i]);
}

void show_int_tick(int d, int i)
{
	gtk_widget_show(body->display[d]->int_ticks[i]);
}

void show_frac_tick(int d, int i)
{
	gtk_widget_show(body->display[d]->frac_ticks[i]);
}

void show_E_and_exp_sign(int d, int exp_sign)
{
	gtk_widget_show(body->display[d]->E);
	switch (exp_sign) {
	case SIGN_PLUS:  
		gtk_widget_show(body->display[d]->exp_plus_sign);
		break;
	case SIGN_MINUS: 
		gtk_widget_show(body->display[d]->exp_minus_sign);
		break;
	}
}

void show_dot(int d, int i)
{
	gtk_widget_show(body->display[d]->dot[i]);
}

void show_error_message(int d)
{
	gtk_widget_show(body->display[d]->error_message);
	strcpy(body->char_display,"error");
}

/*----------- Simple displays -----------------------*/

void create_simple_display(int display_number,
			   int x, int y, int w, int h,
			   int number_of_glyphs,
			   int show_frame, 
			   char ***glyph_xpm)
{
	int i;
	t_body_simple_display *display;

        /* for the sake of brevity: */
	display = body->simple_display[display_number];

	/* Frame */
	if (show_frame) {
		display->frame = gtk_frame_new(NULL);
		gtk_frame_set_shadow_type(GTK_FRAME(display->frame), 
					  GTK_SHADOW_IN);
		gtk_fixed_put(GTK_FIXED(body->fixer), display->frame, x, y);
		gtk_widget_set_usize(display->frame, w, h);
		gtk_widget_show(display->frame);
	}


  	for (i = 0; i < number_of_glyphs; i++)  
  		create_n_put_pixmap(&display->glyph[i], x, y, glyph_xpm[i]); 
  	for (i = number_of_glyphs; i < SD_G_MAX; i++)  
		display->glyph[i] = NULL;
		       
  	clear_simple_display(display_number); 
}

void clear_simple_display(int display_number)
{
	int i;
	t_body_simple_display *display;

        /* for the sake of brevity: */
	display = body->simple_display[display_number];

	for (i = 0; i < OD_G_TOTAL; i++) 
		if (display->glyph[i] != NULL)
			gtk_widget_hide(display->glyph[i]);
}

void show_display_glyph(int display_number, int glyph_code)
{
	t_body_simple_display *display;

        /* for the sake of brevity: */
	display = body->simple_display[display_number];

	if (display->glyph[glyph_code] != NULL)
		gtk_widget_show(display->glyph[glyph_code]);
}

/*------------ Character string -------------*/
/* not quite a display, but a kind of similar */

void set_char_display(char *s)
{
	strcpy(body->char_display,s);
}

