/* 
   'ls' for cadaver
   Copyright (C) 2000-2001, Joe Orton <joe@manyfish.co.uk>, 
   except where otherwise indicated.
                                                                     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "config.h"

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include <time.h>

#include <ne_request.h>
#include <ne_props.h>
#include <ne_uri.h>
#include <ne_alloc.h>
#include <ne_dates.h>

#include "i18n.h"
#include "commands.h"
#include "cadaver.h"
#include "basename.h"

static time_t current_time;

struct fetch_context {
    struct resource **list;
    const char *target; /* Request-URI of the PROPFIND */
    unsigned int include_target; /* Include resource at href */
};    

static const ne_propname ls_props[] = {
    { "DAV:", "getcontentlength" },
    { "DAV:", "getlastmodified" },
    { "DAV:", "displayname" },
    { "http://apache.org/dav/props/", "executable" },
    { "DAV:", "resourcetype" },
    { NULL }
};

#define ELM_resourcetype (NE_ELM_207_UNUSED + 1)
#define ELM_collection (NE_ELM_207_UNUSED + 4)

static const struct ne_xml_elm complex_elms[] = {
    { "DAV:", "resourcetype", ELM_resourcetype, 0 },
    { "DAV:", "collection", ELM_collection, 0 },
    { NULL }
};

static char *format_time(time_t when)
{
    const char *fmt;
    static char ret[256];
    struct tm *local;

    if (when == (time_t)-1) {
	/* Happens on lock-null resources */
	return "  (unknown) ";
    }

    /* from GNU fileutils... this section is 
     *  Copyright (C) 85, 88, 90, 91, 1995-1999 Free Software Foundation, Inc.
     */
    if (current_time > when + 6L * 30L * 24L * 60L * 60L	/* Old. */
	|| current_time < when - 60L * 60L) {
	/* The file is fairly old or in the future.
	   POSIX says the cutoff is 6 months old;
	   approximate this by 6*30 days.
	   Allow a 1 hour slop factor for what is considered "the future",
	   to allow for NFS server/client clock disagreement.
	   Show the year instead of the time of day.  */
	fmt = "%b %e  %Y";
    } else {
	fmt = "%b %e %H:%M";
    }
    /* end FSF copyrighted section. */
    local = localtime(&when);
    if (local != NULL) {
	if (strftime(ret, 256, fmt, local)) {
	    return ret;
	}
    }
    return "???";
}

static int compare_resource(const struct resource *r1, 
			    const struct resource *r2)
{
    /* Sort errors first, then collections, then alphabetically */
    if (r1->type == resr_error) {
	return -1;
    } else if (r2->type == resr_error) {
	return 1;
    } else if (r1->type == resr_collection) {
	if (r2->type != resr_collection) {
	    return -1;
	} else {
	    return strcmp(r1->uri, r2->uri);
	}
    } else {
	if (r2->type != resr_collection) {
	    return strcmp(r1->uri, r2->uri);
	} else {
	    return 1;
	}
    }
}

static void display_ls_line(struct resource *res)
{
    const char *restype;
    char exec_char, *name;

    switch (res->type) {
    case resr_normal: restype = ""; break;
    case resr_reference: restype = _("Ref:"); break;
    case resr_collection: restype = _("Coll:"); break;
    default:
	restype = "???"; break;
    }
    
    if (ne_path_has_trailing_slash(res->uri)) {
	res->uri[strlen(res->uri)-1] = '\0';
    }
    name = strrchr(res->uri, '/');
    if (name != NULL && strlen(name+1) > 0) {
	name++;
    } else {
	name = res->uri;
    }

    name = ne_path_unescape(name);

    if (res->type == resr_error) {
	printf(_("Error: %-30s %d %s\n"), name, res->error_status,
	       res->error_reason?res->error_reason:_("unknown"));
    } else {
	exec_char = res->is_executable ? '*' : ' ';
	printf("%5s %c%-30s %10d  %s\n", restype, exec_char, name,
	       res->size, format_time(res->modtime));
    }

    free(name);
}

void execute_ls(const char *remote)
{
    int ret;
    char *real_remote;
    struct resource *reslist = NULL, *current, *next;
    if (remote != NULL) {
	real_remote = resolve_path(path, remote, true);
    } else {
	real_remote = ne_strdup(path);
    }
    out_start(_("Listing collection"), real_remote);
    ret = fetch_resource_list(session, real_remote, 1, 0, &reslist);
    if (ret == NE_OK) {
	/* Easy this, eh? */
	if (reslist == NULL) {
	    output(o_finish, _("collection is empty.\n"));
	} else {
	    out_success();
	    current_time = time(NULL);
	    for (current = reslist; current!=NULL; current = next) {
		next = current->next;
		if (strlen(current->uri) > strlen(real_remote)) {
		    display_ls_line(current);
		}
		free_resource(current);
	    }
	}
    } else {
	out_result(ret);
    }
    free(real_remote);
}

static void results(void *userdata, const char *uri,
		    const ne_prop_result_set *set)
{
    struct fetch_context *ctx = userdata;
    struct resource *current, *previous, *newres;
    const char *clength, *modtime, *isexec;
    const ne_status *status = NULL;
    ne_uri u;

    NE_DEBUG(NE_DBG_HTTP, "Uri: %s\n", uri);

    newres = ne_propset_private(set);
    
    if (ne_uri_parse(uri, &u))
	return;
    
    if (u.path == NULL) {
	ne_uri_free(&u);
	return;
    }

    NE_DEBUG(NE_DBG_HTTP, "URI path %s in %s\n", u.path, ctx->target);
    
    if (ne_path_compare(ctx->target, u.path) == 0 && !ctx->include_target) {
	/* This is the target URI */
	NE_DEBUG(NE_DBG_HTTP, "Skipping target resource.\n");
	/* Free the private structure. */
	free(newres);
	ne_uri_free(&u);
	return;
    }

    newres->uri = ne_strdup(u.path);

    clength = ne_propset_value(set, &ls_props[0]);    
    modtime = ne_propset_value(set, &ls_props[1]);
    isexec = ne_propset_value(set, &ls_props[3]);
    
    if (clength == NULL)
	status = ne_propset_status(set, &ls_props[0]);
    if (modtime == NULL)
	status = ne_propset_status(set, &ls_props[1]);

    if (newres->type == resr_normal && status) {
	/* It's an error! */
	newres->error_status = status->code;

	/* Special hack for Apache 1.3/mod_dav */
	if (strcmp(status->reason_phrase, "status text goes here") == 0) {
	    const char *desc;
	    if (status->code == 401) {
		desc = _("Authorization Required");
	    } else if (status->klass == 3) {
		desc = _("Redirect");
	    } else if (status->klass == 5) {
		desc = _("Server Error");
	    } else {
		desc = _("Unknown Error");
	    }
	    newres->error_reason = ne_strdup(desc);
	} else {
	    newres->error_reason = ne_strdup(status->reason_phrase);
	}
	newres->type = resr_error;
    }

    if (isexec && strcasecmp(isexec, "T") == 0) {
	newres->is_executable = 1;
    } else {
	newres->is_executable = 0;
    }

    if (modtime)
	newres->modtime = ne_httpdate_parse(modtime);

    if (clength)
	newres->size = atoi(clength);

    NE_DEBUG(NE_DBG_HTTP, "End resource %s\n", newres->uri);

    for (current = *ctx->list, previous = NULL; current != NULL; 
	 previous = current, current=current->next) {
	if (compare_resource(current, newres) >= 0) {
	    break;
	}
    }
    if (previous) {
	previous->next = newres;
    } else {
	*ctx->list = newres;
    }
    newres->next = current;

    ne_uri_free(&u);
}

static int end_element(void *userdata, const struct ne_xml_elm *elm, const char *cdata)
{
    ne_propfind_handler *pfh = userdata;
    struct resource *r = ne_propfind_current_private(pfh);

    if (r == NULL) {
	return 0;
    }

    if (elm->id == ELM_collection) {
	NE_DEBUG(NE_DBG_HTTP, "This is a collection.\n");
	r->type = resr_collection;
    }

    return 0;
}

static int check_context(void *ud, ne_xml_elmid parent, ne_xml_elmid child)
{
    if ((parent == NE_ELM_prop && child == ELM_resourcetype) ||
	(parent == ELM_resourcetype && child == ELM_collection)) {
	return NE_XML_VALID;
    }
    return NE_XML_DECLINE;
}

void free_resource(struct resource *res)
{
    NE_FREE(res->uri);
    NE_FREE(res->displayname);
    NE_FREE(res->error_reason);
    free(res);
}

void free_resource_list(struct resource *res)
{
    struct resource *next;
    for (; res != NULL; res = next) {
	next = res->next;
	free_resource(res);
    }
}

static void *create_private(void *userdata, const char *uri)
{
    return ne_calloc(sizeof(struct resource));
}

int fetch_resource_list(ne_session *sess, const char *uri,
			 int depth, int include_target,
			 struct resource **reslist)
{
    ne_propfind_handler *pfh = ne_propfind_create(sess, uri, depth);
    int ret;
    struct fetch_context ctx = {0};
    
    *reslist = NULL;
    ctx.list = reslist;
    ctx.target = uri;
    ctx.include_target = include_target;

    ne_xml_push_handler(ne_propfind_get_parser(pfh), complex_elms, 
			 check_context, NULL, end_element, pfh);

    ne_propfind_set_private(pfh, create_private, NULL);

    ret = ne_propfind_named(pfh, ls_props, results, &ctx);

    ne_propfind_destroy(pfh);

    return ret;
}
