/*
 * Calculate and display Pascal's Triangle.
 *
 * Copyright (C) 2005 George Gesslein II.
 *
 * Usage: matho-pascal [number-of-lines]
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <errno.h>
#include <math.h>
#include <assert.h>
#include <sys/ioctl.h>
#include <termios.h>

#define	true	1
#define	false	0

#define	MAX_LINES	1000	/* max number of lines of Pascal's triangle allowed */

int center_buf(int i);

int		lines = 26;
int		cell_size = 6;
double		*array[MAX_LINES];
int		screen_columns = 210;
int		centered = true;
char		line_buf[1000];

char		*prog_name;

int
main(int argc, char *argv[])
{
	int		i, j;
	int		len;
	struct winsize	ws;

	prog_name = argv[0];
	ws.ws_col = 0;
	ws.ws_row = 0;
	ioctl(1, TIOCGWINSZ, &ws);
	if (ws.ws_col) {
		screen_columns = ws.ws_col;
	}
	if (screen_columns >= sizeof(line_buf)) {
		fprintf(stderr, "Screen too wide!\n");
		exit(1);
	}
	switch (argc) {
	case 0:
	case 1:
		break;
	case 2:
		centered = false;
		if (isdigit(argv[1][0])) {
			lines = atoi(argv[1]);
			break;
		}
	default:
		fprintf(stderr, "Usage: %s [number-of-lines]\n", prog_name);
		exit(1);
	}
	if (lines <= 0 || lines > MAX_LINES) {
		fprintf(stderr, "Number of lines out of range (1..%d).\n", MAX_LINES);
		exit(1);
	}
	for (i = 0; i < lines; i++) {
		array[i] = (double *) malloc(sizeof(double) * lines);
		if (array[i] == NULL) {
			fprintf(stderr, "malloc() error!\n");
			exit(1);
		}
	}
	for (i = 0; i < lines; i++) {
		for (j = 0; j <= i; j++) {
			if (j == 0 || j == i) {
				array[i][j] = 1;
			} else {
				array[i][j] = array[i-1][j-1] + array[i-1][j];
			}
		}
	}
	if (centered && lines > 20) {
		len = center_buf(19);
		if (len > 0 && len < screen_columns) {
			cell_size = 8;	/* for very wide screens */
		}
	}
	for (i = 0; i < lines; i++) {
		if (centered) {
			len = center_buf(i);
			if (len <= 0) {
				exit(0);
			}
			if (len >= screen_columns) {
				exit(0);	/* stop here because of wrap-around */
			}
			/* center on screen */
			for (j = (screen_columns - len) / 2; j > 0; j--) {
				printf(" ");
			}
			printf("%s", line_buf);
		} else {
			for (j = 0; j <= i; j++) {
				printf("%.14g ", array[i][j]);
			}
		}
		printf("\n");
	}
	exit(0);
}

/*
 * Create a line of output in line_buf[] for centering mode.
 * Return length if successful,
 * otherwise return 0.
 */
int
center_buf(int i)
{
	int	j, k;
	int	i1;
	int	len;
	char	buf2[20];

	assert(cell_size < sizeof(buf2));
	line_buf[0] = '\0';
	for (j = 0; j <= i; j++) {
		assert(strlen(line_buf) + cell_size < sizeof(line_buf));
		len = snprintf(buf2, sizeof(buf2), "%.14g", array[i][j]);
		if (len >= cell_size) {
			return(0);	/* cell_size too small */
		}
		/* center in the cell */
		for (k = i1 = (cell_size - len) / 2; k > 0; k--) {
			strcat(line_buf, " ");
		}
		strcat(line_buf, buf2);
		for (k = len + i1; k < cell_size; k++) {
			strcat(line_buf, " ");
		}
	}
	return(strlen(line_buf));
}
