/*
 * Algebraic manipulator differentiation routines.
 *
 * Copyright (c) 1996 George Gesslein II.
 */

#include "includes.h"

static int	d_recurse();

/*
 * Compute the derivative of an equation side, with respect to the variable "v",
 * using the fast transform method.
 * The result must be simplified by the caller.
 *
 * Return true if successful.
 */
int
differentiate(equation, np, v)
token_type	*equation;	/* source and destination expression */
int		*np;		/* pointer to length of "equation" */
long		v;
{
	int	i;

	organize(equation, np);
/* First put every times and divide on a level by itself. */
	for (i = 1; i < *np; i += 2) {
		switch (equation[i].token.operatr) {
		case TIMES:
		case DIVIDE:
			binary_parenthesize(equation, *np, i);
		}
	}
	return d_recurse(equation, np, 0, 1, v);
}

/*
 * Recursive differentiation routine.
 *
 * Symbolically differentiate expression in "equation"
 * (which is a standard equation side) starting at "loc".
 * The current level of parentheses is "level" and
 * do the differentation with respect to variable "v".
 *
 * Return "true" if successful, "false" if it is beyond this program's
 * capabilities or an error was encountered.
 */
static int
d_recurse(equation, np, loc, level, v)
token_type	*equation;
int		*np, loc, level;
long		v;
{
	int	i, j;
	int	n;
	int	op;
	int	oploc;
	int	endloc;
	complexs	c;

	if (equation[loc].level < level) {
/* First differentiate if it is a single variable or constant. */
/* If it is the specified variable, change it to the constant "1.0", */
/* otherwise change it to "0.0". */
		if (equation[loc].kind == VARIABLE
		    && ((v == MATCH_ANY && (equation[loc].token.variable & VAR_MASK) > SIGN)
		    || equation[loc].token.variable == v)) {
			equation[loc].kind = CONSTANT;
			equation[loc].token.constant = 1.0;
		} else {
			equation[loc].kind = CONSTANT;
			equation[loc].token.constant = 0.0;
		}
		return true;
	}
	op = 0;
	for (endloc = loc + 1;; endloc += 2) {
		if (endloc >= *np || equation[endloc].level < level)
			break;
		if (equation[endloc].level == level) {
			switch (op) {
			case 0:
			case PLUS:
			case MINUS:
				break;
			default:
/* Oops.  More than one operator on the same level in this expression. */
				printf(_("Error in d_recurse().\n"));
				return false;
			}
			op = equation[endloc].token.operatr;
			oploc = endloc;
		}
	}
	switch (op) {
	case 0:
	case PLUS:
	case MINUS:
		break;
	case TIMES:
		goto d_times;
	case DIVIDE:
		goto d_divide;
	case POWER:
		goto d_power;
	default:
/* Differentiate an unsupported operator. */
/* This is possible if the expression doesn't contain the specified variable. */
/* In that case, the expression is replaced with "0", otherwise return false. */
		for (i = loc; i < endloc; i += 2) {
			if (equation[i].kind == VARIABLE
			    && ((v == MATCH_ANY && (equation[i].token.variable & VAR_MASK) > SIGN)
			    || equation[i].token.variable == v)) {
				return false;
			}
		}
		blt(&equation[loc+1], &equation[endloc], (*np - endloc) * sizeof(token_type));
		*np -= (endloc - (loc + 1));
		equation[loc].level = level;
		equation[loc].kind = CONSTANT;
		equation[loc].token.constant = 0.0;
		return true;
	}
/* Differentiate PLUS and MINUS operators. */
/* Use rule: d(u+v) = d(u) + d(v), */
/* where "d()" is the derivative function, */
/* "u" and "v" are expressions. */
	for (i = loc;;) {
		if (i >= *np || equation[i].level < level)
			break;
		if (equation[i].kind != OPERATOR) {
			if (!d_recurse(equation, np, i, level + 1, v))
				return false;
			i++;
			for (; i < *np && equation[i].level > level; i += 2)
				;
			continue;
		}
		i++;
	}
	return true;
d_times:
/* Differentiate TIMES operator. */
/* Use rule: d(u*v) = u*d(v) + v*d(u). */
	if (*np + 1 + (endloc - loc) > n_tokens) {
		error_huge();
	}
	for (i = loc; i < endloc; i++)
		equation[i].level++;
	blt(&equation[endloc+1], &equation[loc], (*np - loc) * sizeof(token_type));
	*np += 1 + (endloc - loc);
	equation[endloc].level = level;
	equation[endloc].kind = OPERATOR;
	equation[endloc].token.operatr = PLUS;
	if (!d_recurse(equation, np, endloc + (oploc - loc) + 2, level + 2, v))
		return false;
	if (!d_recurse(equation, np, loc, level + 2, v))
		return false;
	return true;
d_divide:
/* Differentiate DIVIDE operator. */
/* Use rule: d(u/v) = (v*d(u) - u*d(v))/v^2. */
	if (*np + 3 + (endloc - loc) + (endloc - oploc) > n_tokens) {
		error_huge();
	}
	for (i = loc; i < endloc; i++)
		equation[i].level += 2;
	equation[oploc].token.operatr = TIMES;
	j = 1 + (endloc - loc);
	blt(&equation[endloc+1], &equation[loc], (*np - loc) * sizeof(token_type));
	*np += j;
	equation[endloc].level = level + 1;
	equation[endloc].kind = OPERATOR;
	equation[endloc].token.operatr = MINUS;
	j += endloc;
	blt(&equation[j+2+(endloc-oploc)], &equation[j], (*np - j) * sizeof(token_type));
	*np += 2 + (endloc - oploc);
	equation[j].level = level;
	equation[j].kind = OPERATOR;
	equation[j].token.operatr = DIVIDE;
	blt(&equation[j+1], &equation[oploc+1], (endloc - (oploc + 1)) * sizeof(token_type));
	j += endloc - oploc;
	equation[j].level = level + 1;
	equation[j].kind = OPERATOR;
	equation[j].token.operatr = POWER;
	j++;
	equation[j].level = level + 1;
	equation[j].kind = CONSTANT;
	equation[j].token.constant = 2.0;
	if (!d_recurse(equation, np, endloc + (oploc - loc) + 2, level + 3, v))
		return false;
	if (!d_recurse(equation, np, loc, level + 3, v))
		return false;
	return true;
d_power:
/* Differentiate POWER operator. */
/* Since we don't have logarithms, just do what we can without them. */
	for (i = oploc; i < endloc; i++) {
		if (equation[i].kind == VARIABLE
		    && ((v == MATCH_ANY && (equation[i].token.variable & VAR_MASK) > SIGN)
		    || equation[i].token.variable == v)) {
			if (parse_complex_exp(&equation[loc], oploc - loc, &c)) {
				c = complex_log(c);
				n = (endloc - oploc) + 6;
				if (*np + n > n_tokens) {
					error_huge();
				}
				blt(&equation[endloc+n], &equation[endloc], (*np - endloc) * sizeof(token_type));
				*np += n;
				n = endloc;
				equation[n].level = level;
				equation[n].kind = OPERATOR;
				equation[n].token.operatr = TIMES;
				n++;
				equation[n].level = level + 1;
				equation[n].kind = CONSTANT;
				equation[n].token.constant = c.re;
				n++;
				equation[n].level = level + 1;
				equation[n].kind = OPERATOR;
				equation[n].token.operatr = PLUS;
				n++;
				equation[n].level = level + 2;
				equation[n].kind = CONSTANT;
				equation[n].token.constant = c.im;
				n++;
				equation[n].level = level + 2;
				equation[n].kind = OPERATOR;
				equation[n].token.operatr = TIMES;
				n++;
				equation[n].level = level + 2;
				equation[n].kind = VARIABLE;
				equation[n].token.variable = IMAGINARY;
				n++;
				equation[n].level = level;
				equation[n].kind = OPERATOR;
				equation[n].token.operatr = TIMES;
				n++;
				blt(&equation[n], &equation[oploc+1], (endloc - (oploc + 1)) * sizeof(token_type));
				for (i = loc; i < endloc; i++) {
					equation[i].level++;
				}
				if (!d_recurse(equation, np, n, level + 1, v))
					return false;
				return true;
			}
			return false;
		}
	}
	blt(scratch, &equation[oploc+1], (endloc - (oploc + 1)) * sizeof(token_type));
	n = endloc - (oploc + 1);
	scratch[n].level = level;
	scratch[n].kind = OPERATOR;
	scratch[n].token.operatr = TIMES;
	n++;
	if (n + (endloc - loc) + 2 > n_tokens) {
		error_huge();
	}
	blt(&scratch[n], &equation[loc], (endloc - loc) * sizeof(token_type));
	i = n;
	n += oploc + 1 - loc;
	for (; i < n; i++)
		scratch[i].level++;
	n += endloc - (oploc + 1);
	for (; i < n; i++)
		scratch[i].level += 2;
	scratch[n].level = level + 2;
	scratch[n].kind = OPERATOR;
	scratch[n].token.operatr = MINUS;
	n++;
	scratch[n].level = level + 2;
	scratch[n].kind = CONSTANT;
	scratch[n].token.constant = 1.0;
	n++;
	if (n + (oploc - loc) + 1 > n_tokens) {
		error_huge();
	}
	scratch[n].level = level;
	scratch[n].kind = OPERATOR;
	scratch[n].token.operatr = TIMES;
	n++;
	j = n;
	blt(&scratch[n], &equation[loc], (oploc - loc) * sizeof(token_type));
	n += oploc - loc;
	if (*np - (endloc - loc) + n > n_tokens) {
		error_huge();
	}
	blt(&equation[loc+n], &equation[endloc], (*np - endloc) * sizeof(token_type));
	*np += loc + n - endloc;
	blt(&equation[loc], scratch, n * sizeof(token_type));
	if (!d_recurse(equation, np, loc + j, level + 1, v))
		return false;
	return true;
}

/*
 * The derivative command.
 */
int
derivative(cp)
char	*cp;
{
	long		v;
	int		i, j;
	long		l;
	int		er;
	int		order;
	token_type	*source, *dest;
	int		*nps, *np;

	v = 0;
	order = 1;
	if (notdefined(cur_equation)) {
		return false;
	}
	i = next_espace();
	if (n_rhs[cur_equation]) {
		er = !solved_equation(cur_equation);
		if (er) {
			printf(_("Warning: Current equation is not a solved equation.\n"));
		}
		source = rhs[cur_equation];
		nps = &n_rhs[cur_equation];
		dest = rhs[i];
		np = &n_rhs[i];
	} else {
		source = lhs[cur_equation];
		nps = &n_lhs[cur_equation];
		dest = lhs[i];
		np = &n_lhs[i];
	}
	if (*cp) {
		if (is_all(cp)) {
			v = MATCH_ANY;
			cp = skip_param(cp);
		} else {
			cp = parse_var2(&v, cp);
			if (cp == NULL) {
				return false;
			}
		}
		if (*cp) {
			order = atoi(cp);
			cp = skip_num(cp);
		}
	}
	if (extra_garbage(cp)) {
		return false;
	}
	if (order < 1) {
		printf(_("Order out of range.\n"));
		return false;
	}
	if (v == 0) {
		if (!prompt_var(&v)) {
			return false;
		}
	}
	if (v != MATCH_ANY && !found_var(source, *nps, v)) {
		printf(_("Variable not found.  Result would be zero.\n"));
		return false;
	}
	blt(dest, source, *nps * sizeof(token_type));
	*np = *nps;
	simp_loop(dest, np);
	for (j = 0; j < order; j++) {
		if (!differentiate(dest, np, v)) {
			printf(_("Differentiation failed.\n"));
			return false;
		}
		simpa_side(dest, np, false, true);
	}
	if (n_rhs[cur_equation]) {
		blt(lhs[i], lhs[cur_equation], n_lhs[cur_equation] * sizeof(token_type));
		n_lhs[i] = n_lhs[cur_equation];
		if (!er && order < 10) {
/* increment the number of primes in the LHS variable */
			l = lhs[i][0].token.variable;
			for (j = 1; j <= order; j++) {
				l += PRIME_INCREMENT;
				if (l < 0)
					break;
			}
			if (l >= 0) {
				lhs[i][0].token.variable = l;
			}
		}
	}
	cur_equation = i;
	list_sub(cur_equation);
	return true;
}

/*
 * The extrema command.
 */
int
extrema_cmd(cp)
char	*cp;
{
	int		i;
	long		v;
	token_type	want;
	token_type	*source;
	int		n;

	if (notdefined(cur_equation)) {
		return false;
	}
	if (n_rhs[cur_equation]) {
		if (!solved_equation(cur_equation)) {
			printf(_("Current equation is not solved for a variable.\n"));
			return false;
		}
		source = rhs[cur_equation];
		n = n_rhs[cur_equation];
	} else {
		source = lhs[cur_equation];
		n = n_lhs[cur_equation];
	}
	if (*cp) {
		cp = parse_var2(&v, cp);
		if (cp == NULL) {
			return false;
		}
	} else {
		if (!prompt_var(&v)) {
			return false;
		}
	}
	if (extra_garbage(cp)) {
		return false;
	}
	if (!found_var(source, n, v)) {
		printf(_("Variable not found.\n"));
		return false;
	}
	i = next_espace();
	blt(rhs[i], source, n * sizeof(token_type));
	n_rhs[i] = n;
	simp_loop(rhs[i], &n_rhs[i]);
	if (!differentiate(rhs[i], &n_rhs[i], v)) {
		printf(_("Differentiation failed.\n"));
		return false;
	}
	simpa_side(rhs[i], &n_rhs[i], false, true);
	if (!found_var(rhs[i], n_rhs[i], v)) {
		printf(_("There are no extrema for this equation.\n"));
		return false;
	}
	n_lhs[i] = 1;
	lhs[i][0].level = 1;
	lhs[i][0].kind = CONSTANT;
	lhs[i][0].token.constant = 0.0;
	cur_equation = i;
	want.level = 1;
	want.kind = VARIABLE;
	want.token.variable = v;
	if (!solve_sub(&want, 1, lhs[i], &n_lhs[i], rhs[i], &n_rhs[i])) {
		printf(_("Solve failed.\n"));
		return false;
	}
	simpa_side(rhs[i], &n_rhs[i], false, false);
	list_sub(cur_equation);
	return true;
}

/*
 * The taylor command.
 */
int
taylor(cp)
char	*cp;
{
	long	v;
	int	i, j, k;
	int	i1, j1;
	int	level;
	int	n;
	int	order;
	token_type	*ep;
	double	d;
	char	*cp1;
	int	our;
	int	our_nlhs, our_nrhs;

	if (notdefined(cur_equation)) {
		return false;
	}
	if (*cp == '\0') {
		if (!prompt_var(&v))
			return false;
	} else {
		cp = parse_var2(&v, cp);
		if (cp == NULL) {
			return false;
		}
		if (extra_garbage(cp))
			return false;
	}
	if (!found_var(rhs[cur_equation], n_rhs[cur_equation], v)) {
		printf(_("Variable not found.\n"));
		return false;
	}
	i = next_espace();
	blt(lhs[i], lhs[cur_equation], n_lhs[cur_equation] * sizeof(token_type));
	n_lhs[i] = n_lhs[cur_equation];
	n_rhs[i] = 0;
	our = next_espace();
	n_lhs[i] = 0;
	list_var(v, false);
	printf(_("Taylor approximation of current equation about %s = point.\n"), var_str);
	blt(rhs[our], rhs[cur_equation], n_rhs[cur_equation] * sizeof(token_type));
	our_nrhs = n_rhs[cur_equation];
	if (!differentiate(rhs[our], &our_nrhs, v)) {
		printf(_("Differentiation failed.\n"));
		return false;
	}
oops:
	my_strlcpy(prompt_str, _("Enter point: "), sizeof(prompt_str));
	if (!get_expr(lhs[our], &our_nlhs)) {
		printf(_("Enter 0 if you don't know.\n"));
		goto oops;
	}
	my_strlcpy(prompt_str, _("Enter order (number of derivatives to take): "), sizeof(prompt_str));
	if ((cp1 = getstring((char *) scratch, n_tokens * sizeof(token_type))) == NULL)
		return false;
	if (*cp1 == '\0') {
		printf(_("Derivatives will be taken until they reach zero.\n"));
		order = INT_MAX;
	} else {
		if (!isascii(*cp1) || !isdigit(*cp1) || (order = atoi(cp1)) < 0) {
			printf(_("Positive integer required.\n"));
			return false;
		}
	}
	n = 0;
	i1 = 0;
	blt(rhs[i], rhs[cur_equation], n_rhs[cur_equation] * sizeof(token_type));
	n_rhs[i] = n_rhs[cur_equation];
loop_again:
	for (k = i1; k < n_rhs[i]; k += 2) {
		if (rhs[i][k].kind == VARIABLE && rhs[i][k].token.variable == v) {
			level = rhs[i][k].level;
			if ((n_rhs[i] + our_nlhs - 1) > n_tokens)
				error_huge();
			blt(&rhs[i][k+our_nlhs], &rhs[i][k+1], (n_rhs[i] - (k + 1)) * sizeof(token_type));
			n_rhs[i] += our_nlhs - 1;
			j1 = k;
			blt(&rhs[i][k], lhs[our], our_nlhs * sizeof(token_type));
			k += our_nlhs;
			for (; j1 < k; j1++)
				rhs[i][j1].level += level;
			k--;
		}
	}
	if ((n_rhs[i] + our_nlhs + 7) > n_tokens)
		error_huge();
	for (k = i1; k < n_rhs[i]; k++)
		rhs[i][k].level++;
	k = n_rhs[i];
	ep = &rhs[i][k];
	ep->level = 1;
	ep->kind = OPERATOR;
	ep->token.operatr = TIMES;
	ep++;
	ep->level = 3;
	ep->kind = VARIABLE;
	ep->token.variable = v;
	ep++;
	ep->level = 3;
	ep->kind = OPERATOR;
	ep->token.operatr = MINUS;
	k += 3;
	j1 = k;
	blt(&rhs[i][k], lhs[our], our_nlhs * sizeof(token_type));
	k += our_nlhs;
	for (; j1 < k; j1++)
		rhs[i][j1].level += 3;
	ep = &rhs[i][k];
	ep->level = 2;
	ep->kind = OPERATOR;
	ep->token.operatr = POWER;
	ep++;
	ep->level = 2;
	ep->kind = CONSTANT;
	ep->token.constant = n;
	ep++;
	ep->level = 1;
	ep->kind = OPERATOR;
	ep->token.operatr = DIVIDE;
	ep++;
	for (d = 1.0, j = 2; j <= n; j++)
		d *= j;
	ep->level = 1;
	ep->kind = CONSTANT;
	ep->token.constant = d;
	k += 4;
	n_rhs[i] = k;
	for (; i1 < k; i1++)
		rhs[i][i1].level++;
	simp_loop(rhs[i], &n_rhs[i]);
	side_debug(1, rhs[i], n_rhs[i]);
	if (n < order) {
		if (n > 0) {
			if (!differentiate(rhs[our], &our_nrhs, v)) {
				printf(_("Differentiation failed.\n"));
				return false;
			}
		}
		simpa_side(rhs[our], &our_nrhs, false, true);
		if (our_nrhs != 1 || rhs[our][0].kind != CONSTANT
		    || rhs[our][0].token.constant != 0.0) {
			k = n_rhs[i];
			if ((k + 1 + our_nrhs) > n_tokens)
				error_huge();
			for (i1 = 0; i1 < k; i1++)
				rhs[i][i1].level++;
			rhs[i][k].level = 1;
			rhs[i][k].kind = OPERATOR;
			rhs[i][k].token.operatr = PLUS;
			k++;
			i1 = k;
			blt(&rhs[i][k], rhs[our], our_nrhs * sizeof(token_type));
			n_rhs[i] = k + our_nrhs;
			n++;
			goto loop_again;
		}
	}
	n_lhs[i] = n_lhs[cur_equation];
	cur_equation = i;
	list_sub(cur_equation);
	return true;
}

/*
 * The limit command.
 */
int
limit_cmd(cp)
char	*cp;
{
	int		i, j, k;
	long		v;
	token_type	solved_v, want;
	char		*cp_start;
	int		n;
	int		infinity_flag;
	int		found_den;
	int		num, den;
	int		complex_den;
	int		level;

	cp_start = cp;
	if (notdefined(cur_equation)) {
		return false;
	}
	if (!solved_equation(cur_equation)) {
		printf(_("Current equation is not solved for a variable.\n"));
		return false;
	}
	blt(&solved_v, lhs[cur_equation], sizeof(token_type));
	i = next_espace();
	if (*cp) {
		cp = parse_var2(&v, cp);
		if (cp == NULL) {
			return false;
		}
	} else {
		if (!prompt_var(&v)) {
			return false;
		}
	}
	if (!found_var(rhs[cur_equation], n_rhs[cur_equation], v)) {
		printf(_("Variable not found.\n"));
		return false;
	}
	want.level = 1;
	want.kind = VARIABLE;
	want.token.variable = v;
	if (*cp) {
		input_column += (cp - cp_start);
		if (!case_sensitive_flag) {
			str_tolower(cp);
		}
		if ((cp = parse_section(lhs[i], &n, cp)) == NULL || n <= 0) {
			return false;
		}
		if (extra_garbage(cp))
			return false;
	} else {
		my_strlcpy(prompt_str, _("Enter expression: "), sizeof(prompt_str));
		if (!get_expr(lhs[i], &n)) {
			return false;
		}
	}
	simp_loop(lhs[i], &n);
	infinity_flag = (n == 1 && lhs[i][0].kind == CONSTANT
	    && isinf(lhs[i][0].token.constant));
	simpa_side(rhs[cur_equation], &n_rhs[cur_equation], false, false);
	group_proc(rhs[cur_equation], &n_rhs[cur_equation]);
	found_den = false;
	complex_den = false;
	for (j = 1; j < n_rhs[cur_equation]; j += 2) {
		switch (rhs[cur_equation][j].token.operatr) {
		case DIVIDE:
		case MODULUS:
			num = false;
			den = false;
			level = rhs[cur_equation][j].level;
			for (k = j + 2;; k += 2) {
				if (rhs[cur_equation][k-1].kind == VARIABLE
				    && rhs[cur_equation][k-1].token.variable == v) {
					den = true;
					found_den = true;
				}
				if (k >= n_rhs[cur_equation] || rhs[cur_equation][k].level <= level) {
					break;
				}
				switch (rhs[cur_equation][k].token.operatr) {
				case DIVIDE:
				case MODULUS:
					complex_den = true;
					break;
				}
			}
			for (k = j - 1; k >= 0 && rhs[cur_equation][k].level >= level; k--) {
				if (rhs[cur_equation][k].kind == VARIABLE
				    && rhs[cur_equation][k].token.variable == v) {
					num = true;
				}
			}
			if (num && den) {
				complex_den = true;
			}
			break;
		}
	}
	if (!found_den || (infinity_flag && !complex_den)) {
		subst_var_with_exp(rhs[cur_equation], &n_rhs[cur_equation], lhs[i], n, v);
		simpa_side(rhs[cur_equation], &n_rhs[cur_equation], false, false);
		list_sub(cur_equation);
		return true;
	}
	if (infinity_flag) {
		lhs[i][0].token.constant = 0.0;
		tlhs[0].level = 1;
		tlhs[0].kind = CONSTANT;
		tlhs[0].token.constant = 1.0;
		tlhs[1].level = 1;
		tlhs[1].kind = OPERATOR;
		tlhs[1].token.operatr = DIVIDE;
		tlhs[2].level = 1;
		tlhs[2].kind = VARIABLE;
		tlhs[2].token.variable = v;
		n_tlhs = 3;
		subst_var_with_exp(rhs[cur_equation], &n_rhs[cur_equation], tlhs, n_tlhs, v);
	}
	if (!solve_sub(&want, 1, lhs[cur_equation], &n_lhs[cur_equation], rhs[cur_equation], &n_rhs[cur_equation])) {
		printf(_("Can't take limit of this equation because solve failed.\n"));
		return false;
	}
	blt(lhs[cur_equation], lhs[i], n * sizeof(token_type));
	n_lhs[cur_equation] = n;
	symb_flag = true;
	simpa_side(rhs[cur_equation], &n_rhs[cur_equation], false, false);
	symb_flag = false;
	if (!solve_sub(&solved_v, 1, lhs[cur_equation], &n_lhs[cur_equation], rhs[cur_equation], &n_rhs[cur_equation])) {
		printf(_("Can't take limit of this equation because solve failed.\n"));
		return false;
	}
	symb_flag = true;
	simpa_side(rhs[cur_equation], &n_rhs[cur_equation], false, false);
	symb_flag = false;
	list_sub(cur_equation);
	return true;
}
