/*
    ettercap -- parsing utilities

    Copyright (C) 2001  ALoR <alor@users.sourceforge.net>, NaGA <crwm@freemail.it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/


#include "include/ec_main.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#ifdef HAVE_GETOPT_H
	#include <getopt.h>
#else
	#include "missing/getopt.h"
#endif

#include "include/ec_error.h"
#include "include/ec_inet.h"

#ifdef DEBUG
   #include "include/ec_debug.h"
#endif

extern char *Execute_Plugin;

// protos...

void Parser_ParseConfFile(char *filename);
void Parser_ParseParameters(char *first, char *second, char *third, char *fourth);
int Parser_ParseOptions(int counter, char **values);
extern void Main_Usage(void);
//-----------------------------------


int Parser_ParseOptions(int counter, char **values)
{
   int c;

   static struct option long_options[] = {
      { "help", no_argument, NULL, 'h' },
      { "version", no_argument, NULL, 'v' },
      { "simple", no_argument, NULL, 'N' },
      { "list", no_argument, NULL, 'l' },
      { "arpsniff", no_argument, NULL, 'a' },
      { "sniff", no_argument, NULL, 's' },
      { "macsniff", no_argument, NULL, 'm' },
      { "iface", required_argument, NULL, 'i' },
      { "netmask", required_argument, NULL, 'n' },
      { "check", no_argument, NULL, 'c' },
      { "plugin", required_argument, NULL, 'p' },
      { "hexview", no_argument, NULL, 'x' },
      { "silent", no_argument, NULL, 'z' },
      { "udp", no_argument, NULL, 'u' },
      { "fingerprint", no_argument, NULL, 'f' },
      { "linktype", no_argument, NULL, 't' },
      { "collect", no_argument, NULL, 'C' },
      { "broadping", no_argument, NULL, 'b' },
      { "logtofile", no_argument, NULL, 'L' },
      { "quiet", no_argument, NULL, 'q' },
      { "etterconf", required_argument, NULL, 'e' },
      { "dontresolve", no_argument, NULL, 'd' },
      { 0 , 0 , 0 , 0}
   };

	optind = 0;

#ifdef PERMIT_PLUGINS
   while ((c = getopt_long (counter, values, "hvNlasmci:p:xzuftCbn:Lqe:d",long_options, (int *)0)) != EOF) {
#else
   while ((c = getopt_long (counter, values, "hvNlasmci:xzuftCbn:Lqe:d",long_options, (int *)0)) != EOF) {    // no plugin
#endif

      switch (c) {

         case 'h':
            Main_Usage();
         break;

         case 'v':
            fprintf(stdout,"%s\n",VERSION);
            exit(0);
         break;

         case 'N':   Options.normal = 1;     	break;

         case 'l':   Options.list = 1;       	break;

         case 'a':   Options.arpsniff = 1;   	break;

         case 's':   Options.sniff = 1;      	break;

         case 'm':   Options.macsniff = 1;   	break;

         case 'c':   Options.check = 1;      	break;

         case 'x':   Options.hexview = 1;    	break;

         case 'z':   Options.silent = 1;     	break;

         case 'u':   Options.udp = 1;        	break;

         case 'f':   Options.finger = 1;     	break;

         case 't':   Options.link = 1;       	break;

         case 'C':   Options.collect = 1;    	break;

         case 'b':   Options.broadping = 1;  	break;

         case 'L':   Options.logtofile = 1; 		break;

         case 'q':   Options.quiet = 1;      	break;

         case 'd':   Options.dontresolve = 1;	break;

         case 'e':
         	if (!strcmp(values[0], "etter.conf"))
         		Error_msg("You can't specify the --etterconf option in the conf file !! (safe exit avoiding loops)");
         	else
         		return 1;
         break;

#ifdef PERMIT_PLUGINS
         case 'p':
            Options.plugin = 1;
            Execute_Plugin = strdup(optarg);
         break;
#endif

         case 'i':
            strncpy(Options.netiface, optarg, 10);
         break;

         case 'n':
            strncpy(Options.netmask, optarg, 16);
         break;

         case ':': // missing parameter
            Main_Usage();
         break;

         case '?': // unknown option
            Main_Usage();
         break;
      }
   }

	Parser_ParseParameters(values[optind], values[optind+1], values[optind+2], values[optind+3]);

	return 0;
}


void Parser_ParseParameters(char *first, char *second, char *third, char *fourth)
{

#ifdef DEBUG
   Debug_msg("Parser_ParseParameters");
#endif

#define R(a,b,c) (a & b) | ((a ^ b) & c)     // returns true if more than one was selected

	if (Options.quiet && !Options.normal && !Options.logtofile )
		Error_msg("Demonization is only useful with -NL or -NLC !!\n\n");

   if ( R(Options.arpsniff, Options.sniff, Options.macsniff) )
      Error_msg("Please select only one sniffing method !!\n\n");

   if (Options.silent && Options.broadping)
      Error_msg("Please select only one start up method !!\n\n");


   if (Options.silent)
   {
      if (Options.macsniff)
      {
         int i=0;
         char check[6];

         if (first)
         {
            i++;
            sscanf(first, "%17s", Host_Dest.mac);
            if (second)
            {
               i++;
               sscanf(second, "%17s", Host_Source.mac);
            }
         }
         if (i < 2) Error_msg("Please specify both source and destination MAC !!");

	      if (Inet_GetMACfromString(Host_Dest.mac, check ) == -1)   // check for valid mac
	      	Error_msg("Incorrect parsing of MAC [%s] !!\nIt must be in the form 01:02:03:04:05:06 !!", Host_Dest.mac);
	      if (Inet_GetMACfromString(Host_Source.mac, check ) == -1)   // check for valid mac
	      	Error_msg("Incorrect parsing of MAC [%s] !!\nIt must be in the form 01:02:03:04:05:06 !!", Host_Source.mac);
   		if ( !strcmp(Host_Source.mac, Host_Dest.mac) && strcmp(Host_Source.mac, ""))
      		Error_msg("SOURCE and DEST MAC address must be different !!");
      }

      if (Options.arpsniff)
      {
         int i=0;
         char check[6];

         if (first)
         {
            i++;
            sscanf(first, "%128[^:]:%d", Host_Dest.name, &Host_Dest.port);
            if (second)
            {
               i++;
               sscanf(second, "%128[^:]:%d", Host_Source.name, &Host_Source.port);
               if (third)
               {
                  i++;
                  sscanf(third, "%17s", Host_Dest.mac);
                  if (fourth)
                  {
                     i++;
                     sscanf(fourth, "%17s", Host_Source.mac);
                  }
               }
            }
         }
         if (i < 4) Error_msg("Please specify both source and destination IP and MAC !!");
        	if (Inet_GetMACfromString(Host_Dest.mac, check ) == -1)   // check for valid mac
	      	Error_msg("Incorrect parsing of MAC [%s] !!\nIt must be in the form 01:02:03:04:05:06 !!", Host_Dest.mac);
	      if (Inet_GetMACfromString(Host_Source.mac, check ) == -1)   // check for valid mac
	      	Error_msg("Incorrect parsing of MAC [%s] !!\nIt must be in the form 01:02:03:04:05:06 !!", Host_Source.mac);
   		if ( !strcmp(Host_Source.ip, Host_Dest.ip) && strcmp(Host_Source.ip, "") )
      		Error_msg("SOURCE and DEST IP address must be different !!");
   		if ( !strcmp(Host_Source.mac, Host_Dest.mac) && strcmp(Host_Source.mac, ""))
      		Error_msg("SOURCE and DEST MAC address must be different !!");
      }

      if (Options.sniff || Options.plugin)
      {
         if (first)
         {
            sscanf(first, "%128[^:]:%d", Host_Dest.name, &Host_Dest.port);
            if (second) sscanf(second, "%128[^:]:%d", Host_Source.name, &Host_Source.port);

            if (!strcasecmp(Host_Source.name, "ANY") || !strcmp(Host_Source.name, "0") )
            	strcpy(Host_Source.name, "");

            if (!strcasecmp(Host_Dest.name, "ANY") || !strcmp(Host_Dest.name, "0") )
            	strcpy(Host_Dest.name, "");

   			if ( !strcmp(Host_Source.name, Host_Dest.name) && strcmp(Host_Source.name, "") )
      			Error_msg("SOURCE and DEST IP address must be different !!");
         }
      }

      if (Options.check)   Error_msg("You can't check for poisoners in silent mode !!");

      if (Options.list)    Error_msg("You can't make the list in silent mode !!");
   }
   else // !silent
   {
      if (first)
      {
         sscanf(first, "%128[^:]:%d", Host_Dest.name, &Host_Dest.port);
         if (second) sscanf(second, "%128[^:]:%d", Host_Source.name, &Host_Source.port);
      }

      if (Options.arpsniff || Options.macsniff )
         if ( (!strcmp(Host_Source.name, "")) || (!strcmp(Host_Dest.name, "")) )
            Error_msg("Please specify both source and destination host !!");
   }

   if (strcmp(Host_Source.name, ""))
      strcpy(Host_Source.ip, Inet_NameToIp(Host_Source.name));

   if (strcmp(Host_Dest.name, ""))
      strcpy(Host_Dest.ip, Inet_NameToIp(Host_Dest.name));



#ifdef DEBUG
   Debug_msg("Parser_ParseParameters - name - [%s][%s]", Host_Dest.name, Host_Source.name);
   Debug_msg("Parser_ParseParameters -  IP  - [%s][%s]", Host_Dest.ip, Host_Source.ip);
   Debug_msg("Parser_ParseParameters - port - [%d][%d]", Host_Dest.port, Host_Source.port);
   Debug_msg("Parser_ParseParameters -  MAC - [%s][%s]", Host_Dest.mac, Host_Source.mac);
#endif

}


void Parser_ParseConfFile(char *filename)
{

	FILE *etterconf;
	char line[100];
	char to_be_parsed[1000];	// large enough I suppose... ;)
	char *options = NULL;
	char *iface = NULL;
	char *netmask = NULL;
	char *plugin = NULL;
	char *ip1 = NULL;
	char *ip2 = NULL;
	char *mac1 = NULL;
	char *mac2 = NULL;
	char *ptr;
	int pargc = 0;
	char *pargv[256];

#ifdef DEBUG
   Debug_msg("Parser_ParseConfFile - %s", filename);
#endif

	fprintf (stdout, "Loading options from %s...\n", filename);

	memset(to_be_parsed, 0, 1000);
	strcat(to_be_parsed, "etter.conf ");

	etterconf = fopen(filename, "r");
	if (etterconf == NULL)
         Error_msg("ec_parser:%d fopen(%s) | ERRNO : %d | %s", filename, __LINE__, errno, strerror(errno));

	do
	{
		fgets(line, 100, etterconf);
		if ( (ptr = strchr(line, '#')) )
			*ptr = 0;

		if ( (ptr = strchr(line, '\n')) )
			*ptr = 0;

		if (!strlen(line))	// skip 0 length line
			continue;

#ifdef DEBUG
   Debug_msg("Parser_ParseConfFile - [%s]", line);
#endif

		if (!strncasecmp(line, "OPTIONS: ", 9))
			options = strdup(strchr(line, '-'));

		if (!strncasecmp(line, "IFACE: ", 7))
			iface = strdup(line+7);

		if (!strncasecmp(line, "NETMASK: ", 9))
			netmask = strdup(line+9);

#ifdef PERMIT_PLUGINS
		if (!strncasecmp(line, "PLUGIN: ", 8))
			plugin = strdup(line+8);
#endif

		if (!strncasecmp(line, "IP1: ", 5))
			ip1 = strdup(line+5);

		if (!strncasecmp(line, "IP2: ", 5))
			ip2 = strdup(line+5);

		if (!strncasecmp(line, "MAC1: ", 6))
			mac1 = strdup(line+6);

		if (!strncasecmp(line, "MAC2: ", 6))
			mac2 = strdup(line+6);

	} while (!feof(etterconf));

	if (options)
	{
		strcat(to_be_parsed, options);
		free(options);
	}
	if (iface)
	{
		strcat(to_be_parsed, " --iface ");
		strcat(to_be_parsed, iface);
		free(iface);
	}
	if (netmask)
	{
		strcat(to_be_parsed, " --netmask ");
		strcat(to_be_parsed, netmask);
		free(netmask);
	}
	if (plugin)
	{
		strcat(to_be_parsed, " --plugin ");
		strcat(to_be_parsed, plugin);
		free(plugin);
	}
	if (ip1)
	{
		strcat(to_be_parsed, " ");
		strcat(to_be_parsed, ip1);
		free(ip1);
	}
	if (ip2)
	{
		strcat(to_be_parsed, " ");
		strcat(to_be_parsed, ip2);
		free(ip2);
	}
	if (mac1)
	{
		strcat(to_be_parsed, " ");
		strcat(to_be_parsed, mac1);
		free(mac1);
	}
	if (mac2)
	{
		strcat(to_be_parsed, " ");
		strcat(to_be_parsed, mac2);
		free(mac2);
	}

	if (!strcmp(to_be_parsed, "etter.conf "))	// no options in the file....
		return;

	ptr = strtok(to_be_parsed, " ");
	pargv[pargc++] = strdup(ptr);

	while( (ptr = strtok(NULL, " ")) )
		pargv[pargc++] = strdup(ptr);

	pargc--;

#ifdef DEBUG
{
	int i;
	for(i=0; i<pargc; i++)
	   Debug_msg("Parser_ParseConfFile - [%d] %s", i, pargv[i]);
}
#endif

	Parser_ParseOptions(pargc, pargv);


}


/* EOF */

