/*
    ettercap -- a ncurses-based sniffer/interceptor utility for switched LAN

    Copyright (C) 2001  ALoR <alor@users.sourceforge.net>, NaGA <crwm@freemail.it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include "include/ec_main.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/ioctl.h>
#ifdef HAVE_GETOPT_H
	#include <getopt.h>
#else
	#include "missing/getopt.h"
#endif


#include "include/ec_error.h"
#include "include/ec_inet.h"
#include "include/ec_simple.h"
#include "include/ec_signal.h"
#include "include/ec_parser.h"

#ifdef HAVE_NCURSES
    #include "include/ec_interface.h"
#endif

#ifdef DEBUG
   #include "include/ec_debug.h"
#endif

// global variables

char *program_argv0;

HOST *Host_In_LAN = NULL;        // ec_main.h
int number_of_hosts_in_lan;

CONNECTION *Conn_Between_Hosts = NULL;       // ec_main.h
int number_of_connections;

HOST Host_Source;
HOST Host_Dest;

char *Execute_Plugin;

OPTIONS Options;

int pipe_with_illithid;
int pipe_with_illithid_data;
int pipe_inject_stod[2];
int pipe_inject_dtos[2];
int pipe_filter_stod[2];
int pipe_filter_dtos[2];

char active_dissector = 1;    // ec_main.h

// protos...

void Main_Usage(void);
void Main_Interactive(void);
void Main_Normal(void);
void Main_CheckForRun(void);
void Main_ParseParameters(char *first, char *second, char *third, char *fourth);
void Main_Devel_Release(void);

//-----------------------------------

int main(int argc, char *argv[])
{

   program_argv0 = argv[0];

#ifdef DEBUG
   Debug_Init();
   Debug_msg("Main");
#endif

   Main_CheckForRun();     // it is ok ?

   Signal_SigBuster();     // signal masking

#if DEVEL_RELEASE == 1
	Main_Devel_Release();
#endif

#ifdef DEBUG
{
	int i;
	for(i=0; i<argc; i++)
	   Debug_msg("Main_ParameterList - [%d] %s", i, argv[i]);
}
#endif

	if (Parser_ParseOptions(argc, argv) == 1)
		Parser_ParseConfFile(optarg);

   if (!strcmp(Options.netiface, ""))        // set the default interface
      if (Inet_FindIFace(Options.netiface) == -1)
         Error_msg("No suitable Network Interface found !!");

   if ( Inet_CorrectIface(Options.netiface) < 0)
#ifdef LINUX
   	Error_msg("%s (%s)", strerror(errno), Options.netiface);
#else
      Error_msg("Network Interface %s is invalid !!", Options.netiface);
#endif


   if (Options.normal)
      Main_Normal();
   else
      Main_Interactive();

return 0;
}




void Main_Usage(void)
{

#ifdef DEBUG
   Debug_msg("Main_Usage");
#endif

   fprintf (stdout, "\n\033[01m\033[1m%s %s (c) 2001 %s\033[0m\n\n", PROGRAM, VERSION, AUTHORS);
   fprintf (stdout, "\nUsage: %s [OPTION] [HOST:PORT] [HOST:PORT] [MAC] [MAC]\n\n", PROGRAM);

   fprintf (stdout, "  -N, --simple                 NON interactive mode (without ncurses)\n\n");
   fprintf (stdout, "  -z, --silent                 silent mode (no arp storm on start up)\n\n");
   fprintf (stdout, "  -b, --broadping              broadcast ping instead of arp storm on start up\n\n");
   fprintf (stdout, "  -u, --udp                    sniff only udp connection (default is tcp)\n\n");
   fprintf (stdout, "  -a, --arpsniff               arp based sniffing (must specify two host)\n");
   fprintf (stdout, "                               in silent mode : must specify two IP and two MAC\n");
   fprintf (stdout, "                                    i.e.: ettercap -za IP IP MAC MAC\n\n");
   fprintf (stdout, "  -s, --sniff                  ip based sniffing\n");
   fprintf (stdout, "                               you can specify null ip that means ALL hosts\n\n");
   fprintf (stdout, "  -m, --macsniff               mac based sniffing (must specify two host)\n");
   fprintf (stdout, "                               in silent mode : must specify two MAC address\n");
   fprintf (stdout, "                                    i.e.: ettercap -zm MAC MAC\n\n");
   fprintf (stdout, "  -l, --list                   list all hosts in the lan\n\n");
   fprintf (stdout, "  -f, --fingerprint HOST       do OS fingerprinting on HOST\n\n");
   fprintf (stdout, "  -C, --collect                collect users and passwords only\n");
   fprintf (stdout, "                               this options must be used with a sniffing method\n");
   fprintf (stdout, "                                    Eg: ettercap -NCzs\n\n");
   fprintf (stdout, "  -x, --hexview                display data in hex mode\n\n");
   fprintf (stdout, "  -L, --logtofile              logs all data to specific file(s)\n\n");
   fprintf (stdout, "  -q, --quiet                  \"demonize\" ettercap (useful with -L)\n\n");
   fprintf (stdout, "  -i, --iface IFACE            network interface to be used\n\n");
   fprintf (stdout, "  -n, --netmask NETMASK        the netmask used to scan the lan\n\n");
   fprintf (stdout, "  -c, --check                  check for other poisoners in the LAN\n\n");
   fprintf (stdout, "  -t, --linktype               am I in a switched LAN or not ?\n\n");
#ifdef PERMIT_PLUGINS
   fprintf (stdout, "  -p, --plugin [name]          run the \"name\" plugin (\"list\" for available ones)\n\n");
#endif
	fprintf (stdout, "  -e, --etterconf [filename]   use the config file instead of command line options\n\n");
	fprintf (stdout, "  -d, --dontresolve            don't resolve the IPs (speed the startup)\n\n");
   fprintf (stdout, "  -v, --version                print version\n\n");
   fprintf (stdout, "  -h, --help                   print this help\n\n");
   fprintf (stdout, "\n");

   exit (0);
}



void Main_Interactive(void)
{
#ifdef HAVE_NCURSES
   struct winsize  ws = {0, 0, 0, 0};
   #include <ncurses.h>
#endif

#ifdef DEBUG
   Debug_msg("Main_Interactive");
#endif

#ifdef HAVE_NCURSES

   if ( ioctl(0, TIOCGWINSZ, &ws) < 0)          // syscall for the window size
   	Error_msg("ec_main:%d ioctl(TIOCGWINSZ) | ERRNO : %d | %s", __LINE__, errno, strerror(errno));

   if ( (ws.ws_row < 25) || (ws.ws_col < 80) )
      Error_msg("Screen must be at least 25x80 !!");

   if (!Options.silent) printf("Building host list for netmask %s, please wait...\n", Inet_MySubnet());
   number_of_hosts_in_lan = Inet_HostInLAN();

   Interface_InitTitle(Host_In_LAN[0].ip, Host_In_LAN[0].mac, Inet_MySubnet());
   Interface_InitScreen();
   Interface_Run();
#else
   #ifdef DEBUG
      Debug_msg("Ncurses not supported -- turning to non interactive mode...");
   #endif
   Main_Normal();
#endif

#ifdef DEBUG
   Debug_msg("Main_Interactive_END");
#endif
}




void Main_Normal(void)
{

#ifdef DEBUG
   Debug_msg("Main_Normal");
#endif

   printf ("\n\033[01m\033[1m%s %s (c) 2001 %s\033[0m\n\n", PROGRAM, VERSION, AUTHORS);
   printf ("Your IP: %s with MAC: %s on Iface: %s\n", Inet_MyIPAddress(), Inet_MyMACAddress(), Options.netiface);


   if (Options.list || Options.check || Options.arpsniff || Options.sniff  || Options.macsniff || Options.link)
   {
      if (!Options.silent) printf("Building host list for netmask %s, please wait...\n", Inet_MySubnet());
      number_of_hosts_in_lan = Inet_HostInLAN();
   }

   if (Options.list)
      Simple_HostList();

   if (Options.check)
      Simple_CheckForPoisoner();

   if (Options.link)
      Simple_CheckForSwitch();

   if (Options.finger)
      Simple_FingerPrint();

#ifdef PERMIT_PLUGINS
   if (Options.plugin)
      Simple_Plugin();
#endif

   if (Options.arpsniff || Options.sniff  || Options.macsniff)
      Simple_Run();

   printf("\n");
#ifdef DEBUG
   Debug_msg("Main_Normal_END");
#endif

exit(0);
}



void Main_CheckForRun(void)
{

#ifdef DEBUG
   Debug_msg("Main_CheckForRun");
#endif

#ifdef PERMIT_SUID
   if (geteuid() != 0)
      Error_msg("Sorry EUID %d, must be EUID = 0 to run %s !!", geteuid(), PROGRAM);
#else
   if (getuid() != 0)
      Error_msg("Sorry UID %d, must be root to run %s !!", getuid(), PROGRAM);
#endif

	if (!strstr(program_argv0, PROGRAM))		// just for script-kiddies ;)
		Error_msg("ehi guy ! my name is \"%s\" ! I really don't like \"%s\"...", PROGRAM, program_argv0);

}


void Main_Devel_Release(void)
{

	fprintf (stdout, "\n\n");
	fprintf (stdout, "  %s %s IS STILL IN DEVELOPMENT STATE. ABSOLUTELY NO WARRANTY !\n\n", PROGRAM, VERSION);
	fprintf (stdout, "  if you are a betatester please report bugs to :\n");
	fprintf (stdout, "      http://ettercap.sourceforge.net/forum/viewforum.php?forum=7\n\n");
	fprintf (stdout, "  or send an email to:\n");
	fprintf (stdout, "      alor@users.sourceforge.net\n");
	fprintf (stdout, "      crwm@freemail.it\n\n");
	fprintf (stdout, "  if you are NOT a betatester, I don't know where you downloaded this release\n");
	fprintf (stdout, "  but this is NOT for you, so don't blame us for any bugs or problems !\n");
	fprintf (stdout, "\n\n");

}

/* EOF */


