#!/usr/bin/env python

# telepathy-pinocchio - dummy Telepathy connection manager for instrumentation
# :: test suite
#
# Copyright (C) 2008 Nokia Corporation
# Copyright (C) 2008 Collabora Ltd.
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# version 2.1 as published by the Free Software Foundation.
#
# This library is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
# 02110-1301 USA

import os
import sys
import commands
from xml.dom import minidom

COMMON_METADATA_KEYS = ('username', 'alias')

def contacts_file_get_contacts(contacts_filename):
    """Returns contacts from the given contact list file.
    
    Returns:
    contacts_xml -- list of contacts from file as xml.dom.minidom.DOMElement
                    objects
    """
    # do some initial contact list parsing
    contacts_file = open(contacts_filename, 'rU')
    contacts_file_xml = minidom.parse(contacts_file)
    contacts_file.close()

    list = contacts_file_xml.getElementsByTagName('roster')[0]
    contacts_xml = list.getElementsByTagName('contact')

    return contacts_xml

def contacts_xml_get_contacts(contacts_xml):
    """Returns flattened Contact objects from contact XML objects.

    Arguments:
    contacts_xml -- dictionary mapping protocol-specific UIDs to
                    xml.dom.minidom.DOMEelement objects to parse
    """
    contact_objs = {}
    for contact in contacts_xml:
        contact_obj = pin.server._Contact(contact)
        contact_objs[contact_obj.metadata['username']] = contact_obj

    return contact_objs

def contact_get_objs_from_output(output):
    """Returns Contact objects based on the output from pinocchio-ctl.

    Arguments:
    output -- full, unmodified output from pinocchio-ctl

    Returns:
    dictionary mapping contact usernames to pinocchio.server._Contact objects
    """
    contact_objs_from_output = {}

    lists = output.split('\n\n\n')
    for list in lists:
        list_lines = list.strip().split('\n')

        list_name = list_lines[0].strip('[]')

        # only check the output of subscribe for now
        if list_name == 'subscribe':
            contacts_block = '\n'.join(list_lines[1:])
            contacts_text = contacts_block.split('\n\n')

            # dissect each block of contacts and store their components in
            # contact_objs_from_output (to compare with contact_objs_from_file)
            username_cur = None
            for contact_text in contacts_text:
                contact_lines = contact_text.split('\n')

                for contact_line in contact_lines:
                    try:
                        key, value = contact_line.split(': ')
                    except ValueError:
                        key = contact_line.split(': ')[0]
                        value = ''

                    key = key.strip(':')
                    value = value.strip()
                    value = value.lstrip('"')
                    value = value.rstrip('"')

                    # attribute all following key/value pairs to this username
                    # (until we find the next username line)
                    if key == 'username':
                        username_cur = value
                        contact_objs_from_output[username_cur] = \
                                                    pin.server._Contact(None)

                    if key in COMMON_METADATA_KEYS:
                        contact_objs_from_output[username_cur].metadata[key] \
                                                                        = value

    return contact_objs_from_output

def output_is_subset_of_file(contact_objs_from_file, contact_objs_from_output):
    """Returns True if the contacts in pinocchio-ctl's output are a subset of
    contacts read from the file.

    Arguments:
    contact_objs_from_file -- dictionary of pinocchio.server._Contact objects
                              based on file contents
    contact_objs_from_output -- dictionary of pinocchio.server._Contact objects
                                based on pinocchio-ctl's output

    Returns:
    True if contact_objs_from_output is a subset of contact_objs_from_file
    """
    is_subset = True

    for username, contact_obj_output in contact_objs_from_output.items():
        if username in contact_objs_from_file:
            contact_obj_file = contact_objs_from_file[username]
            for key_file, value_file in contact_obj_file.metadata.items():
                if key_file in COMMON_METADATA_KEYS:
                    value_output = contact_obj_output.metadata[key_file]

                    # the shell output is assumed ASCII; ensure it's converted
                    # to utf-8 before comparing
                    value_output = value_output.decode('utf-8')
                    if value_output != value_file:
                        print >> sys.stderr, 'metadata mismatch:'
                        format = '    file: key: "%s", value: "%s"'
                        print >> sys.stderr, format % \
                                                    (key_file,
                                                     value_file.encode('utf-8'))
                        format = '  output: key: "%s", value: "%s"'
                        print >> sys.stderr, format % (key_file, value_output)
                        print >> sys.stderr

                        is_subset = False
        else:
            format = 'contact from output not found in file:\n%s\n'
            print >> sys.stderr, format % contact_obj_output.metadata
            is_subset = False

    return is_subset

def file_is_subset_of_output(contact_objs_from_file, contact_objs_from_output):
    """Returns True if the contacts read from the file are a subset of the
    contacts in pinocchio-ctl's output.

    Arguments:
    contact_objs_from_file -- dictionary of pinocchio.server._Contact objects
                              based on file contents
    contact_objs_from_output -- dictionary of pinocchio.server._Contact objects
                                based on pinocchio-ctl's output

    Returns:
    True if contact_objs_from_file is a subset of contact_objs_from_output
    """
    is_subset = True

    for username, contact_obj_file in contact_objs_from_file.items():
        if username not in contact_objs_from_output:
            format = 'contact from file not found in output:\n%s\n\n'
            print >> sys.stderr, format % contact_obj_file.metadata
            is_subset = False

    return is_subset


if __name__ == '__main__':
    import telepathy as tp

    try:
        import pinocchio as pin
    except ImportError, err:
        print >> sys.stderr, """telepathy-pinocchio must be installed before
        these tests can be run
        """
        sys.exit(1)

    # TODO: factor this out as an arg
    ACCOUNT_NAME = 'default@default'
    account_id = tp.server.conn._escape_as_identifier(ACCOUNT_NAME)
    contacts_filename = pin.common.get_contacts_file(account_id=account_id)
    contacts_xml = contacts_file_get_contacts(contacts_filename)

    # create a dict of contact objects to compare to the output of pinocchio-ctl
    contact_objs_from_file = contacts_xml_get_contacts(contacts_xml)
    # create corresponding dict of contact objects to validate output
    contact_objs_from_output = {}

    command_str = "pinocchio-ctl --username %s list" % ACCOUNT_NAME

    status, output = commands.getstatusoutput(command_str)

    contact_objs_from_output = contact_get_objs_from_output(output)

    # assume the output from pinocchio-ctl is valid unless we find a clear
    # problem (simplifies logic below)
    results_valid = True

    # check file and output contacts are subsets of each other -- if they're
    # mutual subsets, they're equal
    results_valid &= output_is_subset_of_file(contact_objs_from_file,
                                              contact_objs_from_output)

    results_valid &= file_is_subset_of_output(contact_objs_from_file,
                                              contact_objs_from_output)

    if not results_valid:
        print >> sys.stderr, 'Test FAILED!'
        sys.exit(1)
