/*
 *	Copyright (c) 1993-1997 JSC Rinet, Novosibirsk, Russia
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 * Redistribution in binary form may occur without any restrictions.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

/* color.c -- trafshow color support */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef	HAVE_HAS_COLORS

#ifdef	HAVE_SLCURSES
#include <slcurses.h>
#endif
#ifdef	HAVE_NCURSES
#include <ncurses.h>
#endif
#ifdef	HAVE_CURSES
#include <curses.h>
#endif

#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include <unistd.h>
#include <ctype.h>
#include <netdb.h>

#include "trafshow.h"

#ifndef	INADDR_NONE
#define	INADDR_NONE	0xffffffff
#endif

static struct m_entry *color_mask = NULL;
static int n_masks = 0;
static int n_pairs = 0;
static char *rc_file;
static int rc_line;

/* The SLcurses can't handle blink attribute as ncurses; so hack it */
#ifdef	HAVE_SLCURSES
#ifdef	A_BOLD
#undef	A_BOLD
#endif
#define	A_BOLD		SLTT_BOLD_MASK
#ifdef	A_BLINK
#undef	A_BLINK
#endif
#define	A_BLINK		SLTT_BLINK_MASK

static void
slang_init_pair(pair, fc, bc, at)
	short pair, fc, bc;
	int at;
{
	SLtt_set_color_object(pair, ((fc | (bc << 8)) << 8) | at);
}

static int
slang_pair_content(pair, fc, bc)
	short pair, *fc, *bc;
{
	int attr;
	SLtt_Char_Type at;

	at = SLtt_get_color_object(pair);
	attr = at & (A_BOLD | A_BLINK);
	at &= ~(A_BOLD | A_BLINK);
	at >>= 8;
	*fc = at & 0xff;
	*bc = (at >> 8) & 0xff;

	return attr;
}
#endif	/* HAVE_SLCURSES */

static short
findpair(f, b, a)
	short f, b;
	int a;
{
	int i, a1;
	short f1, b1;
	struct m_entry *m;

	for (m = color_mask, i = 0; m != NULL && i < n_masks-1; m++, i++) {
#ifdef	HAVE_SLCURSES
		a1 = slang_pair_content(m->pair, &f1, &b1);
		if (f == f1 && b == b1 && a == a1) return m->pair;
#else
		pair_content(m->pair, &f1, &b1);
		if (f == f1 && b == b1) return m->pair;
#endif
	}
	return 0;
}

static void
addcolormask(s, m)
	char *s;
	struct m_entry *m;
{
	int i, attr = 0;
	short fc, bc;
	char f[100], *b;
	static short fc_def = COLOR_WHITE, bc_def = COLOR_BLACK;
	static char *ctab[8] = { "black", "red", "green", "yellow",
				"blue",	"magenta", "cyan", "white" };

	if ((b = strchr(strcpy(f, s), ':')) != NULL) *b++ = '\0';

	if (*f) {
		for (i = 0; i < 8; i++)
			if (!strcasecmp(ctab[i], f)) break;
		if (i < 8) fc = i;
		else {
			fc = atoi(f);
			if (fc < 1 || fc > COLORS)
				error(0, "%s: line %d: Unknown color `%s'", rc_file, rc_line, f);
		}
		if (isupper(*f)) attr |= A_BOLD;
	} else fc = fc_def;

	if (b && *b) {
		for (i = 0; i < 8; i++)
			if (!strcasecmp(ctab[i], b)) break;
		if (i < 8) bc = i;
		else {
			bc = atoi(b);
			if (bc < 1 || bc > COLORS)
				error(0, "%s: line %d: Unknown color `%s'",
				      rc_file, rc_line, b);
		}
		if (isupper(*b)) attr |= A_BLINK;
	} else bc = bc_def;

	if (m != NULL) {
		if ((color_mask = realloc(color_mask, ++n_masks * sizeof(struct m_entry))) == NULL)
			error(1, "addcolormask: realloc");
		if ((m->pair = findpair(fc, bc, attr)) == 0) {
			if (++n_pairs < COLOR_PAIRS-1)
#ifdef	HAVE_SLCURSES
				slang_init_pair(n_pairs, fc, bc, attr);
#else
				init_pair(n_pairs, fc, bc);
#endif
			else	error(0, "%s: line %d: Max %d color-pairs can be used",
				      rc_file, rc_line, COLOR_PAIRS-1);
			m->pair = n_pairs;
		}
		m->attr = attr;
		memcpy(color_mask + (n_masks-1), m, sizeof(struct m_entry));
	} else {	/* default colors */
#ifdef	HAVE_SLCURSES
		slang_init_pair(0, fc, bc, attr);
#else
#ifdef	HAVE_BKGD
		init_pair(COLOR_PAIRS-1, fc, bc);
		bkgd(COLOR_PAIR(COLOR_PAIRS-1) | attr);
#elif	HAVE_WBKGD
		init_pair(COLOR_PAIRS-1, fc, bc);
		wbkgd(stdscr, COLOR_PAIR(COLOR_PAIRS-1) | attr);
#else /* assume the color-pair 0 is background for whole screen */
		init_pair(0, fc, bc);
#endif
#endif
		fc_def = fc;
		bc_def = bc;
	}
}

static u_int32_t
addr_mask(addr)
	u_int32_t addr;
{
	register u_int32_t m = INADDR_BROADCAST;

	if (addr) {
		while ((addr & IN_CLASSA_NET) == 0)
			addr <<= IN_CLASSC_NSHIFT, m >>= IN_CLASSC_NSHIFT;
	} else m = INADDR_ANY;
	return m;
}

/*
 * Left justify 'addr' and return its resulting network mask.
 */
static u_int32_t
net_mask(addr)
	u_int32_t *addr;
{
	register u_int32_t m = INADDR_BROADCAST;

	if (*addr)
		while ((*addr & 0xff000000) == 0)
			*addr <<= 8, m <<= 8;
	return m;
}

static int
isany(s)
	char *s;
{
	return (!strcmp(s, "*") || !strcasecmp(s, "any") || !strcasecmp(s, "all"));
}

static u_int32_t
str2addr(str, mask)
	char *str;
	u_int32_t *mask;
{
	struct in_addr in;
	struct netent *np;
	struct hostent *hp;

	if ((in.s_addr = inet_network(str)) != INADDR_NONE) {
		net_mask(&in.s_addr);
		in.s_addr = htonl(in.s_addr);
		*mask = addr_mask(in.s_addr);
		return in.s_addr;
	}
	if (isany(str)) {
		*mask = INADDR_ANY;
		return INADDR_ANY;
	}
	if ((np = getnetbyname(str)) != NULL) {
		in.s_addr = np->n_net;
		*mask = htonl(net_mask(&in.s_addr));
		return htonl(in.s_addr);
	}
	if ((hp = gethostbyname(str)) != NULL) {
		in.s_addr = *(u_int32_t *)*hp->h_addr_list;
		*mask = addr_mask(in.s_addr);
		return in.s_addr;
	}
	error(0, "%s: line %d: Unknown host or network `%s'",
	      rc_file, rc_line, str);

	/* NOTREACHED */
	return 0;
}

static u_short
str2port(str, proto)
	char *str, *proto;
{
	u_short port;
	struct servent *sp;

	if ((port = atoi(str)) < 1) {
		if (isany(str)) return (u_short)0;
		if ((sp = getservbyname(str, proto)) != NULL)
			port = ntohs(sp->s_port);
		else {
			if (proto)
				error(0, "%s: line %d: Unknown port `%s' protocol `%s'",
				      rc_file, rc_line, str, proto);
			else	error(0, "%s: line %d: Unknown port `%s'",
				      rc_file, rc_line, str);
		}
	}
	return port;
}

static u_int32_t
parsehostport(s, mask, port, proto)
	char *s, *proto;
	u_int32_t *mask;
	u_short *port;
{
	u_int32_t addr;
	char h[100], *m, *p;

	if ((p = strchr(strcpy(h, s), ':')) != NULL) *p++ = '\0';
	if ((m = strchr(h, '/')) != NULL) *m++ = '\0';
	addr = str2addr(h, mask);
	if (m) {
		if (strchr(m, '.') == NULL) {	/* mean number of bits */
			int i, n = atoi(m);
			*mask = 0;
			for (i = 0; i < n; i++) {
				*mask >>= 1;
				*mask |= 0x80000000L;
			}
			*mask = htonl(*mask);
		} else *mask = inet_addr(m);
	}
	*port = (u_short)0;
	if (p) *port = str2port(p, proto);
	return addr;
}

static char *
str2proto(str, proto)
	char *str;
	u_short *proto;
{
	short num;

	if (isany(str)) {
		*proto = (u_short)0;
		return NULL;
	}
	if ((num = getprotonum(str)) == -1)
		error(0, "%s: line %d: Unknown protocol `%s'",
		      rc_file, rc_line, str);

	*proto = num;
	return getprotoname(*proto);
}

int
init_colormask()
{
	FILE *fp;
	int ns;
	struct m_entry m;
	struct passwd *pw;
	char *p, buf[1024];
	char s1[100], s2[100], s3[100], s4[100];
	extern char *program_name;

	if ((pw = getpwuid(getuid())) == NULL)
		error(1, "init_color_mask: getpwuid");
	(void) sprintf(buf, "%s/.%s", pw->pw_dir, program_name);
	if ((fp = fopen(buf, "r")) == NULL) {
		(void) strcpy(buf, "/etc/");
		(void) strcat(buf, program_name);
		if ((fp = fopen(buf, "r")) == NULL) return 0;
	}
	if ((rc_file = (char *)malloc(strlen(buf)+1)) == NULL)
		error(1, "init_color_mask: malloc");
	(void)strcpy(rc_file, buf);
	rc_line = 0;
	while (fgets(buf, sizeof(buf), fp) != NULL) {
		rc_line++;
		if (buf[0] == '\n' || buf[0] == '#') continue;
		if ((p = strchr(buf, '#')) != NULL) {
			*p++ = '\n';
			*p = '\0';
		}
		memset(&m, 0, sizeof(struct m_entry));
		ns = sscanf(buf, "%s %s %s %s\n", s1, s2, s3, s4);
		if (ns == 2) {
			if (strcasecmp(s1, "default")) {
				if ((p = strchr(s1, '/')) != NULL) {
					*p++ = '\0';
					m.dport = str2port(s1, str2proto(p, &m.proto));
				} else	m.dport = str2port(s1, NULL);
				addcolormask(s2, &m);
				m.sport = m.dport;
				m.dport = 0;
				addcolormask(s2, &m);
			} else addcolormask(s2, NULL);
		} else if (ns == 4) {
			p = str2proto(s3, &m.proto);
			m.src.s_addr = parsehostport(s1, &m.sm.s_addr, &m.sport, p);
			m.dst.s_addr = parsehostport(s2, &m.dm.s_addr, &m.dport, p);
			addcolormask(s4, &m);
		} else error(0, "%s: line %d: Bad format", rc_file, rc_line);
	}
	fclose(fp);
#ifdef	DEBUG
	error(0, "n_masks=%d, n_pairs=%d", n_masks, n_pairs);
#endif
	return n_masks;
}

int
color_on(e)
	register struct t_entry *e;
{
	int i;
	register struct m_entry *m;

	for (m = color_mask, i = 0; m != NULL, i < n_masks; m++, i++) {
		if ((e->src.s_addr & m->sm.s_addr) ^ m->src.s_addr)
			continue;
		if ((e->dst.s_addr & m->dm.s_addr) ^ m->dst.s_addr)
			continue;
		if (m->proto && e->proto != m->proto)
			continue;
		if (m->sport && e->sport != m->sport)
			continue;
		if (m->dport && e->dport != m->dport)
			continue;
#ifdef	HAVE_SLCURSES
		attron(COLOR_PAIR(m->pair));
#else
		attron(COLOR_PAIR(m->pair) | m->attr);
#endif
		return TRUE;
	}
	return FALSE;
}

#endif	/* HAVE_HAS_COLORS */
