#! /usr/bin/perl

use Getopt::Long;
use POSIX;
use strict;
use vars qw($clo $CXX);

eval {
	require "../test/common.pl";
}; if ($@) {
	require "../common.pl";
}

$| = 1;

my %tests;
my %permutations;
my @test_names;
my %clo;

GetOptions(
	\%clo,
	'only=s',
	'savep=s',
	'loadp=s',
);

print "#" x 70, "\n";
print "                     Type Permutation Test\n";
print "#" x 70, "\n";

opendir(DATADIR, "data") || die $!;
my @files = grep {! /^\./} readdir(DATADIR);
closedir(DATADIR);

foreach my $file (@files) {
	parse_data_file($file);
}

if ($clo{'only'}) {
	unless (grep {$_ eq $clo{'only'}} keys %tests) {
		die "no such test $clo{'only'}\n";
	}

	@test_names = ($clo{'only'});
} else {
	@test_names = sort keys %tests;
}

if ($clo{'savep'}) {
	open(SAVEP, ">$clo{'savep'}") || die $!;
} elsif ($clo{'loadp'}) {
	open(LOADP, $clo{'loadp'}) || die $!;
	while (<LOADP>) {
		chomp;
		my ($tt, $valist) = split(/\s*=\s*/, $_, 2);
		push @{$permutations{$tt}}, [split(/, /, $valist)];
	}
	close(LOADP);
}

mkdir "sandbox", 0777 || die $!;
chdir "sandbox" || die $!;

my (%pids, $pid, $kill_pid);
$SIG{CHLD} = sub {
	while (($kill_pid = waitpid(-1, WNOHANG)) > 0) {
		delete $pids{$kill_pid};
	}
};

foreach my $test (@test_names) {
	mkdir $test, 0777 || die $!;
	FORK: { # fork Camel Style
		if ($pid = fork) {
			$pids{$pid} = 1;
		}
		elsif (defined $pid) {
			chdir $test || die $!;
			child($test);
			exit;
		}
		elsif ($! =~ /No more process/) {
			sleep 5;
			redo FORK;
		}
		else {
			die "$0: cannot fork: $!\n";
		}
	}
}

while (1) {
	last if keys %pids == 0;
	print scalar localtime, ": Tests Running: ", scalar keys %pids, "\n";
	sleep 4;
}	

sub child {
	my $test_type = shift;
	my ($state, $val);
	my $u_permute = {};
	my $c_total = 0;

	open(OUTPUT, ">test.output") || die;
	select((select(OUTPUT), $|=1)[0]);

	print OUTPUT "===> Permutation Test for Type $test_type\n";
	$state = permute_init() unless $clo{'loadp'};

	for (;;) {
		unless ($clo{'loadp'}) {
			($state, $val) = permute_next($state, $u_permute);
		} else {
			$val = shift @{$permutations{$test_type}};
		}

		last unless defined $val;
		next unless grep {$_ eq $test_type} @$val;

		$c_total++;

		print OUTPUT "\t===> Permutation [$c_total] (", join(", ", @$val), ")\n";

		if ($clo{'savep'}) {
			print SAVEP "$test_type = " . join(", ", @$val) . "\n";
		} else {
			write_xml_file(@$val);
			write_cxx_file($test_type);
			run_test($test_type) || die "$test_type test failed\n";

			unlink "Clo.hh";
			unlink "test";
			unlink "test.cc";
			unlink "test.clo++";
		}
	}

	close(OUTPUT);
	exit;
}

## end of program ##

sub run_test {
	my $test = shift;
	my @questions = split(/\n/, $tests{$test}->[2]);
	my @answers   = split(/\n/, $tests{$test}->[3]);
	my ($qa, $response);

	if (@questions != @answers) {
		die "questions and answers are different sizes for $test\n";
	}

	# make a Clo.hh file
	print OUTPUT "\t\t===> Generating: Clo.hh\n";
	`$clo test.clo++` and return 0;
	print OUTPUT "\t\t===> Compiling test program\n";
	`nice -18 $CXX -o test test.cc` and return 0;


	for ($qa=0; $qa < @questions; $qa++) {
		print OUTPUT "\t\t===> Running test $questions[$qa]\n";
		chomp($response = `./test $questions[$qa]`);
		print OUTPUT "\t\t===> Results: $response (expected $answers[$qa])\n";
		
		if ($response ne $answers[$qa]) {
			print OUTPUT "bad answer on test $test question $questions[$qa]\n";
			return 0;
		}
	}
	
	return 1;
}

sub parse_data_file {
	my $file = shift;
	my $element = 0;

	open(FILE, "data/$file") || die $!;
	while (<FILE>) {
		if ($_ eq "\n") {
			$element++;
			next;
		}

		$tests{$file}->[$element] .= $_;
	}
	close(FILE);
}

sub write_xml_file {
	my @tests = @_;

	print OUTPUT "\t\t===> Generating: test.clo++\n";
	open(XMLFILE, ">test.clo++") || die $!;

	print XMLFILE qq{<?xml version="1.0"?>\n<clo>\n};

	foreach my $test (@tests) {
		print XMLFILE $tests{$test}->[0];
	}

	print XMLFILE qq{</clo>\n};
	close XMLFILE;
}

sub write_cxx_file {
	my $test = shift;

	print OUTPUT "\t\t===> Generating: test.cc\n";
	open(CXXFILE, ">test.cc") || die $!;
	
	print CXXFILE <<EOT;
#include "Clo.hh"

int main (int argc, char *argv[]) {
	Clo::Parser clo;

	try {
		clo.parse(argc, argv);
	} catch (...) {
		cerr << "error" << endl;
		return 1;
	}

$tests{$test}->[1]

	return 0;
}
EOT

	close(CXXFILE);
}

sub permute_init {
	return [ [[], [sort keys %tests]] ];
}

sub permute_next {
	my $state = shift;
	my $u_permute = shift;
	my ($left, $right, $newleft, $newright);
	my ($val, $item, $i, $join);

	$item = pop(@$state);
	return (undef, undef) unless $item;
	($left, $right) = @$item;

	for ( ; ; ) {
		if (@$right) {
			# fill er up
			foreach $i (0 .. $#{$right}) {
				$newright	= [@$right];
				$newleft 	= [@$left];
				push(@$newleft, splice(@$newright, $i, 1));
				unshift(@$state, [$newleft, $newright]);
			}
		}

		if (@$left) {
			$val = [sort @$left];
			$join = join("/", @$val);
			if (not exists $u_permute->{$join}) {
				$u_permute->{$join} = 1;
				last;
			}
		}

		return (undef, undef) unless @$state;
		($left, $right) = @{pop(@$state)};
	}

	return ($state, $val);
}
