/*
 * Copyright (C) 2000-2001 Peter J Jones (pjones@pmade.org)
 * All Rights Reserved
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Author nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include "Option.hh"

// Constants
const string k_default_bool_argname = "boolean";
const string k_default_int_argname = "interger";
const string k_default_double_argname = "number";
const string k_default_string_argname = "string";

//# method Option #############################################################
/** Class constructor
	
	@author Peter Jones
**/
//#############################################################################
Option::Option() {
	m_name_short = '\0';
	m_type = type_flag;
	m_is_list = false;
	m_is_map = false;
	m_is_mandatory = false;
	m_appended_desc = false;
}
//# method get_name_long ######################################################
/** Get the long name of this option
	
	This method returns the long name for this option. A long
	name is one that is preceded by a double dash '--'. Long
	names can be any character except a equal sign '=' or a
	space ' '

	@return A string containing this options long name
	@author Peter Jones
**/
//#############################################################################
string Option::get_name_long() const {
	string result;

	if (m_name_long.empty())
		result = get_name_short();
	else
		result = m_name_long;

	return result;
}
//# method get_name_short #####################################################
/** Get the short name of this option
	
	This method returns the short name for this option. A short
	name is one that is preceded by a dash '-' and only one
	character.

	@return A chr containing the short name of this option.
	@author Peter Jones
**/
//#############################################################################
char Option::get_name_short() const {
	return m_name_short;
}
//# method get_name_length ####################################################
/** Get the length of the name, including the name of the argument that it takes
	
	This method will calculate the length of the name and take into account
	any argument that it might take. It is mainly used for creating the usage
	string.

	@return the name length
	@author Peter Jones
**/
//#############################################################################
string::size_type Option::get_name_length() {
	string::size_type result = get_name_long().size();

	// account for dashes and short/long names
	if (result == 1)
		result++; // for the dash
	else if (m_name_short != '\0')
		result += 6; // for putting the short name in front and double dash
	else 
		result += 2; // for putting the double dash

	if (m_type != type_flag && m_type != type_count) {
		result += 1 + get_argname().size();
	}

	return result;
}
//# method get_argname ########################################################
/** Get the name of the argument for this option
	
	This method will return the name of the argument that this option
	takes, if it takes an argument.

	@return string containing the argument name
	@author Peter Jones
**/
//#############################################################################
string Option::get_argname() {
	if (m_argname.size()) return m_argname;
	string result;

	switch (m_type) {
		case type_bool:
			result = k_default_bool_argname;
			break;
		case type_int:
			result = k_default_int_argname;
			break;
		case type_double:
			result = k_default_double_argname;
			break;
		case type_string:
			result = k_default_string_argname;
			break;
		default:
			return "";
	}

	if (m_is_map) result.insert(0, "key=");
	m_argname = result;
	return result;
}
//# method get_varname ########################################################
/** Get a valid C++ variable name for this option
	
	This method will generate a vaild C++ variable name
	from this options long name or short name, which ever is set.

	@return A string containing the C++ variable name
	@author Peter Jones
**/
//#############################################################################
string Option::get_varname() {
	if (m_varname.size()) return m_varname;

	if (m_name_long.size())
		m_varname = m_name_long;
	else
		m_varname = m_name_short;

	string::iterator	i, end=m_varname.end();

	for (i=m_varname.begin(); i!=end; i++) {
		if (!isalnum(*i) && *i != '_')
			*i = '_';
	}

	i = m_varname.begin();
	m_varname.insert(0, "v_");

	return m_varname;
}
//# method get_default ########################################################
/** Get the default setting for this option
	
	This method will return the default setting for this option.
	That setting is used when this option is not present in
	the command line.

	@return A string containing the default setting.
	@author Peter Jones
**/
//#############################################################################
string Option::get_default() {
	if (m_default.empty()) {
		switch (m_type) {
			case type_flag:
			case type_bool:
				return "false";
				break;
			case type_int:
			case type_count:
				return "0";
				break;
			case type_double:
				return "0.0";
				break;
			case type_string:
				return "\"\"";
				break;
		}
	} else {
		switch (m_type) {
			case type_flag:
			case type_bool:
				return m_default == "true" ? "true" : "false";
				break;
			case type_count:
			case type_int:
			case type_double:
				return m_default;
				break;
			case type_string:
				string result = m_default;
				StrUtil::quotemeta(result);
				result.insert(result.begin(), '"');
				result.insert(result.end(), '"');
				return result;
				break;
		}
	}
}
//# method get_description ####################################################
/** Get the description string for this option
	
	This method will return a string containing the description
	for this option if one is set.

	@return a string containing the description
	@author Peter Jones
**/
//#############################################################################
string Option::get_description() {
	if (m_description.empty()) m_description = "this option is missing a description";

	if (!m_appended_desc) {
		if ((m_is_list || m_is_map) && !m_is_mandatory)
			m_description += " (may be repeated)";
		else if ((m_is_list || m_is_map) && m_is_mandatory)
			m_description += " (must be given at least once, but may be repeated)";
		else if (m_is_mandatory)
			m_description += " (must be present on command line)";

		m_appended_desc = true;
	}

	return m_description;
}
//# method get_type ###########################################################
/** Get this option's type
	
	This method will return this option's type. The various
	types can be found in the enum in the class declaration.

	@return This option's type
	@author Peter Jones
**/
//#############################################################################
Option::Type Option::get_type() {
	return m_type;
}
//# method is_list ############################################################
/** Find out if this option can be repeated
	
	This method will return a bool flag to tell you wheather
	or not this option can be repeated on the command line more
	then once.  If it is of any type other then flag, the values
	should be kept in a vector.

	@return True if this option can repeat, otherwise False
	@author Peter Jones
**/
//#############################################################################
bool Option::is_list() {
	return m_is_list;
}
//# method is_map #############################################################
/** Find out if this option takes two values, and can be repeated
	
	This method will return a bool flag to tell you wheater or
	not this option can take two values, and also be repeated.
	This cannot be true if the type is type_flag.

	@return True if this option is a map option.
	@author Peter Jones
**/
//#############################################################################
bool Option::is_map() {
	return m_is_map;
}
//# method is_mandatory #######################################################
/** Find out if this option is mandatory or not.
	
	This method will return a bool flag indicating wheater or
	not this option is mandatory. A mandatory option is one
	that must be present on the command line.

	@return True if this option is mandatory
	@author Peter Jones
**/
//#############################################################################
bool Option::is_mandatory() {
	return m_is_mandatory;
}
//# method set_name_long ######################################################
/** Set the long name of this option.
	
	@param name A string containing the long name of this option
	@author Peter Jones
**/
//#############################################################################
void Option::set_name_long(string name) {
	m_name_long = name;
}
//# method set_name_short #####################################################
/** Set the short name for this option
	
	@param name A char containing the short name of this option
	@author Peter Jones
**/
//#############################################################################
void Option::set_name_short(char name) {
	m_name_short = name;
}
//# method set_varname ########################################################
/** Set the name variables that refer to this option
	
	@param name A string containing the new varname
	@author Peter Jones
**/
//#############################################################################
void Option::set_varname(string name) {
	m_varname = name;
}
//# method set_default ########################################################
/** Set the default value for this option.
	
	@param default_value A string containing the value
	@author Peter Jones
**/
//#############################################################################
void Option::set_default(string default_value) {
	m_default = default_value;
}
//# method set_description ####################################################
/** Set the description for this option
	
	@param description A string containing the description
	@author Peter Jones
**/
//#############################################################################
void Option::set_description(string description) {
	m_description = description;
}
//# method set_type ###########################################################
/** Set the type for this option

	@param type The type of option
	@author Peter Jones
**/
//#############################################################################
void Option::set_type(Option::Type type) {
	m_type = type;
}
//# method is_list ############################################################
/** Set the is_list flag

	@param flag True or False
	@author Peter Jones
**/
//#############################################################################
void Option::is_list(bool flag) {
	m_is_list = flag;
}
//# method is_map #############################################################
/** Set the is_map flag
	
	@param flag True or False
	@author Peter Jones
**/
//#############################################################################
void Option::is_map(bool flag) {
	m_is_map = flag;
}
//# method is_mandatory #######################################################
/** Set the is_mandatory flag
	
	@param flag True or False
	@author Peter Jones
**/
//#############################################################################
void Option::is_mandatory(bool flag) {
	m_is_mandatory = flag;
}
