# openlinux.pl
# Networking functions for openlinux

$net_scripts_dir = "/etc/sysconfig/network-scripts";
$network_config = "/etc/sysconfig/network";
$static_route_config = "/etc/sysconfig/static-routes";
$nis_conf = "/etc/nis.conf";

# active_interfaces()
# Returns a list of currently ifconfig'd interfaces
sub active_interfaces
{
local(@rv, @lines, $l);
open(IFC, "ifconfig -a |");
while(<IFC>) {
	s/\r|\n//g;
	if (/^\S+/) { push(@lines, $_); }
	else { $lines[$#lines] .= $_; }
	}
close(IFC);
foreach $l (@lines) {
	local %ifc;
	$l =~ /^([^:\s]+)/; $ifc{'name'} = $1;
	$l =~ /^(\S+)/; $ifc{'fullname'} = $1;
	if ($l =~ /^(\S+):(\d+)/) { $ifc{'virtual'} = $2; }
	if ($l =~ /inet addr:(\S+)/) { $ifc{'address'} = $1; }
	else { next; }
	if ($l =~ /Mask:(\S+)/) { $ifc{'netmask'} = $1; }
	if ($l =~ /Bcast:(\S+)/) { $ifc{'broadcast'} = $1; }
	if ($l =~ /HWaddr (\S+)/) { $ifc{'ether'} = $1; }
	if ($l =~ /MTU:(\d+)/) { $ifc{'mtu'} = $1; }
	$ifc{'up'}++ if ($l =~ /\sUP\s/);
	$ifc{'edit'} = ($ifc{'name'} !~ /^ppp/);
	$ifc{'index'} = scalar(@rv);
	push(@rv, \%ifc);
	}
return @rv;
}

# activate_interface(&details)
# Create or modify an interface
sub activate_interface
{
local $a = $_[0];
local $cmd = "ifconfig $a->{'name'}";
if ($a->{'virtual'} ne "") { $cmd .= ":$a->{'virtual'}"; }
$cmd .= " $a->{'address'}";
if ($a->{'netmask'}) { $cmd .= " netmask $a->{'netmask'}"; }
if ($a->{'broadcast'}) { $cmd .= " broadcast $a->{'broadcast'}"; }
if ($a->{'mtu'}) { $cmd .= " mtu $a->{'mtu'}"; }
if ($a->{'up'}) { $cmd .= " up"; }
else { $cmd .= " down"; }
local $out = `$cmd 2>&1`;
if ($?) { &error($out); }
if ($a->{'ether'}) {
	$out = `ifconfig $a->{'name'} hw ether $a->{'ether'} 2>&1`;
	if ($?) { &error($out); }
	}
}

# deactivate_interface(&details)
# Shutdown some active interface
sub deactivate_interface
{
local $cmd = "ifconfig $a->{'name'}";
if ($a->{'virtual'} ne "") { $cmd .= ":$a->{'virtual'}"; }
$cmd .= " down";
local $out = `$cmd 2>&1`;
if ($?) { &error($out); }
}

# boot_interfaces()
# Returns a list of interfaces brought up at boot time
sub boot_interfaces
{
local(@rv, $f);
opendir(CONF, $net_scripts_dir);
while($f = readdir(CONF)) {
	next if ($f !~ /^ifcfg-(\S+)/);
	local (%conf, $b);
	&read_file("$net_scripts_dir/$f", \%conf);
	$b->{'fullname'} = $conf{'DEVICE'} ? $conf{'DEVICE'} : $1;
	if ($b->{'fullname'} =~ /(\S+):(\d+)/) {
		$b->{'name'} = $1;
		$b->{'virtual'} = $2;
		}
	else { $b->{'name'} = $b->{'fullname'}; }
	$b->{'up'} = ($conf{'ONBOOT'} eq 'yes');
	$b->{'address'} = $conf{'IPADDR'} ? $conf{'IPADDR'} : "Automatic";
	$b->{'netmask'} = $conf{'NETMASK'} ? $conf{'NETMASK'} : "Automatic";
	$b->{'broadcast'} = $conf{'BROADCAST'} ? $conf{'BROADCAST'}
					       : "Automatic";
	$b->{'dhcp'} = $conf{'DYNAMIC'} eq 'dhcp';
	$b->{'edit'} = ($b->{'name'} !~ /^ppp|plip/);
	$b->{'index'} = scalar(@rv);
	push(@rv, $b);
	}
closedir(CONF);
return @rv;
}

# save_interface(&details)
# Create or update a boot-time interface
sub save_interface
{
local(%conf);
local $name = $_[0]->{'virtual'} ne "" ? $_[0]->{'name'}.":".$_[0]->{'virtual'}
				       : $_[0]->{'name'};
&read_file("$net_scripts_dir/ifcfg-$name", \%conf);
$conf{'DEVICE'} = $name;
if ($_[0]->{'dhcp'}) {
	$conf{'DYNAMIC'} = 'dhcp';
	}
else {
	$conf{'IPADDR'} = $_[0]->{'address'};
	delete($conf{'DYNAMIC'});
	}
local($ip1, $ip2, $ip3, $ip4) = split(/\./, $_[0]->{'address'});
$conf{'NETMASK'} = $_[0]->{'netmask'};
local($nm1, $nm2, $nm3, $nm4) = split(/\./, $_[0]->{'netmask'});
$conf{'NETWORK'} = sprintf "%d.%d.%d.%d",
			($ip1 & int($nm1))&0xff,
			($ip2 & int($nm2))&0xff,
			($ip3 & int($nm3))&0xff,
			($ip4 & int($nm4))&0xff;
$conf{'BROADCAST'} = $_[0]->{'broadcast'};
$conf{'ONBOOT'} = $_[0]->{'up'} ? "yes" : "no";
&write_file("$net_scripts_dir/ifcfg-$name", \%conf);
}

# delete_interface(&details)
# Delete a boot-time interface
sub delete_interface
{
local $name = $_[0]->{'virtual'} ne "" ? $_[0]->{'name'}.":".$_[0]->{'virtual'}
				       : $_[0]->{'name'};
unlink("$net_scripts_dir/ifcfg-$name");
}

# iface_type(name)
# Returns a human-readable interface type name
sub iface_type
{
return "PPP" if ($_[0] =~ /^ppp/);
return "SLIP" if ($_[0] =~ /^sl/);
return "PLIP" if ($_[0] =~ /^plip/);
return "Ethernet" if ($_[0] =~ /^eth/);
return "Arcnet" if ($_[0] =~ /^arc/);
return "Token Ring" if ($_[0] =~ /^tr/);
return "Pocket/ATP" if ($_[0] =~ /^atp/);
return "Loopback" if ($_[0] =~ /^lo/);
return "Unknown";
}

# iface_hardware(name)
# Does some interface have an editable hardware address
sub iface_hardware
{
return $_[0] =~ /^eth/;
}

# can_edit(what)
# Can some boot-time interface parameter be edited?
sub can_edit
{
return $_[0] ne "bootp" && $_[0] ne "mtu";
}

# valid_boot_address(address)
# Is some address valid for a bootup interface
sub valid_boot_address
{
return &check_ipaddress($_[0]);
}

# get_dns_config()
# Returns a hashtable containing keys nameserver, domain, search & order
sub get_dns_config
{
local $dns;
open(RESOLV, "/etc/resolv.conf");
while(<RESOLV>) {
	s/\r|\n//g;
	if (/^nameserver\s+(.*)/) {
		push(@{$dns->{'nameserver'}}, split(/\s+/, $1));
		}
	elsif (/^domain\s+(\S+)/) {
		$dns->{'domain'} = [ $1 ];
		}
	elsif (/^search\s+(.*)/) {
		$dns->{'domain'} = [ split(/\s+/, $1) ];
		}
	}
close(RESOLV);
open(SWITCH, "/etc/nsswitch.conf");
while(<SWITCH>) {
	s/\r|\n//g;
	if (/hosts:\s+(.*)/) {
		$dns->{'order'} = $1;
		}
	}
close(SWITCH);
return $dns;
}

# save_dns_config(&config)
# Writes out the resolv.conf and nsswitch.conf files
sub save_dns_config
{
open(RESOLV, "/etc/resolv.conf");
local @resolv = <RESOLV>;
close(RESOLV);
open(RESOLV, ">/etc/resolv.conf");
foreach (@{$_[0]->{'nameserver'}}) {
	print RESOLV "nameserver $_\n";
	}
if ($_[0]->{'domain'}) {
	if ($_[0]->{'domain'}->[1]) {
		print RESOLV "search ",join(" ", @{$_[0]->{'domain'}}),"\n";
		}
	else {
		print RESOLV "domain $_[0]->{'domain'}->[0]\n";
		}
	}
foreach (@resolv) {
	print RESOLV $_ if (!/^\s*(nameserver|domain|search)\s+/);
	}
close(RESOLV);

open(SWITCH, "/etc/nsswitch.conf");
local @switch = <SWITCH>;
close(SWITCH);
open(SWITCH, ">/etc/nsswitch.conf");
foreach (@switch) {
	if (/hosts:\s+/) {
		print SWITCH "hosts:\t$_[0]->{'order'}\n";
		}
	else { print SWITCH $_; }
	}
close(SWITCH);
}

$max_dns_servers = 3;

# order_input(&dns)
# Returns HTML for selecting the name resolution order
sub order_input
{
if ($_[0]->{'order'} =~ /\[/) {
	# Using a complex resolve list
	return "<input name=order size=45 value=\"$_[0]->{'order'}\">\n";
	}
else {
	# Can select by menus
	local @o = split(/\s+/, $_[0]->{'order'});
	@o = map { s/nis\+/nisplus/; s/yp/nis/; $_; } @o;
	local ($rv, $i, $j);
	local @srcs = ( "", "files", "dns", "nis", "nisplus" );
	local @srcn = ( "", "Hosts", "DNS", "NIS", "NIS+" );
	for($i=1; $i<@srcs; $i++) {
		local $ii = $i-1;
		$rv .= "<select name=order_$ii>\n";
		for($j=0; $j<@srcs; $j++) {
			$rv .= sprintf "<option value=\"%s\" %s>%s\n",
					$srcs[$j],
					$o[$ii] eq $srcs[$j] ? "selected" : "",
					$srcn[$j];
			}
		$rv .= "</select>\n";
		}
	return $rv;
	}
}

# parse_order(&dns)
# Parses the form created by order_input()
sub parse_order
{
if (defined($in{'order'})) {
	$in{'order'} =~ /\S/ || &error($text{'dns_eorder'});
	$_[0]->{'order'} = $in{'order'};
	}
else {
	local($i, @order);
	for($i=0; defined($in{"order_$i"}); $i++) {
		push(@order, $in{"order_$i"}) if ($in{"order_$i"});
		}
	$_[0]->{'order'} = join(" ", @order);
	}
}

sub get_hostname
{
return &get_system_hostname();
}

# save_hostname(name)
sub save_hostname
{
local %conf;
system("hostname $_[0] >/dev/null 2>&1");
open(HOST, ">/etc/HOSTNAME");
print HOST $_[0],"\n";
close(HOST);
&read_file("$network_config", \%conf);
$conf{'HOSTNAME'} = $_[0];
&write_file("$network_config", \%conf);
}

# get_domainname()
sub get_domainname
{
local $d = `domainname`;
chop($d);
return $d eq "(none)" ? "" : $d;
}

# save_domainname(domain)
sub save_domainname
{
local %conf;
system("domainname \"$_[0]\" >/dev/null 2>&1");
&read_file("$network_config", \%conf);
if ($_[0]) {
	$conf{'NISDOMAIN'} = $_[0];
	}
else {
	delete($conf{'NISDOMAIN'});
	}
&write_file("$network_config", \%conf);
# XXX need to update nis.conf
}

sub routing_input
{
local (%conf, %ifc, $f, $gateway, $gatewaydev);
&read_file($network_config, \%conf);
opendir(CONF, $net_scripts_dir);
while($f = readdir(CONF)) {
	next if ($f !~ /^ifcfg-(\S+)/);
	&read_file("$net_scripts_dir/$f", \%ifc);
	if (&check_ipaddress($ifc{'GATEWAY'})) {
		$gateway = $ifc{'GATEWAY'};
		$gatewaydev = $ifc{'DEVICE'};
		last;
		}
	}
closedir(CONF);

print "<tr> <td><b>$text{'routes_default'}</b></td> <td>\n";
printf "<input type=radio name=gateway_def value=1 %s> %s\n",
	$gateway ? "" : "checked", $text{'routes_none'};
printf "<input type=radio name=gateway_def value=0 %s> %s\n",
	$gateway ? "checked" : "", $text{'routes_gateway'};
printf "<input name=gateway size=15 value=\"%s\"> %s\n",
	$gateway, $text{'routes_device'};
printf "<input name=gatewaydev size=6 value=\"%s\"></td> </tr>\n",
	$gatewaydev;

# show routing
print "<tr> <td><b>$text{'routes_forward'}</b></td> <td>\n";
printf "<input type=radio name=forward value=1 %s> $text{'yes'}\n",
	$conf{'IPFORWARDING'} =~ /yes|true/i ? "checked" : "";
printf "<input type=radio name=forward value=0 %s> $text{'no'}</td> </tr>\n",
	$conf{'IPFORWARDING'} =~ /yes|true/i ? "" : "checked";
}

sub parse_routing
{
local %conf;
&read_file($network_config, \%conf);
if ($in{'forward'}) { $conf{'IPFORWARDING'} = 'yes'; }
else { delete($conf{'IPFORWARDING'}); }

if (!$in{'gateway_def'}) {
	gethostbyname($in{'gateway'}) ||
		&error(&text('routes_edefault', $in{'gateway'}));
	-r "$net_scripts_dir/ifcfg-$in{'gatewaydev'}" ||
		&error(&text('routes_edevice', $in{'gatewaydev'}));
	}

opendir(CONF, $net_scripts_dir);
while($f = readdir(CONF)) {
	next if ($f !~ /^ifcfg-(\S+)/);
	&read_file("$net_scripts_dir/$f", \%ifc);
	if ($in{'gateway_def'} || $ifc{'DEVICE'} ne $in{'gatewaydev'}) {
		delete($ifc{'GATEWAY'});
		}
	else {
		$ifc{'GATEWAY'} = $in{'gateway'};
		}
	&write_file("$net_scripts_dir/$f", \%ifc);
	}
closedir(CONF);

&write_file($network_config, \%conf);
}

1;

