/*
 * Copyright 2003 Niels Provos <provos@citi.umich.edu>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Niels Provos.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef _POOL_
#define _POOL_

struct pool_entry {
	TAILQ_ENTRY(pool_entry) next;
	void *data;
	size_t size;
};

struct pool {
	TAILQ_HEAD(poolq, pool_entry) entries;
	size_t size;
	int nalloc;
};

struct pool *pool_init(size_t);
void *pool_alloc_size(struct pool *, size_t);

/* 
 * The pool interface cached allocation of fixed sized objects,
 * but it can also be used to allocate larger buffers if necessary.
 */

static __inline void *
pool_alloc(struct pool *pool)
{
	struct pool_entry *entry;

	if ((entry = TAILQ_FIRST(&pool->entries)) == NULL)
		return (pool_alloc_size(pool, 0));

	TAILQ_REMOVE(&pool->entries, entry, next);
	return (entry->data);
}

static __inline void
pool_free(struct pool *pool, void *addr)
{
	struct pool_entry *entry = addr - sizeof(struct pool_entry);

	if (entry->data != addr)
		errx(1, "%s: bad address: %p != %p", __func__,
		    addr, entry->data);

	if (entry->size == pool->size)
		TAILQ_INSERT_HEAD(&pool->entries, entry, next);
	else {
		free(entry);
		pool->nalloc--;
	}
}

#endif /* _POOL_ */
