%  Copyright (C) 2002-2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\subsection{darcs pull}
\begin{code}
module Pull ( pull, merge_with_us_and_pending, save_patches,
              check_unrecorded_conflicts,
            ) where
import System ( ExitCode(..), exitWith )
import Monad ( when, unless, liftM )
import Maybe ( catMaybes, isNothing )

import SignalHandler ( withSignalsBlocked )
import DarcsCommands ( DarcsCommand(..) )
import DarcsArguments ( DarcsFlag( AnyOrder, Test, Verbose, Quiet ),
                        want_external_merge, nocompress, ignoretimes,
                        no_deps, use_external_merge,
                        match_several, fix_filepath,
                        all_gui_interactive,
                        print_dry_run_message_and_exit,
                        any_verbosity, test, dry_run,
                        set_default, summary, working_repo_dir,
                        set_scripts_executable,
                      )
import Repository ( slurp_recorded, slurp_recorded_and_unrecorded,
                    add_to_inventory,
                    write_patch, get_unrecorded, read_repo, am_in_repo,
                    write_pending, read_pending,
                    sync_repo,
                    absolute_dir
                  )
import Patch ( Patch, join_patches, merge, patch2patchinfo,
               unjoin_patches, apply_to_slurpy, list_touched_files,
               invert, list_conflicted_files, eq_patches, null_patch,
               showPatch,
             )
import SelectChanges ( promptChar )
import PatchInfo ( PatchInfo, human_friendly, showPatchInfo )
import SlurpDirectory ( slurp_write_dirty, wait_a_moment )
import RepoPrefs ( defaultrepo, set_defaultrepo, get_preflist )
import Motd (show_motd )
import Pristine ( identifyPristine, write_dirty_Pristine )
import Depends ( get_common_and_uncommon )
import Resolution ( standard_resolution, external_resolution )
import Lock ( withRepoLock )
import SelectChanges ( with_selected_changes )
import DarcsUtils ( putStrLnError, putDocLnError, clarify_errors )
import DarcsURL ( is_relative )
import DarcsUtils ( nubsort )
import Test ( test_slurpy )
import Printer ( Doc, putDocLn, errorDoc, vcat, ($$), text )
#include "impossible.h"
\end{code}
\begin{code}
pull_description :: String
pull_description =
 "Copy and apply patches from another repository to this one."
\end{code}

\options{pull}

\haskell{pull_help}
\begin{code}
pull_help :: String
pull_help =
 "Pull is used to bring changes made in another repo into the current repo\n"++
 "(that is, either the one in the current directory, or the one specified with\n"++
 "the --repodir option). Pull allows you to bring over all or some of the\n"++
 "patches that are in that repo but not in this one. Pull accepts an argument,\n"++
 "which is the URL from which to pull, and when called without an argument,\n"++
 "pull will use the repository from which you have most recently either pushed\n"++
 "or pulled.\n"
\end{code}
\begin{code}
pull :: DarcsCommand
pull = DarcsCommand {command_name = "pull",
                     command_help = pull_help,
                     command_description = pull_description,
                     command_extra_args = 1,
                     command_extra_arg_help = ["[REPOSITORY]"],
                     command_command = pull_cmd,
                     command_prereq = am_in_repo,
                     command_get_arg_possibilities = get_preflist "repos",
                     command_argdefaults = defaultrepo,
                     command_darcsoptions = [match_several,
                                             all_gui_interactive,
                                             use_external_merge,nocompress,
                                             test, dry_run, summary,
                                             any_verbosity,
                                             ignoretimes,no_deps,
                                             set_default,
                                             working_repo_dir,
                                             set_scripts_executable]}
\end{code}
\begin{code}
pull_cmd :: [DarcsFlag] -> [String] -> IO ()

pull_cmd opts [unfixedrepodir] =
  let putInfo = if Quiet `elem` opts then \_ -> return () else putDocLn
      putVerbose = if Verbose `elem` opts then putDocLn else \_ -> return ()
      repodir = if is_relative unfixedrepodir
                then fix_filepath opts unfixedrepodir
                else unfixedrepodir

  in withRepoLock $ do
  -- Test to make sure we aren't trying to pull from the current repo
  cur_absolute_repo_dir <- absolute_dir "."
  req_absolute_repo_dir <- absolute_dir repodir
  when (cur_absolute_repo_dir == req_absolute_repo_dir) $
        fail "Can't pull from current repo!"
  them <- read_repo repodir
  old_default <- defaultrepo []
  set_defaultrepo repodir opts
  when (old_default == [repodir]) $
      putInfo $ text $ "Pulling from \""++repodir++"\"..."
  show_motd opts repodir
  us <- read_repo "."
  case get_common_and_uncommon (us, them) of
    (_, us', them') -> do
     putVerbose $ text "We have the following new (to them) patches:"
               $$ (vcat $ map (human_friendly.fst) $ head us')
     putVerbose $ text "They have the following patches to pull:"
               $$ (vcat $ map (human_friendly.fst) $ head them')
     case them' of
         [[]] -> do putInfo $ text "No remote changes to pull in!"
                    exitWith ExitSuccess
         _ -> return ()
     s <- slurp_recorded "."
     with_selected_changes "pull" opts s (map fromJustPatch $ reverse $ head them') $
      \ (_,to_be_pulled) -> do
      print_dry_run_message_and_exit "pull" opts to_be_pulled
      when (null to_be_pulled) $ do
          putStrLn "You don't want to pull any patches, and that's fine with me!"
          exitWith ExitSuccess
      putVerbose $ text "Getting and merging the following patches:"
      putVerbose $ format_patches_inventory to_be_pulled
      (pc,pw) <- merge_with_us_and_pending opts
                 (map fromJustPatch $ reverse $ head us', to_be_pulled)
      standard_resolved_pw <- standard_resolution pw
      announce_merge_conflicts standard_resolved_pw
      check_unrecorded_conflicts pc
      (recorded, working) <- slurp_recorded_and_unrecorded "."
      pw_resolved <-
          case want_external_merge opts of
          Nothing -> return $ join_patches standard_resolved_pw
          Just c -> do pend <- get_unrecorded (AnyOrder:opts)
                       join_patches `liftM` external_resolution c working
                            (join_patches $ (++catMaybes [pend]) $
                             map fromJustPatch $ reverse $ head us')
                            (join_patches to_be_pulled) pw
      putVerbose $ text "Applying patches to the local directories..."
      case apply_to_slurpy pc recorded of
        Nothing -> do putStrLnError "Error applying patch to recorded!"
                      putStrLnError "The patch was:"
                      putDocLnError $ showPatch pc
                      exitWith $ ExitFailure 1
        Just rec' ->
          case apply_to_slurpy pw_resolved working of
          Nothing -> do putStrLnError "Error applying patch to working dir."
                        putStrLnError "Pristine patch is:"
                        putDocLnError $ showPatch pc
                        putStrLnError "Working patch is:"
                        putDocLnError $ showPatch pw_resolved
                        exitWith $ ExitFailure 1
          Just work' -> do
              when (Test `elem` opts) $
                   do recb <- slurp_recorded "."
                      testproblem <- test_slurpy opts $
                                     fromJust $ apply_to_slurpy pc recb
                      when (testproblem /= ExitSuccess) $ do
                          putStrLnError "Error in test..."
                          exitWith $ ExitFailure 1
              save_patches opts $ unjoin_patches pc
              mp <- get_unrecorded (AnyOrder:opts)
              withSignalsBlocked $ do
                  repairable $ identifyPristine >>= write_dirty_Pristine rec'
                  (sequence $ map (add_to_inventory ".".fromJust.patch2patchinfo)
                           to_be_pulled)
                  unless (isNothing mp && pw_resolved `eq_patches` pc) $
                       write_pending $ join_patches
                           [invert pc, fromMaybePatch mp, pw_resolved]
                  -- so work will be more recent than rec:
                  revertable wait_a_moment
                  revertable $ slurp_write_dirty opts work'
              sync_repo
              putInfo $ text "Finished pulling and applying."
          where fromMaybePatch Nothing = null_patch
                fromMaybePatch (Just p) = p
                repairable x = x `clarify_errors` unlines
                   ["Your repository is now in an inconsistent state.",
                    "This must be fixed by running darcs repair."]
                revertable x = x `clarify_errors` unlines
                  ["This may have left your working directory an inconsistent",
                   "but recoverable state. If you had no un-recorded changes",
                   "by using 'darcs revert' you should be able to make your",
                   "working directory  consistent again."]
pull_cmd _ _ = impossible
format_patches_inventory :: [Patch] -> Doc
format_patches_inventory ps =
    vcat $ map (showPatchInfo.fromJust.patch2patchinfo) ps

fromJustPatch :: (PatchInfo, Maybe Patch) -> Patch
fromJustPatch (pinfo, Nothing)
    = errorDoc $ text "Error reading patch:"
              $$ human_friendly pinfo
fromJustPatch (_, Just p) = p
\end{code}

\begin{options}
--external-merge
\end{options}

You can use an external interactive merge tool to resolve conflicts with the
flag \verb!--external-merge!.  For more details see
subsection~\ref{resolution}.

\begin{options}
--matches, --no-deps, --patches, --tags
\end{options}

The \verb!--patches!, \verb!--matches!, and \verb!--tags! options can be
used to select which patches to pull, as described in
subsection~\ref{selecting}.  darcs will silently pull along any other patches
upon which the selected patches depend.  So \verb!--patches bugfix! means
``pull all the patches with `bugfix' in their name, along with any patches
they require.''  If you really only want the patches with `bugfix' in their
name, you should use the \verb!--no-deps! option, which makes darcs pull in
only the selected patches which have no dependencies (apart from other
selected patches).

\begin{options}
--no-test, --test
\end{options}

If you specify the \verb!--test! option, pull will run the test (if a test
exists) on a scratch copy of the repo contents prior to actually performing
the pull.  If the test fails, the pull will be aborted.

\begin{options}
--verbose
\end{options}

Adding the \verb!--verbose! option causes another section to appear in the
output which also displays a summary of patches that you have and the remote
repo lacks. Thus, the following syntax can be used to show you all the patch
differences between two repos:

\begin{verbatim}
darcs pull --dry-run --verbose
\end{verbatim}

\begin{code}
save_patches :: [DarcsFlag] -> Maybe [Patch] -> IO ()
save_patches _ (Just []) = return ()
save_patches _ Nothing = return ()
save_patches opts (Just (p:ps)) = do write_patch opts p
                                     save_patches opts $ Just ps
\end{code}

\begin{code}
merge_with_us_and_pending :: [DarcsFlag] -> ([Patch],[Patch]) ->
                             IO (Patch, Patch)
merge_with_us_and_pending opts (us,them) =
  case (join_patches us, join_patches them) of
  (usp, themp) ->
      case merge (themp, usp) of
      Nothing -> fail "There was a bug in merging... giving up!"
      Just (themp',_) -> do
         past_pending <- merge_with_pending opts themp'
         return (themp', past_pending)
merge_with_pending :: [DarcsFlag] -> Patch -> IO Patch
merge_with_pending opts p = do
  pend <- get_unrecorded (AnyOrder:opts) -- we don't care if it looks pretty...
  case pend of
    Nothing -> return p
    Just pendp ->
      case merge (p,pendp) of
      Nothing -> fail "Bug in merging with pending..."
      Just (p',_) -> return p'
\end{code}

\begin{code}
announce_merge_conflicts :: [Patch] -> IO ()
announce_merge_conflicts resolved_pw =
    case nubsort $ list_touched_files $ join_patches $ tail resolved_pw of
    [] -> return ()
    cfs -> do putStrLn "We have conflicts in the following files:"
              putStrLn $ unwords cfs

check_unrecorded_conflicts :: Patch -> IO ()
check_unrecorded_conflicts pc =
    do mpend <- read_pending
       case mpend of
         Nothing -> return ()
         Just pend ->
             case merge (pend, pc) of
             Nothing -> impossible
             Just (pend',_) ->
                 case list_conflicted_files pend' of
                 [] -> return ()
                 fs -> do yorn <- promptChar
                                  ("You have conflicting local changes to:\n"
                                   ++ unwords fs++"\nProceed?") "yn"
                          when (yorn /= 'y') $
                               do putStr "Cancelled."
                                  exitWith ExitSuccess
\end{code}
