%  Copyright (C) 2003-2004 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\section{Dependencies}
\begin{code}
module Depends ( get_common_and_uncommon, get_tags_right,
                 get_common_and_uncommon_or_missing,
                 optimize_patchset, deep_optimize_patchset,
                 slightly_optimize_patchset,
                 get_patches_beyond_tag, get_patches_in_tag,
                 is_tag,
                 commute_to_end,
               ) where
import List ( delete, intersect )
import Monad ( liftM, liftM2 )
import Control.Monad.Error (Error(..), MonadError(..))

import Patch ( Patch, getdeps, join_patches, flatten, commute, patch2patchinfo )
import PatchInfo ( PatchInfo, just_name, human_friendly )
import RepoTypes ( PatchSet, PatchSequence )
import Printer ( errorDoc, ($$), text )
#include "impossible.h"
\end{code}

\begin{code}
get_tags_right :: PatchSet -> [PatchInfo]
get_common_and_uncommon :: (PatchSet,PatchSet) ->
                           ([PatchInfo],PatchSet,PatchSet)
get_common_and_uncommon_or_missing :: (PatchSet,PatchSet) ->
                                      Either PatchInfo ([PatchInfo],PatchSet,PatchSet)
\end{code}

\begin{code}
get_common_and_uncommon = 
    either missingPatchError id . get_common_and_uncommon_err

get_common_and_uncommon_or_missing = 
    either (\(MissingPatch x) -> Left x) Right . get_common_and_uncommon_err

get_common_and_uncommon_err :: (PatchSet,PatchSet) ->
                               Either MissingPatch ([PatchInfo],PatchSet,PatchSet)
get_common_and_uncommon_err (ps1,ps2) = gcau (optimize_patchset ps1, ps2)

gcau :: (PatchSet, PatchSet) -> Either MissingPatch ([PatchInfo],PatchSet,PatchSet)
gcau (ps1,ps2) | null ps1 || null ps2 = return ([],[concat ps1],[concat ps2])
gcau (ps1,[[]]) = return ([],[concat ps1],[[]])
gcau ([[]],ps2) = return ([],[[]],[concat ps2])
gcau ([(pi1,_)]:_,[(pi2,_)]:_) | pi1 == pi2 = return ([pi1],[[]],[[]])
gcau (ps1:ps1b:ps1s,ps2:ps2b:ps2s) =
  if (fst $ last ps1) == (fst $ last ps2)
  then gcau_simple (ps1, ps2)
  else if length ps1 > length ps2
       then gcau (ps1:ps1b:ps1s, (ps2++ps2b):ps2s)
       else gcau ((ps1++ps1b):ps1s, ps2:ps2b:ps2s)

gcau (ps1:ps1b:ps1s,[ps2]) =
  if (fst $ last ps1) == (fst $ last ps2)
  then gcau_simple (ps1, ps2)
  else gcau ((ps1++ps1b):ps1s, [ps2])
gcau ([ps1],ps2:ps2b:ps2s) =
  if (fst $ last ps1) == (fst $ last ps2)
  then gcau_simple (ps1, ps2)
  else gcau ([ps1], (ps2++ps2b):ps2s)

gcau ([ps1],[ps2]) = gcau_simple (ps1, ps2)
gcau ([ps1],ps2s) = gcau ([ps1],[concat ps2s])
gcau (ps1s,[ps2]) = gcau ([concat ps1s],[ps2])
gcau _ = bug "Unchecked args possibility in get_common_and_uncommon"

gcau_simple :: ([(PatchInfo, Maybe Patch)], [(PatchInfo, Maybe Patch)]) 
            -> Either MissingPatch ([PatchInfo],PatchSet,PatchSet)
gcau_simple (ps1, ps2) = do
    ex1 <- get_extra (return []) common ps1
    ex2 <- get_extra (return []) common ps2
    return $
        ( map fst $ head $ (optimize_patchset [filter ((`elem` common).fst) ps1]) ++ [[]]
        , [ex1]
        , [ex2]
        )
  where
    common = (map fst ps1) `intersect` (map fst ps2)

newtype MissingPatch = MissingPatch PatchInfo

instance Error MissingPatch where
    -- we don't really need those methods
    noMsg = MissingPatch (error "MissingPatch: bug in get_extra (noMsg)")
    strMsg msg = MissingPatch (error ("MissingPatch: " ++ msg))

get_extra :: Either MissingPatch [Patch]
          -> [PatchInfo]
          -> [(PatchInfo, Maybe Patch)]
          -> Either MissingPatch [(PatchInfo, Maybe Patch)]
get_extra _ _ [] = return []
get_extra skipped common ((pinfo, mp):pps) =
    if pinfo `elem` common && is_tag pinfo
    then case liftM getdeps mp of
         Just ds -> get_extra (liftM2 (:) ep skipped) (ds++delete pinfo common) pps
         Nothing -> get_extra (liftM2 (:) ep skipped) (delete pinfo common) pps
    else if pinfo `elem` common
         then get_extra (liftM2 (:) ep skipped) (delete pinfo common) pps
         else do
            p <- ep
            skpd <- skipped
            case commute (join_patches skpd, p) of
              Just (p', skipped_patch') -> do
                  x <- get_extra (return (flatten skipped_patch')) common pps
                  return ((pinfo, Just p') : x)
              Nothing -> error $ "bug in get_extra.\nMost likely this is "++
                         "caused by a bug that existed in darcs prior\n"++
                         "to version 1.0.1.  Details for dealing with this"++
                         " issue can be found\nat http://darcs"++
                         ".net/DarcsWiki/Issues1.0.1"
    where ep = case mp of
              Just p' -> return p'
              Nothing -> throwError (MissingPatch pinfo)

missingPatchError :: MissingPatch -> a
missingPatchError (MissingPatch pinfo) =
    errorDoc
        ( text "failed to read patch in get_extra:"
          $$ human_friendly pinfo
          $$ text "Perhaps this is a 'partial' repository?" )

get_extra_old :: [Patch]
              -> [PatchInfo]
              -> [(PatchInfo, Maybe Patch)]
              -> [(PatchInfo, Maybe Patch)]
get_extra_old skipped common pps =
    either missingPatchError id (get_extra (return skipped) common pps)
              
\end{code}

\begin{code}
get_patches_beyond_tag :: PatchInfo -> PatchSet -> PatchSet
get_patches_beyond_tag t ([(pinfo,_)]:_) | pinfo == t = [[]]
get_patches_beyond_tag t patchset@(((pinfo,mp):ps):pps) =
    if pinfo == t
    then [get_extra_old [] [t] $ concat patchset]
    else (pinfo,mp) -:- get_patches_beyond_tag t (ps:pps)
get_patches_beyond_tag t ([]:pps) = get_patches_beyond_tag t pps
get_patches_beyond_tag _ [] = [[]]

get_patches_in_tag :: PatchInfo -> PatchSet -> PatchSet
get_patches_in_tag t pps@([(pinfo,_)]:xs)
    | pinfo == t = pps
    | otherwise = get_patches_in_tag t xs

get_patches_in_tag t (((pinfo,_):ps):xs)
    | pinfo /= t = get_patches_in_tag t (ps:xs)

get_patches_in_tag _ ((pa@(_, Just tp):ps):xs) = gpit thepis [pa] (ps:xs)
    where thepis = getdeps tp
          gpit _ sofar [] = [reverse sofar]
          gpit deps sofar ([(tinfo,thisp)]:xs')
              | tinfo `elem` deps = (reverse $ (tinfo,thisp) : sofar) : xs'
              | otherwise = gpit deps sofar xs'
          gpit deps sofar ([]:xs') = gpit deps sofar xs'
          gpit deps sofar (((pinf, Just p):ps'):xs')
              | pinf `elem` deps
                  = let odeps = filter (/=pinf) deps
                        alldeps = if is_tag pinf
                                  then odeps ++ getdeps p
                                  else odeps
                    in gpit alldeps ((pinf, Just p):sofar) (ps':xs')
              | otherwise
                  = gpit deps (commute_by sofar p) (ps':xs')
          gpit _ _ (((pinf, Nothing):_):_)
              = errorDoc $ text "Failure reading patch file"
                        $$ human_friendly pinf

get_patches_in_tag t _ = errorDoc $ text "Couldn't read tag"
                                 $$ human_friendly t

commute_by :: [(PatchInfo, Maybe Patch)] -> Patch
           -> [(PatchInfo, Maybe Patch)]
commute_by [] _ = []
commute_by ((pinf, Just a):xs) p =
    case commute (a,p) of
    Nothing -> bug "Failure commuting patches in commute_by called by gpit!"
    Just (p', a') -> (pinf, Just a') : commute_by xs p'
commute_by ((pinf, Nothing):_) _ =
    errorDoc $ text "Couldn't read patch"
            $$ human_friendly pinf
\end{code}

\begin{code}
is_tag :: PatchInfo -> Bool
is_tag pinfo = take 4 (just_name pinfo) == "TAG "

get_tags_right [] = []
get_tags_right (ps:_) = get_tags_r ps
    where
    get_tags_r [] = []
    get_tags_r ((pinfo,mp):pps)
        | is_tag pinfo = case liftM getdeps mp of
                         Just ds -> pinfo : get_tags_r (drop_tags_r ds pps)
                         Nothing -> pinfo : map fst pps
        | otherwise = pinfo : get_tags_r pps
    drop_tags_r :: [PatchInfo] -> PatchSequence -> PatchSequence
    drop_tags_r [] pps = pps
    drop_tags_r _ [] = []
    drop_tags_r ds ((pinfo,mp):pps)
        | pinfo `elem` ds && is_tag pinfo =
            case liftM getdeps mp of
            Just ds' -> drop_tags_r (ds'++delete pinfo ds) pps
            Nothing -> drop_tags_r (delete pinfo ds) pps
        | pinfo `elem` ds = drop_tags_r (delete pinfo ds) pps
        | otherwise = (pinfo,mp) : drop_tags_r ds pps
\end{code}

\begin{code}
deep_optimize_patchset :: PatchSet -> PatchSet
deep_optimize_patchset pss = optimize_patchset [concat pss]

optimize_patchset :: PatchSet -> PatchSet
optimize_patchset [] = []
optimize_patchset (ps:pss) = opsp ps ++ pss
opsp :: [(PatchInfo,Maybe Patch)] -> PatchSet
opsp [] = []
opsp ((pinfo,mp):pps)
     | is_tag pinfo && get_tags_right [(pinfo,mp):pps] == [pinfo]
         = [(pinfo,mp)] : opsp pps
     | otherwise = (pinfo,mp) -:- opsp pps

(-:-) :: (PatchInfo, Maybe Patch) -> PatchSet -> PatchSet
pp -:- [] = [[pp]]
pp -:- (p:ps) = ((pp:p) : ps)

slightly_optimize_patchset :: PatchSet -> PatchSet
slightly_optimize_patchset [] = []
slightly_optimize_patchset (ps:pss) = sops ps ++ pss
    where sops [] = []
          sops [(pinfo,mp)] = [[(pinfo,mp)]]
          sops ((pinfo,mp):pps) | is_tag pinfo && get_tags_right [(pinfo,mp):pps] == [pinfo]
                                 = [(pinfo,mp)] : [pps]
                             | otherwise = (pinfo,mp) -:- sops pps
\end{code}

\begin{code}
commute_to_end :: [Patch] -> PatchSet -> ([Patch],[Patch])
commute_to_end select from =
   ctt [] (map (fromJust.patch2patchinfo) select) (concat from)
   where
      ctt :: [Patch] -> [PatchInfo] -> PatchSequence -> ([Patch], [Patch])
      ctt skp [] _ = ([],skp)
      ctt skp sel ((pinf, Just p):ps)
         | pinf `elem` sel
            = case cmt_by (skp, p) of
              Nothing -> bug "patches to commute_to_end does not commute (1)"
              Just (p', skp') ->
                 let (ps', skp'') = ctt skp' (delete pinf sel) ps
                 in (p':ps', skp'')
         | otherwise
            = ctt (p:skp) sel ps
      ctt _ _ _ = bug "patches to commute_to_end does not commute (2)"
      cmt_by :: ([Patch], Patch) -> Maybe (Patch, [Patch])
      cmt_by ([], a) = Just (a, [])
      cmt_by (p:ps, a) =
          case commute (p, a) of
          Nothing -> Nothing
          Just (a', p') -> case cmt_by (ps, a') of
                           Nothing -> Nothing
                           Just (a'', ps') -> Just (a'', p':ps')
\end{code}
