/*
 * GQmpeg
 * (C)1998, 1999 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License.
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at you own risk!
 */

#include "gqmpeg.h"

/* these two includes are _only_ used in iconify_button_pressed() */
#include <gdk/gdkx.h>
#include <X11/Xlib.h>

/*
 *-----------------------------------------------------------------------------
 * static vars
 *-----------------------------------------------------------------------------
 */

static gint time_seek = 0;
static gint time_seek_pos = 0;
static gint time_seek_max = 0;

static gint song_seek = 0;
static gint song_seek_pos = -1;

static gint vol_btn_down = 0;
static gint bal_btn_down = 0;

/*
 *-----------------------------------------------------------------------------
 * static funcs
 *-----------------------------------------------------------------------------
 */

static gint time_seeking_callback();
static void time_seeking_start();
static void time_seeking_stop();

static void adjust_volume_by(gint n);
static void adjust_balance_by(gint n);

static void load_preset(gint n);

/*
 *-----------------------------------------------------------------------------
 * main window widget time dependant routine
 *-----------------------------------------------------------------------------
 */

gint main_window_widgets_update()
{
	static gint old_song_seek = 0;

	if (time_seek != 0) return TRUE;

	if (song_seek != 0)
		{
		static int trip_cnt;
		if (song_seek < 8 && song_seek > -8)
			{
			if (song_seek > 0)
				{
				song_seek++;
				}
			else
				{
				song_seek--;
				}
			trip_cnt = 0;
			}
		else
			{
			if (song_seek < 32 && song_seek > -32)
				{
				trip_cnt++;
				if (trip_cnt > 2) trip_cnt = 0;
				if (song_seek > 0)
					{
					song_seek++;
					}
				else
					{
					song_seek--;
					}
				}
			else
				{
				trip_cnt = 0;
				}
			if (trip_cnt == 0)
				{
				gint p;
				if (song_seek > 0)
					{
					p = playlist_get_next(song_seek_pos);
					}
				else
					{
					p = playlist_get_prev(song_seek_pos);
					}
				if (p >= 0)
					{
					song_seek_pos = p;
					display_draw_song_number(song_seek_pos, FALSE);
					display_draw_title(playlist_get_title(song_seek_pos), TRUE);
					}
				}
			}
		old_song_seek = song_seek;
		}
	else if (old_song_seek != song_seek)
		{
		old_song_seek = song_seek;
		if (song_seek_pos != current_song_get_number())
                        {
                        display_draw_song_number(current_song_get_number(), FALSE);
			display_draw_title(current_song_get_title(), TRUE);
			}
		}

	if (vol_btn_down !=0)
		{
		if (vol_btn_down == 1)
			adjust_volume_by(3);
		else
			adjust_volume_by(-3);
		}

	if (bal_btn_down !=0)
		{
		if (bal_btn_down == 1)
			adjust_balance_by(-3);
		else
			adjust_balance_by(3);
		}

	/* title, author, and misc fonts */
	update_font(skin->title);
	update_font(skin->artist);
	update_font(skin->album);
	update_font(skin->genre);

	return FALSE;
}

/*
 *-----------------------------------------------------------------------------
 * main play function button callbacks
 *-----------------------------------------------------------------------------
 */

void btn_play_pressed()
{
	if (status == STATUS_PLAY)
		{
		playback_pause(current_song_get_data());
		}
	else if (status == STATUS_PAUSE)
		{
		playback_continue(current_song_get_data());
		}
	else
		{
		playback_start(current_song_get_data(), 0);
		}
}

void btn_stop_pressed()
{
	if (status == STATUS_STOP)
		{
		/* already stopped, now reset to top of playlist */
		if (current_song_is_in_playlist() && current_song_get_number() != playlist_get_first())
			{
			current_song_set(playlist_get_first(), NULL);
			}
		return;
		}
	playback_stop(current_song_get_data());
}

void btn_pause_pressed()
{
	if (status == STATUS_STOP) return;
	playback_pause(current_song_get_data());
}

/*
 *-----------------------------------------------------------------------------
 * song positioning button callbacks
 *-----------------------------------------------------------------------------
 */

void btn_next_pressed()
{
	song_seek = 0;
	if (song_seek_pos == -1) return;
	current_song_set(song_seek_pos, NULL);
}

void btn_prev_pressed()
{
	btn_next_pressed();
}

void btn_next_down()
{
	gint p;
	if (playlist_get_count() > 0)
		{
		song_seek = 1;
		}
	else
		{
		song_seek = 0;
		song_seek_pos = -1;
		return;
		}
	p = playlist_get_next(current_song_get_number());
	if (p >= 0)
		{
		song_seek_pos = p;
		display_draw_song_number(song_seek_pos, FALSE);
		display_draw_title(playlist_get_title(song_seek_pos), TRUE);
		}
}

void btn_next_up()
{
	song_seek = 0;
}

void btn_prev_down()
{
	if (playlist_get_count() > 0)
		{
		song_seek = -1;
		}
	else
		{
		song_seek = 0;
		song_seek_pos = -1;
		return;
		}
	song_seek_pos = current_song_get_number();
	if (!((status == STATUS_PLAY || status == STATUS_PAUSE) && prev_includes_current && seconds > 4))
		{
		gint p;
		p = playlist_get_prev(current_song_get_number());
		if (p >= 0)
			{
			song_seek_pos = p;
			display_draw_song_number(song_seek_pos, FALSE);
			display_draw_title(playlist_get_title(song_seek_pos), TRUE);
			}
		}
}

void btn_prev_up()
{
	song_seek = 0;
}

/*
 *-----------------------------------------------------------------------------
 * time positioning slider callbacks
 *-----------------------------------------------------------------------------
 */

void position_slider_moved(float pos)
{
	gint secs;
	gint frms;
	if (seconds_remaining + seconds == 0 || status == STATUS_STOP) return;
	secs = (float)(seconds_remaining + seconds) * pos;
	frms = (float)(frames_remaining + frames) * pos;
	display_draw_frame_count(frms, FALSE);
	display_draw_time(secs, seconds_remaining + seconds - secs, TRUE);
}

void position_slider_released(float pos)
{
	gint secs;
	if (seconds_remaining + seconds == 0 || status == STATUS_STOP) return;
	secs = (float)pos * (seconds_remaining + seconds);

	playback_seek(current_song_get_data(), secs);
}

/*
 *-----------------------------------------------------------------------------
 * time positioning button callbacks
 *-----------------------------------------------------------------------------
 */

static gint time_seeking_callback()
{
	gint fp;
	if (time_seek == 0) return FALSE;

	/* fast forwarding */
	if (time_seek > 0)
		{
		if (time_seek < 10)
			{
			time_seek_pos += 1;
			time_seek++;
			}
		else
			{
			time_seek_pos += 5;
			}
		if (time_seek_pos > time_seek_max - 2) time_seek_pos = 0;
		}
	/* rewinding */
	if (time_seek < 0)
		{
		if (time_seek > -10)
			{
			time_seek_pos -= 1;
			time_seek--;
			}
		else
			{
			time_seek_pos -= 5;
			}
		if (time_seek_pos < 0) time_seek_pos = time_seek_max - 2;
		}
	display_draw_time(time_seek_pos, time_seek_max - time_seek_pos, FALSE);
	draw_slider(skin->position, (float)time_seek_pos / time_seek_max,
		FALSE, TRUE, FALSE, FALSE, FALSE);
	fp = (float) time_seek_pos / time_seek_max * (frames + frames_remaining);
	display_draw_frame_count(fp, TRUE);
	return TRUE;
}

static void time_seeking_start()
{
	time_seek_max = seconds + seconds_remaining;
	if (time_seek_max == 0) time_seek_max = 1; /* avoid div by zero */
	time_seek_pos = seconds;
	gtk_timeout_add(200, (GtkFunction)time_seeking_callback, NULL);
}

static void time_seeking_stop()
{
	if (time_seek == 0) return;
	time_seek = 0;
	playback_seek(current_song_get_data(), time_seek_pos);
}

void btn_rw_pressed()
{
	if (time_seek != 0) return;
	if (status != STATUS_PLAY && status != STATUS_PAUSE) return;
	time_seek = -1;
	time_seeking_start();
}

void btn_ff_pressed()
{
	if (time_seek != 0) return;
	if (status != STATUS_PLAY && status != STATUS_PAUSE) return;
	time_seek = 1;
	time_seeking_start();
}

void btn_rw_down()
{
	btn_rw_pressed();
}

void btn_rw_up()
{
	time_seeking_stop();
}

void btn_ff_down()
{
	btn_ff_pressed();
}

void btn_ff_up()
{
	time_seeking_stop();
}

/*
 *-----------------------------------------------------------------------------
 * volume callbacks
 *-----------------------------------------------------------------------------
 */

static void adjust_volume_by(gint n)
{
	gint vol = get_volume();
	vol += n;
	if (vol < 0) vol = 0;
	if (vol > 100) vol = 100;
	set_volume(vol);
	display_draw_volume(TRUE);
}

void volume_slider_moved(float pos)
{
	set_volume((gint)(pos * 100));
}

void btn_volume_up_pressed()
{
	vol_btn_down = 1;
	adjust_volume_by(3);
}

void btn_volume_down_pressed()
{
	vol_btn_down = 2;
	adjust_volume_by(-3);
}

void btn_volume_released()
{
	vol_btn_down = 0;
}

/*
 *-----------------------------------------------------------------------------
 * balance callbacks
 *-----------------------------------------------------------------------------
 */

static void adjust_balance_by(gint n)
{
	gint bal = get_balance();
	bal += n;
	if (bal < 0) bal = 0;
	if (bal > 100) bal = 100;
	if (bal > 50 - abs(n) && bal < 50 + abs(n)) bal = 50; /* basic balance auto-centering */
	set_balance(bal);
	display_draw_balance(TRUE);
}

void balance_slider_moved(float pos)
{
	gint p = pos * 100;
	if (p > 44 && p < 56) p = 50; /* to help balance 'lock' on center */
	set_balance(p);
}

void btn_balance_left_pressed()
{
	bal_btn_down = 1;
	adjust_balance_by(-3);
}

void btn_balance_right_pressed()
{
	bal_btn_down = 2;
	adjust_balance_by(3);
}

void btn_balance_released()
{
	bal_btn_down = 0;
}

/*
 *-----------------------------------------------------------------------------
 * misc callbacks
 *-----------------------------------------------------------------------------
 */

void btn_shuffle_pressed()
{
	if (shuffle_mode)
		{
		shuffle_mode = FALSE;
		shuffle_list_destroy();
		}
	else
		{
		shuffle_mode = TRUE;
		shuffle_list_create(TRUE);
		}
	display_draw_shuffle(TRUE);
}

void btn_repeat_pressed()
{
	if (repeat_mode)
		repeat_mode = FALSE;
	else
		repeat_mode = TRUE;
	display_draw_repeat(TRUE);
}

void time_toggle_pressed()
{
	show_remaining_time = !show_remaining_time;
	if (status == STATUS_PLAY || status == STATUS_PAUSE)
		{
		display_draw_time(seconds, seconds_remaining, TRUE);
		}
}

void btn_iconify_pressed()
{
	Window xwindow;

	if (!mainwindow->window) return;

	xwindow = GDK_WINDOW_XWINDOW(mainwindow->window);
	XIconifyWindow (GDK_DISPLAY (), xwindow, DefaultScreen (GDK_DISPLAY ()));
}

void btn_mixer_pressed()
{
	if (mixer_command)
		{
		gchar *command = g_strconcat(mixer_command, " &", NULL);
		system(command);
		g_free(command);
		}
}

void current_song_info_pressed()
{
	view_song_info(current_song_get_path());
}

/*
 *-----------------------------------------------------------------------------
 * load callbacks
 *-----------------------------------------------------------------------------
 */

void btn_load_playlist_pressed()
{
	gchar *buf = g_strconcat(homedir(), "/", GQMPEG_RC_DIR_PLAYLIST, NULL);
	playlist_dialog_load(buf);
	g_free(buf);
}

void btn_load_file_pressed()
{
	playlist_dialog_load_file(homedir(), FALSE);
}

void btn_load_custom_pressed()
{
	playlist_dialog_add_custom_type(FALSE);
}

/*
 *-----------------------------------------------------------------------------
 * preset callbacks
 *-----------------------------------------------------------------------------
 */

static void load_preset(gint n)
{
	if (!preset_file[n]) return;
        if (playlist_load_from_file(preset_file[n], FALSE, TRUE) == TRUE)
                {
                if (play_presets) playback_start(current_song_get_data(), 0);
                }
}

void preset_1_pressed()
{
	load_preset(0);
}

void preset_2_pressed()
{
	load_preset(1);
}

void preset_3_pressed()
{
	load_preset(2);
}

void preset_4_pressed()
{
	load_preset(3);
}

void preset_5_pressed()
{
	load_preset(4);
}

void preset_6_pressed()
{
	load_preset(5);
}

void preset_7_pressed()
{
	load_preset(6);
}

void preset_8_pressed()
{
	load_preset(7);
}

void preset_9_pressed()
{
	load_preset(8);
}

void preset_10_pressed()
{
	load_preset(9);
}

/*
 *-----------------------------------------------------------------------------
 * 
 *-----------------------------------------------------------------------------
 */


