/*

Copyright (C) 2000 - 2007 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#ifdef LINUX
#define __FAVOR_BSD
#endif

#include <gtk/gtk.h>
#include <nd.h>
#include <nd_dialog.h>
#include <nd_gui.h>
#include <nd_packet.h>
#include <nd_trace.h>
#include <nd_tp.h>
#include <nd_prefs.h>
#include <nd_recent.h>
#include <nd_protocol_inst.h>
#include <nd_trace_registry.h>
#include <callbacks.h>
#include <support.h>


typedef struct nd_trace_save_data
{
  LND_Trace          *trace;
  char               *filename;
  ND_DialogCallback   callback_finished;
  void               *user_data;

} ND_TraceSaveData;




static void
trace_save_no_cb                    (void *user_data)
{
  LND_Trace *trace = (LND_Trace *) user_data;

  nd_trace_registry_remove(trace);
  nd_trace_free(trace);
}

static void
trace_save_yes_cb                   (void *user_data)
{
  LND_Trace *trace = (LND_Trace *) user_data;

  nd_trace_save_as_dialog(trace, trace_save_no_cb, trace);
}


/* Callback that gets called for each registered protocol
 * when a new trace is created. It hooks each protocol's gui
 * into the trace and initializes any protocol's state information,
 * given that a protocol is stateful.
 */
static void
trace_proto_cb(LND_Protocol *proto, void *user_data)
{
  LND_Trace *trace = (LND_Trace *) user_data;

  nd_trace_add_proto_tab(trace, proto, 0);
}

LND_Trace *
nd_trace_new(const char *filename)
{
  LND_Trace *trace;
  ND_Trace  *trace_gui;

  D_ENTER;

  trace = libnd_trace_new(filename);
  if (!trace)
    D_RETURN_(NULL);

  trace->iterator_mode = LND_PACKET_IT_SEL_RW;

  if (! (trace_gui = g_new0(ND_Trace, 1)))
    {
      libnd_trace_free(trace);
      D_RETURN_(NULL);
    }

  libnd_reg_set_data(trace->registry, "trace_gui", trace_gui);
  trace_gui->sel_handler = -1;
  
  /* Create this trace's GUI content: */
  nd_gui_trace_new_tab(trace);
  libnd_proto_registry_foreach_proto(trace_proto_cb, trace);
  gtk_notebook_set_page(nd_trace_get_notebook(trace), 0);
  nd_gui_list_set_incomplete_column_visible(trace, FALSE);

  if (filename)
    {
      /* libnd_tpm_load_packets() fires a LND_TRACE_RELOAD event,
       * in which out handler automatically updates the GUI packet list.
       */
      libnd_tpm_load_packets(trace->tpm);
      nd_gui_statusbar_set(_("File loaded."));
      
      /* Update "recently-used" menu entries: */
      nd_recent_add_file(filename);      
    }

  D_RETURN_(trace);
}


void 
nd_trace_close(LND_Trace *trace)
{
  D_ENTER;

  if (!trace)
    D_RETURN;

  if (trace->dirty)
    {
      nd_dialog_ync(_("Trace modified"),
		    _("The trace file has changed, do yo want\n"
		      "to save changes before closing it?"),
		    trace_save_yes_cb,
		    trace_save_no_cb,
		    NULL,
		    trace);
    }
  else
    {
      nd_trace_registry_remove(trace);
      nd_trace_free(trace);
    }

  D_RETURN;
}


void    
nd_trace_free(LND_Trace *trace)
{
  ND_Trace *trace_gui = nd_trace_get(trace);

  D_ENTER;

  if (trace_gui)
    {
      libnd_reg_del_data(trace->registry, "trace_gui");
      nd_gui_del_monowidth_widget(trace_gui->list);
      
      /* Clean up the gui memory -- needs to be
	 done before cleaning up packets */
      gtk_widget_unref(GTK_WIDGET(trace_gui->tab));

      g_free(trace_gui->cur_pi);
      g_free(trace_gui->cur_pi_sel);
      g_free(trace_gui);
    }
  
  libnd_trace_free(trace);

  D_RETURN;
}


static void
trace_ob_cleanness(LND_Trace *trace)
{
  D_ENTER;
  nd_gui_set_trace_status(trace);
  D_RETURN;
}

static void
trace_ob_reload(LND_Trace *trace)
{
  D_ENTER;
  nd_trace_set_current_packet(trace, NULL);
  nd_gui_list_update(trace);
  nd_gui_num_packets_set();
  nd_gui_update_view_indicator();
  D_RETURN;
}

static void
trace_ob_jumped(LND_Trace *trace)
{
  D_ENTER;
  libnd_tpm_load_packets(trace->tpm);
  nd_trace_set_current_packet(trace, NULL);
  nd_gui_list_update(trace);
  nd_gui_num_packets_set();
  nd_gui_update_view_indicator();
  D_RETURN;
}

static void
trace_ob_clear(LND_Trace *trace)
{
  D_ENTER;
  nd_trace_set_current_packet(trace, NULL);
  nd_gui_list_clear(trace);
  nd_gui_num_packets_set();
  D_RETURN;
}

static void
trace_ob_it_area_set(LND_Trace *trace)
{
  D_ENTER;
  nd_gui_update_area_indicator();
  D_RETURN;
  TOUCH(trace);
}

void
nd_trace_init(void)
{
  LND_TraceObserver *ob;

  D_ENTER;

  if (! (ob = libnd_trace_observer_new()))
    {
      D(("Out of memory.\n"));
      D_RETURN;
    }

  ob->trace_modified      = trace_ob_cleanness;
  ob->trace_cleaned       = trace_ob_cleanness;
  ob->trace_reload        = trace_ob_reload;
  ob->trace_jumped        = trace_ob_jumped;
  ob->trace_clear         = trace_ob_clear;
  ob->trace_it_area_set   = trace_ob_it_area_set;

  libnd_trace_add_observer(ob);
  D_RETURN;
}


void
nd_trace_add_proto_tab(LND_Trace *trace,
		       LND_Protocol *proto,
		       guint nesting)
{
  GtkNotebook   *notebook;
  GtkWidget     *scrolledwin, *viewport;
  const char    *pi_string;
  LND_ProtoInfo *pinf;
  ND_ProtoInfo  *pinf_gui;
  ND_Protocol   *proto_gui;

  if (!trace || !proto)
    return;

  proto_gui = nd_proto_get(proto);

  if (!proto_gui)
    {
      D(("No GUI plugin for %s found.\n", proto->name));
      return;
    }

  D(("Initializing protocol %s, nesting %i, for trace %s\n",
     proto->name, nesting, trace->filename));
  
  notebook = nd_trace_get_notebook(trace);
  D_ASSERT_PTR(notebook);

  pi_string = libnd_proto_inst_to_string(proto, nesting);

  if (gtk_object_get_data(GTK_OBJECT(notebook), pi_string))
    return;

  /* Create a scrolled window -- everything else is placed
     into it: */

  scrolledwin = gtk_scrolled_window_new (NULL, NULL);
  gtk_widget_ref(scrolledwin);

  /* From the protocol notebook of this trace, each tab can
     be retrieved by a protocol instance: */

  gtk_object_set_data_full (GTK_OBJECT (notebook), pi_string, scrolledwin,
			    (GtkDestroyNotify) gtk_widget_unref);

  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwin),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

  /* Create the protocol's content viewport: */
  
  viewport = gtk_viewport_new (NULL, NULL);
  gtk_widget_ref (viewport);
  gtk_object_set_data_full (GTK_OBJECT (scrolledwin), "viewport", viewport,
			    (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (viewport);
  gtk_viewport_set_shadow_type (GTK_VIEWPORT (viewport), GTK_SHADOW_NONE);
  gtk_container_add(GTK_CONTAINER(scrolledwin), viewport);

  /* And now create and hook in the protocol's GUI, as it desires: */

  pinf = nd_proto_info_new(proto, nesting);
  pinf_gui = nd_proto_info_get(pinf);
  pinf_gui->proto_tab = scrolledwin;
  pinf_gui->proto_gui = proto_gui->create_gui(trace, pinf);
  pinf_gui->proto_label = gtk_label_new(proto->name);

  gtk_container_add(GTK_CONTAINER(viewport), pinf_gui->proto_gui);
  gtk_widget_show(pinf_gui->proto_gui);
  gtk_widget_ref(pinf_gui->proto_gui);
  gtk_object_set_data_full (GTK_OBJECT (scrolledwin), "content", pinf_gui->proto_gui,
			    (GtkDestroyNotify) gtk_widget_unref);
  
  gtk_notebook_append_page(notebook, scrolledwin,
			   pinf_gui->proto_label);

  /* Via the string version of the proto info, every protocol info
     becomes available from the notebook widget */
  gtk_object_set_data(GTK_OBJECT(notebook), pi_string, pinf);
  
  /* From each protocol's tab in the GUI, information in a ND_ProtoInfo
   * structure is made available: */

  gtk_object_set_data(GTK_OBJECT(scrolledwin), "pinf", pinf);

  if (proto->is_stateful)
    proto->init_state(trace);
}


LND_ProtoInst *
nd_trace_get_current_proto_inst(const LND_Trace *trace)
{
  ND_Trace *trace_gui = nd_trace_get(trace);

  if (!trace_gui)
    return NULL;

  return trace_gui->cur_pi;
}


guchar *
nd_trace_get_current_proto_header(const LND_Trace *trace)
{
  ND_Trace *trace_gui = nd_trace_get(trace);

  if (!trace_gui)
    return NULL;
  
  return trace_gui->cur_header;
}


void
nd_trace_set_current_proto_selection(LND_Trace *trace,
				     const LND_ProtoInst *pi)
{
  ND_Trace *trace_gui = nd_trace_get(trace);

  if (!trace_gui || !pi || !pi->proto)
    return;

  if (!trace_gui->cur_pi_sel)
    {
      trace_gui->cur_pi_sel = g_new0(LND_ProtoInst, 1);
      
      if (!trace_gui->cur_pi_sel)
	return;
    }

  D(("Setting current proto selection to %s/%i\n",
     pi->proto->name, pi->nesting));
  *trace_gui->cur_pi_sel = *pi;
}


void
nd_trace_set_current_proto(LND_Trace *trace,
			   const LND_ProtoInst *pi,
			   gboolean gui_update)
{
  ND_Trace    *trace_gui = nd_trace_get(trace);
  GtkNotebook *notebook;
  LND_Packet *packet = NULL;
  int num;

  if (! trace_gui || ! trace_gui->cur_packet || ! pi ||
      ! (packet = trace_gui->cur_packet) ||
      ! libnd_packet_has_proto_nested(packet, pi->proto, pi->nesting))
    {
      if (gui_update)
	nd_gui_statusbar_clear(NULL);
      return;
    }
  
  D(("Setting current proto to %s/%i\n", pi->proto->name, pi->nesting));
  
  if (!trace_gui->cur_pi)
    {
      trace_gui->cur_pi = g_new0(LND_ProtoInst, 1);      
      if (!trace_gui->cur_pi)
	return;
    }
  
  *trace_gui->cur_pi = *pi;
  
  if (packet)
    {
      trace_gui->cur_header = libnd_packet_get_data(packet, pi->proto, pi->nesting);
      D(("NEW current header: %p\n", trace_gui->cur_header));
    }
  
  if (gui_update)
    {
      LND_Protocol *pcap_proto = libnd_proto_registry_find(LND_PROTO_LAYER_LINK, 0xa1b2c3d4);
      LND_ProtoInfo *pinf;
      ND_ProtoInfo  *pinf_gui;
      char buf[512];

      notebook = nd_trace_get_notebook(trace);
      pinf = nd_trace_get_proto_info(trace, pi->proto, pi->nesting);
      pinf_gui = nd_proto_info_get(pinf);

      D_ASSERT_PTR(pinf_gui);
      if (!pinf_gui)
	return;

      num = gtk_notebook_page_num(notebook, pinf_gui->proto_tab);
      
      gtk_signal_handler_block_by_data(GTK_OBJECT(notebook), notebook);
      if (num != gtk_notebook_get_current_page(notebook))
	gtk_notebook_set_page(notebook, num);
      gtk_signal_handler_unblock_by_data(GTK_OBJECT(notebook), notebook);

      /* Report the protocol's size in bytes in the statusbar if feasible. */
      if (! pi->proto || (pcap_proto && pi->proto == pcap_proto))
	nd_gui_statusbar_clear(NULL);
      else
	{
	  /* Find out the index of the given protocol instance, and use it
	   * to get the next one. We compute the difference in data start pointers
	   * to obtain the size of the displayed data chunk.
	   */
	  int proto_index = libnd_packet_get_proto_index(packet, pi);
	  LND_ProtoData *pd = libnd_packet_get_nth_proto(packet, proto_index + 1);
	  guchar *data_end = pd ? pd->data : libnd_packet_get_end(packet);
	  
	  g_snprintf(buf, 512, _("%i %s bytes shown."),
		     data_end - libnd_packet_get_data(packet, pi->proto, pi->nesting),
		     pi->proto->name);
	  
	  nd_gui_statusbar_set_noclear(buf);
	}
    }
}


void           
nd_trace_set_current_packet(LND_Trace *trace, LND_Packet *packet)
{    
  ND_Trace *trace_gui = nd_trace_get(trace);

  D_ENTER;

  if (!trace_gui)
    D_RETURN;

  if (packet && trace != libnd_packet_get_trace(packet))
    D_RETURN;
  
  D(("Current packet now %p\n", packet));
  trace_gui->cur_packet = packet;
  
  nd_trace_set_current_proto(trace, trace_gui->cur_pi, FALSE);
  nd_packet_set_gui(packet);

  D_RETURN;
}


static void
trace_load_dialog_cb(const char *filename,
		     void *user_data)
{
  if (libnd_misc_is_tcpdump_file(filename))
    {
      LND_Trace *trace;
      char *dir;
      nd_dialog_filesel_close();
      
      dir = g_dirname(filename);      
      libnd_prefs_set_str_item(ND_DOM_NETDUDE, "load_dir", dir);
      g_free(dir);
      trace = nd_trace_new(filename);
      D_ASSERT_PTR(trace);
      nd_trace_registry_add(trace);
    }
  else
    {
      nd_dialog_message(_("Error loading file."),
			_("This file does not seem\n"
			  "to be a tcpdump tracefile."),
			TRUE);
    }

  return;
  TOUCH(user_data);
}


void           
nd_trace_load_dialog(void)
{
  char *load_dir = NULL;

  if (libnd_prefs_get_str_item(ND_DOM_NETDUDE, "load_dir", &load_dir))
    {
      load_dir = libnd_misc_add_slash(g_strdup(load_dir));
    }

  nd_dialog_filesel(_("Load Tcpdump Tracefile"),
		    load_dir,
		    trace_load_dialog_cb, NULL);

  g_free(load_dir);		    
}


static void
trace_save_overwrite_yes_cb(ND_TraceSaveData *data)
{
  char *dirname;

  nd_dialog_filesel_close();

  dirname = g_dirname(data->filename);
  libnd_prefs_set_str_item(ND_DOM_NETDUDE, "save_dir", dirname);

  if (! nd_trace_save_as(data->trace, data->filename))
    return;

  if (data->callback_finished)
    data->callback_finished(data->user_data);

  g_free(dirname);
  g_free(data->filename);
  g_free(data);
}


static void
trace_save_as_dialog_cb(const char *filename,
			void *user_data)
{
  ND_TraceSaveData *data;

  data = (ND_TraceSaveData *) user_data;
  data->filename = g_strdup(filename);

  if (libnd_misc_exists(filename))
    {
      nd_dialog_generic(ND_DIALOG_QUESTION,
			_("File exists."),
			_("The file already exists.\n"
			  "Do you want to overwrite it?"),
			TRUE,
			NULL,
			data,
			2,
			_("Yes"), (ND_DialogCallback) trace_save_overwrite_yes_cb,
			_("No"), NULL);
	  
      return;
    }

  trace_save_overwrite_yes_cb(data);
}


static gboolean
trace_save_possible(const LND_Trace *trace)
{
  if (! libnd_trace_initialized(trace))
    {
      /* FIXME this is ugly -- libnetdude should just tell us this. */
      nd_dialog_message(_("Error saving file."),
			_("You cannot save an empty new trace since the file format\n"
			  "requires information about the type of packets stored\n"
			  "in the trace. If you really want to save an empty trace,\n"
			  "please paste a few packets to provide this information\n"
			  "and then delete them before saving the file.\n"), TRUE);
      return FALSE;
    }

  return TRUE;
}


void           
nd_trace_save_as_dialog(LND_Trace *trace,
			ND_DialogCallback callback_finished,
			void *user_data)
{
  ND_TraceSaveData *data;
  char filename[MAXPATHLEN];
  char *save_dir = NULL;
  const char *trace_filename_only;

  if (!trace)
    return;

  if (! trace_save_possible(trace))
    return;

  data = g_new0(ND_TraceSaveData, 1);
  data->trace = trace;
  data->callback_finished = callback_finished;
  data->user_data = user_data;

  filename[0] = 0;

  if (trace->unnamed)
    trace_filename_only = trace->unnamed;
  else
    trace_filename_only = g_basename(trace->filename);

  if (libnd_prefs_get_str_item(ND_DOM_NETDUDE, "save_dir", &save_dir))
    g_snprintf(filename, MAXPATHLEN, "%s/%s", save_dir, trace_filename_only);
  else
    g_snprintf(filename, MAXPATHLEN, "%s", trace_filename_only);

  nd_dialog_filesel(_("Save Tcpdump Tracefile"),
		    filename,
		    trace_save_as_dialog_cb,
		    data); 
}


gboolean
nd_trace_save(LND_Trace *trace)
{
  char msg[MAXPATHLEN];

  D_ENTER;

  if (!trace)
    D_RETURN_(FALSE);

  if (! trace_save_possible(trace))
    D_RETURN_(FALSE);

  /* Live captures don't yet have a filename ... */
  if (!trace->filename)
    {
      nd_trace_save_as_dialog(trace, NULL, NULL);
      D_RETURN_(FALSE);
    }

  nd_gui_statusbar_set(_("Saving file..."));

  if (! libnd_trace_save(trace))
    {      
      g_snprintf(msg, MAXPATHLEN,
		 _("The file could not be saved -- the error message is\n'%s'."),
		 strerror(errno));
      nd_dialog_message(_("Error saving file."), msg, TRUE);
      D_RETURN_(FALSE);
    }
  
  nd_gui_statusbar_set(_("File saved."));
  nd_recent_add_file(trace->filename);
  libnd_trace_set_dirty(trace, FALSE);
  nd_gui_set_trace_status(trace);

  D_RETURN_(TRUE);
}


gboolean
nd_trace_save_as(LND_Trace *trace, const char *filename)
{
  D_ENTER;

  if (!trace || !filename || !*filename)
    D_RETURN_(FALSE);

  g_free(trace->filename);
  trace->filename = g_strdup(filename);

  if (nd_trace_save(trace))
    {
      nd_gui_set_windowtitle(libnd_trace_get_name(trace));
      nd_gui_trace_set_name(trace);

      D_RETURN_(TRUE);
    }

  D_RETURN_(FALSE);
}


GtkNotebook   *
nd_trace_get_notebook(const LND_Trace *trace)
{
  ND_Trace *trace_gui = nd_trace_get(trace);
  GtkNotebook *nb;

  if (!trace_gui || !trace_gui->tab)
    return NULL;

  ND_GTK_GET(nb, trace_gui->tab, "notebook");

  return nb;
}


LND_ProtoInfo  *
nd_trace_get_proto_info(const LND_Trace *trace,
			const LND_Protocol *proto,
			guint nesting)
{
  LND_ProtoInfo *pinf;
  GtkNotebook   *notebook;
  const char    *key;
  
  if (!trace || !proto)
    return NULL;

  notebook = nd_trace_get_notebook(trace);
  D_ASSERT_PTR(notebook);

  if (!notebook)
    return NULL;

  key = libnd_proto_inst_to_string(proto, nesting);
  ND_GTK_GET(pinf, notebook, key);

  return pinf;
}


LND_Packet     *
nd_trace_get_current_packet(const LND_Trace *trace)
{
  ND_Trace *trace_gui = nd_trace_get(trace);

  if (!trace_gui)
    return NULL;

  return trace_gui->cur_packet;
}


void
nd_trace_goto(LND_Trace *trace, ND_TraceLocation loc)
{
  LND_Packet *packet = NULL;
  LND_ProtoInst *pi;
  ND_Trace *trace_gui = nd_trace_get(trace);
  GtkScrolledWindow *scr;
  GtkAdjustment *adj;

  D_ENTER;

  if (!trace_gui)
    D_RETURN;

  if (loc == ND_TRACE_JUMP_TOP)
    {
      ND_GTK_GET(scr, trace_gui->tab, "scrolledwin");
      adj = gtk_scrolled_window_get_vadjustment(scr);
      
      if (adj)
	{
	  adj->value = adj->lower;
	  gtk_scrolled_window_set_vadjustment(scr, adj);
	}
      D_RETURN;
    }

  if (loc == ND_TRACE_JUMP_BOTTOM)
    {
      ND_GTK_GET(scr, trace_gui->tab, "scrolledwin");
      adj = gtk_scrolled_window_get_vadjustment(scr);
      
      if (adj)
	{
	  adj->value = adj->upper;
	  gtk_scrolled_window_set_vadjustment(scr, adj);
	}
      D_RETURN;
    }
  
  packet = nd_trace_get_current_packet(trace);
  pi = nd_trace_get_current_proto_inst(trace);

  if (!packet)
    D_RETURN;

  switch(loc)
    {
    case ND_TRACE_JUMP_NEXT_UNFILTERED:
      do {
	packet = packet->next;
      } while (packet && libnd_packet_is_filtered(packet));

      if (packet)
	{
	  int index = libnd_packet_get_index(packet);
	  	  
	  nd_gui_list_unselect_all(trace, TRUE);
	  nd_gui_list_select_packet(trace, index);
	}
      break;

    case ND_TRACE_JUMP_PREV_UNFILTERED:
      do {
	packet = packet->prev;
      } while (packet && libnd_packet_is_filtered(packet));

      if (packet)
	{
	  int index = libnd_packet_get_index(packet);
	  	  
	  nd_gui_list_unselect_all(trace, TRUE);
	  nd_gui_list_select_packet(trace, index);
	}
      break;

    case ND_TRACE_JUMP_NEXT_SEL:
      if (packet->sel_next)
	{
	  int index = libnd_packet_get_index(packet->sel_next);
	  nd_gui_list_select_packet(trace, index);
	}
      break;

    case ND_TRACE_JUMP_PREV_SEL:
      if (packet->sel_prev)
	{
	  int index = libnd_packet_get_index(packet->sel_prev);
	  nd_gui_list_select_packet(trace, index);
	}
      break;

    case ND_TRACE_JUMP_NEXT_PROT:
      if (!pi)
	D_RETURN;

      do {
	packet = packet->next;
      } while (packet &&
	       (! libnd_packet_get_data(packet, pi->proto, pi->nesting) ||
		libnd_packet_is_filtered(packet)));
      
      if (packet)
	{
	  int index = libnd_packet_get_index(packet);
	  
	  nd_gui_list_unselect_all(trace, TRUE);
	  nd_gui_list_select_packet(trace, index);
	}
      break;

    case ND_TRACE_JUMP_PREV_PROT:
      if (!pi)
	D_RETURN;

      do {
	packet = packet->prev;
      } while (packet &&
	       (! libnd_packet_get_data(packet, pi->proto, pi->nesting) ||
		libnd_packet_is_filtered(packet)));
      
      if (packet)
	{
	  int index = libnd_packet_get_index(packet);
	  
	  nd_gui_list_unselect_all(trace, TRUE);
	  nd_gui_list_select_packet(trace, index);
	}
      break;

    case ND_TRACE_JUMP_PREV:
      packet = packet->prev;
      if (packet)
	{
	  int index = libnd_packet_get_index(packet);
	  
	  nd_gui_list_unselect_all(trace, TRUE);
	  nd_gui_list_select_packet(trace, index);
	}
      break;

    case ND_TRACE_JUMP_NEXT:
    default:
      packet = packet->next;
      if (packet)
	{
	  int index = libnd_packet_get_index(packet);
	  
	  nd_gui_list_unselect_all(trace, TRUE);
	  nd_gui_list_select_packet(trace, index);
	}
    }

  D_RETURN;
}


ND_Trace *
nd_trace_get(const LND_Trace *trace)
{
  ND_Trace *trace_gui = NULL;

  if (!trace)
    return NULL;

  trace_gui = libnd_reg_get_data(trace->registry, "trace_gui");
  D_ASSERT_PTR(trace_gui);
  
  return trace_gui;
}

