/* anbio.c -- asynchronous/non-blocking input/output
 * 
 *   Copyright (C) 1996 1997 1998 1999 2000 2001 Ian Piumarta and individual
 *      authors/contributors listed elsewhere in this file.
 *   All rights reserved.
 *   
 *   This file is part of Unix Squeak.
 * 
 *   This file is distributed in the hope that it will be useful, but WITHOUT
 *   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *   FITNESS FOR A PARTICULAR PURPOSE.
 *   
 *   You may use and/or distribute this file ONLY as part of Squeak, under
 *   the terms of the Squeak License as described in `LICENSE' in the base of
 *   this distribution, subject to the following restrictions:
 * 
 *   1. The origin of this software must not be misrepresented; you must not
 *      claim that you wrote the original software.  If you use this software
 *      in a product, an acknowledgment to the original author(s) (and any
 *      other contributors mentioned herein) in the product documentation
 *      would be appreciated but is not required.
 * 
 *   2. This notice may not be removed or altered in any source distribution.
 * 
 *   Using or modifying this file for use in any context other than Squeak
 *   changes these copyright conditions.  Read the file `COPYING' in the base
 *   of the distribution before proceeding with any such use.
 * 
 *   You are STRONGLY DISCOURAGED from distributing a modified version of
 *   this file under its original name without permission.  If you must
 *   change it, rename it first.
 * 
 * Author: Ian.Piumarta@INRIA.Fr
 * 
 * Last edited: Fri Aug 25 05:29:53 2000 by piumarta (Ian Piumarta) on emilia
 */

#include "anbio.h"

#include <sys/ioctl.h>	/* ioctl */
#include <sys/time.h>	/* select */
#include <unistd.h>	/* select, fcntl */
#include <fcntl.h>	/* fcntl */
#include <signal.h>	/* signal */
#include <string.h>	/* strerror */
#include <stdio.h>	/* printf */

#ifndef __GNUC__
# define __FILE__ "anbio.c"
# define __LINE__ 666
#endif

#define where()			(fprintf(stderr, "%s:%d: ", __FILE__, __LINE__))

#define warningError(msg)	(where(), perror(msg))
#define warning(msg)		(where(), fprintf(stderr, "%s\n", msg))

#define fatalError(msg)		(exit((warningError(msg), 1)))
#define fatal(msg)		(exit((warning(msg), 1)))

#ifdef DEBUG
# define info(msg)		(fprintf(stderr, msg"\n"))
# define info1(msg, arg)	(fprintf(stderr, msg"\n", arg))
#else
# define info(msg)
# define info1(msg, arg)
#endif

static aioHandler_t    handlers[FD_SETSIZE];
static aioClientData_t clientData[FD_SETSIZE];

static fd_set fds;
static int    lastFd= -1;

char *maskString(int mask)
{
  switch (mask)
    {
    case (     0 |      0 |      0): return "---";
    case (     0 |      0 | AIO_RD): return "--R";
    case (     0 | AIO_WR |      0): return "-W-";
    case (     0 | AIO_WR | AIO_RD): return "-WR";
    case (AIO_EX |      0 |      0): return "X--";
    case (AIO_EX |      0 | AIO_RD): return "X-R";
    case (AIO_EX | AIO_WR |      0): return "XW-";
    case (AIO_EX | AIO_WR | AIO_RD): return "XWR";
    }
  warning("illegal mask value");
  return "BAD";
}

static void sigio(int signum)
{
  fd_set rd= fds;
  fd_set wr= fds;
  fd_set ex= fds;
  info("sigio");
  {
    struct timeval tv= { 0, 0 };
    int n= select(lastFd + 1, &rd, &wr, &ex, &tv);
    if      (n <  0) fatalError("select");
    else if (n == 0) warning("select returned 0");
    else
      {
	for (n= lastFd; n >= 0 ; --n)
	  {
	    int mask= (  (FD_ISSET(n, &rd) ? AIO_RD : 0)
		       | (FD_ISSET(n, &wr) ? AIO_WR : 0)
		       | (FD_ISSET(n, &ex) ? AIO_EX : 0));
	    if (mask != 0)
	      {
		if (handlers[n] == 0) fatal("no handler");
		{
		  aioHandler_t next= (aioHandler_t)(handlers[n](n, mask, clientData[n]));
		  if (next == 0) FD_CLR(n, &fds);
		  handlers[n]= next;
		}
	      }
	  }
      }
  }
}

void nbioEnable(int fd)
{
  int one= 1;
  info1("%d enabling non-blocking", fd);
  if (ioctl(fd, FIONBIO, (char *)&one)) fatalError("ioctl(FIONBIO)");
}

void nbioDisable(int fd)
{
  int zero= 0;
  info1("%d disabling non-blocking", fd);
  if (clientData[fd] != 0)                fatal("descriptor in use");
  if (ioctl(fd, FIONBIO,  (char *)&zero)) fatalError("ioctl(FIONBIO)");
}

void aioHandle(int fd, aioHandler_t hdlr)
{
  handlers[fd]= hdlr;
  FD_SET(fd, &fds);
}

void aioSuspend(int fd)
{
  FD_CLR(fd, &fds);
  handlers[fd]= 0;
}

void aioEnable(int fd, aioClientData_t data)
{
  info1("%d enabling notification", fd);
  if (data == 0)           fatal("missing client data");
  if (clientData[fd] != 0) fatal("descriptor busy");
  if (handlers[fd] == 0)   fatal("descriptor not handled");
  clientData[fd]= data;
  if (lastFd < fd) lastFd= fd;
  {
    int one= 1;
    if (ioctl(fd, FIOASYNC, (char *)&one)) fatalError("ioctl(FIOASYNC)");
    if (fcntl(fd, F_SETOWN, getpid()))     fatalError("fcntl(F_SETOWN)");
  }
}

void aioDisable(int fd)
{
  int zero= 0;
  info1("%d disabling notification", fd);
  if (clientData[fd] == 0)                fatal("descriptor not asynchronous");
  if (ioctl(fd, FIOASYNC, (char *)&zero)) fatalError("ioctl(FIOASYNC)");
  if (fcntl(fd, F_SETOWN, getpid()))      fatalError("fcntl(F_SETOWN)");
  FD_CLR(fd, &fds);
  while ((lastFd >= 0) && (!FD_ISSET(lastFd, &fds)))
    --lastFd;
}

static int nClients= 0;

void aioInit(void)
{
  if (nClients++ == 0)
    {
      FD_ZERO(&fds);
      lastFd= -1;
      {
	int i= 0;
	for (i= FD_SETSIZE - 1; i >= 0; --i)
	  handlers[i]= 0;
      }
      {
	struct sigaction sa;
	sigset_t mask;
	sigemptyset(&mask);
	sigaddset(&mask, SIGIO);
	sigaddset(&mask, SIGPIPE);
	sa.sa_handler= sigio;
	sa.sa_mask= mask;
	sa.sa_flags= 0;
#	ifdef SA_RESTART
	sa.sa_flags|= SA_RESTART;
#	endif
	sigaction(SIGIO, &sa, 0);
      }
    }
}

void aioFini(void)
{
  if (--nClients == 0)
    {
      if (lastFd != -1)
	{
	  warning("aioFini: disabling active descriptors");
	  while (lastFd > -1)
	    if (clientData[lastFd] != 0)
	      aioDisable(lastFd);
	}
      signal(SIGIO, SIG_DFL);
      FD_ZERO(&fds);
      lastFd= -1;
    }
}
