/* valaclass.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaclass.h>
#include <gee/arraylist.h>
#include <gee/readonlylist.h>
#include <gee/collection.h>
#include <vala/valacodevisitor.h>
#include <vala/valatypeparameter.h>
#include <vala/valadatatype.h>
#include <vala/valaconstant.h>
#include <vala/valafield.h>
#include <vala/valamethod.h>
#include <vala/valaproperty.h>
#include <vala/valasignal.h>
#include <vala/valastruct.h>
#include <vala/valaenum.h>
#include <vala/valadelegate.h>
#include <vala/valaconstructor.h>
#include <vala/valadestructor.h>
#include <vala/valasourcereference.h>
#include <vala/valasymbol.h>
#include <vala/valascope.h>
#include <vala/valamember.h>
#include <vala/valacreationmethod.h>
#include <vala/valaformalparameter.h>
#include <vala/valaobjecttype.h>
#include <vala/valavoidtype.h>
#include <vala/valalocalvariable.h>
#include <vala/valaexpression.h>
#include <vala/valareport.h>
#include <vala/valaattribute.h>
#include <vala/valatypesymbol.h>




struct _ValaClassPrivate {
	ValaClass* _base_class;
	gboolean _is_abstract;
	char* _type_check_function;
	gboolean _has_private_fields;
	char* cname;
	char* const_cname;
	char* lower_case_cprefix;
	char* lower_case_csuffix;
	char* type_id;
	char* ref_function;
	char* unref_function;
	char* param_spec_function;
	char* copy_function;
	char* free_function;
	char* marshaller_type_name;
	char* get_value_function;
	char* set_value_function;
	char* type_signature;
	gboolean _is_compact;
	gboolean _is_immutable;
	GeeList* type_parameters;
	GeeList* base_types;
	GeeList* constants;
	GeeList* fields;
	GeeList* methods;
	GeeList* properties;
	GeeList* signals;
	GeeList* classes;
	GeeList* structs;
	GeeList* enums;
	GeeList* delegates;
	ValaMethod* _default_construction_method;
	ValaConstructor* _constructor;
	ValaConstructor* _class_constructor;
	ValaConstructor* _static_constructor;
	ValaDestructor* _destructor;
};

#define VALA_CLASS_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CLASS, ValaClassPrivate))
enum  {
	VALA_CLASS_DUMMY_PROPERTY
};
static GeeList* vala_class_real_get_methods (ValaObjectTypeSymbol* base);
static GeeList* vala_class_real_get_properties (ValaObjectTypeSymbol* base);
static GeeList* vala_class_real_get_signals (ValaObjectTypeSymbol* base);
static void vala_class_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_class_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_class_real_get_cprefix (ValaSymbol* base);
static char* vala_class_real_get_cname (ValaTypeSymbol* base, gboolean const_type);
static char* vala_class_get_lower_case_csuffix (ValaClass* self);
static char* vala_class_real_get_lower_case_cname (ValaSymbol* base, const char* infix);
static char* vala_class_real_get_lower_case_cprefix (ValaSymbol* base);
static char* vala_class_real_get_upper_case_cname (ValaTypeSymbol* base, const char* infix);
static char* vala_class_real_get_type_signature (ValaTypeSymbol* base);
static gboolean vala_class_real_is_reference_type (ValaTypeSymbol* base);
static void vala_class_process_ccode_attribute (ValaClass* self, ValaAttribute* a);
static char* vala_class_real_get_type_id (ValaTypeSymbol* base);
static char* vala_class_real_get_marshaller_type_name (ValaTypeSymbol* base);
static char* vala_class_real_get_param_spec_function (ValaTypeSymbol* base);
static char* vala_class_real_get_get_value_function (ValaTypeSymbol* base);
static char* vala_class_real_get_set_value_function (ValaTypeSymbol* base);
static gboolean vala_class_real_is_reference_counting (ValaTypeSymbol* base);
static char* vala_class_real_get_ref_function (ValaTypeSymbol* base);
static char* vala_class_real_get_unref_function (ValaTypeSymbol* base);
static char* vala_class_real_get_dup_function (ValaTypeSymbol* base);
static char* vala_class_real_get_free_function (ValaTypeSymbol* base);
static gboolean vala_class_real_is_subtype_of (ValaTypeSymbol* base, ValaTypeSymbol* t);
static gint vala_class_real_get_type_parameter_index (ValaTypeSymbol* base, const char* name);
static void vala_class_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static void vala_class_set_has_private_fields (ValaClass* self, gboolean value);
static gpointer vala_class_parent_class = NULL;
static void vala_class_finalize (ValaCodeNode* obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Returns a copy of the list of classes.
 *
 * @return list of classes
 */
GeeList* vala_class_get_classes (ValaClass* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return ((GeeList*) (gee_read_only_list_new (VALA_TYPE_CLASS, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, self->priv->classes)));
}


/**
 * Returns a copy of the list of structs.
 *
 * @return list of structs
 */
GeeList* vala_class_get_structs (ValaClass* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return ((GeeList*) (gee_read_only_list_new (VALA_TYPE_STRUCT, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, self->priv->structs)));
}


/**
 * Returns a copy of the list of enums.
 *
 * @return list of enums
 */
GeeList* vala_class_get_enums (ValaClass* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return ((GeeList*) (gee_read_only_list_new (VALA_TYPE_ENUM, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, self->priv->enums)));
}


/**
 * Returns a copy of the list of delegates.
 *
 * @return list of delegates
 */
GeeList* vala_class_get_delegates (ValaClass* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return ((GeeList*) (gee_read_only_list_new (VALA_TYPE_DELEGATE, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, self->priv->delegates)));
}


/**
 * Creates a new class.
 *
 * @param name   type name
 * @param source reference to source code
 * @return       newly created class
 */
ValaClass* vala_class_construct (GType object_type, const char* name, ValaSourceReference* source_reference) {
	ValaClass* self;
	g_return_val_if_fail (name != NULL, NULL);
	self = ((ValaClass*) (g_type_create_instance (object_type)));
	self = ((ValaClass*) (vala_object_type_symbol_construct (object_type, name, source_reference)));
	return self;
}


ValaClass* vala_class_new (const char* name, ValaSourceReference* source_reference) {
	return vala_class_construct (VALA_TYPE_CLASS, name, source_reference);
}


/**
 * Adds the specified class or interface to the list of base types of
 * this class.
 *
 * @param type a class or interface reference
 */
void vala_class_add_base_type (ValaClass* self, ValaDataType* type) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (type != NULL);
	gee_collection_add (((GeeCollection*) (self->priv->base_types)), type);
	vala_code_node_set_parent_node (((ValaCodeNode*) (type)), ((ValaCodeNode*) (self)));
}


/**
 * Returns a copy of the base type list.
 *
 * @return list of base types
 */
GeeList* vala_class_get_base_types (ValaClass* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return ((GeeList*) (gee_read_only_list_new (VALA_TYPE_DATA_TYPE, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, self->priv->base_types)));
}


/**
 * Appends the specified parameter to the list of type parameters.
 *
 * @param p a type parameter
 */
void vala_class_add_type_parameter (ValaClass* self, ValaTypeParameter* p) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (p != NULL);
	gee_collection_add (((GeeCollection*) (self->priv->type_parameters)), p);
	p->type = ((ValaTypeSymbol*) (self));
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) (self))), vala_symbol_get_name (((ValaSymbol*) (p))), ((ValaSymbol*) (p)));
}


/**
 * Returns a copy of the type parameter list.
 *
 * @return list of type parameters
 */
GeeList* vala_class_get_type_parameters (ValaClass* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return ((GeeList*) (gee_read_only_list_new (VALA_TYPE_TYPEPARAMETER, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, self->priv->type_parameters)));
}


/**
 * Adds the specified constant as a member to this class.
 *
 * @param c a constant
 */
void vala_class_add_constant (ValaClass* self, ValaConstant* c) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (c != NULL);
	gee_collection_add (((GeeCollection*) (self->priv->constants)), c);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) (self))), vala_symbol_get_name (((ValaSymbol*) (c))), ((ValaSymbol*) (c)));
}


/**
 * Adds the specified field as a member to this class.
 *
 * @param f a field
 */
void vala_class_add_field (ValaClass* self, ValaField* f) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (f != NULL);
	gee_collection_add (((GeeCollection*) (self->priv->fields)), f);
	if (vala_symbol_get_access (((ValaSymbol*) (f))) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE && vala_field_get_binding (f) == MEMBER_BINDING_INSTANCE) {
		vala_class_set_has_private_fields (self, TRUE);
	}
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) (self))), vala_symbol_get_name (((ValaSymbol*) (f))), ((ValaSymbol*) (f)));
}


/**
 * Returns a copy of the list of fields.
 *
 * @return list of fields
 */
GeeList* vala_class_get_fields (ValaClass* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return ((GeeList*) (gee_read_only_list_new (VALA_TYPE_FIELD, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, self->priv->fields)));
}


/**
 * Returns a copy of the list of constants.
 *
 * @return list of constants
 */
GeeList* vala_class_get_constants (ValaClass* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return ((GeeList*) (gee_read_only_list_new (VALA_TYPE_CONSTANT, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, self->priv->constants)));
}


/**
 * Adds the specified method as a member to this class.
 *
 * @param m a method
 */
void vala_class_add_method (ValaClass* self, ValaMethod* m) {
	GeeList* _tmp2;
	gboolean _tmp3;
	g_return_if_fail (self != NULL);
	g_return_if_fail (m != NULL);
	if (vala_method_get_binding (m) == MEMBER_BINDING_INSTANCE || VALA_IS_CREATION_METHOD (m)) {
		ValaFormalParameter* _tmp1;
		ValaObjectType* _tmp0;
		if (vala_method_get_this_parameter (m) != NULL) {
			vala_scope_remove (vala_symbol_get_scope (((ValaSymbol*) (m))), vala_symbol_get_name (((ValaSymbol*) (vala_method_get_this_parameter (m)))));
		}
		_tmp1 = NULL;
		_tmp0 = NULL;
		vala_method_set_this_parameter (m, (_tmp1 = vala_formal_parameter_new ("this", ((ValaDataType*) ((_tmp0 = vala_object_type_new (((ValaObjectTypeSymbol*) (self)))))), NULL)));
		(_tmp1 == NULL ? NULL : (_tmp1 = (vala_code_node_unref (_tmp1), NULL)));
		(_tmp0 == NULL ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL)));
		vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) (m))), vala_symbol_get_name (((ValaSymbol*) (vala_method_get_this_parameter (m)))), ((ValaSymbol*) (vala_method_get_this_parameter (m))));
	}
	_tmp2 = NULL;
	if ((_tmp3 = !(VALA_IS_VOID_TYPE (vala_method_get_return_type (m))) && gee_collection_get_size (((GeeCollection*) ((_tmp2 = vala_method_get_postconditions (m))))) > 0, (_tmp2 == NULL ? NULL : (_tmp2 = (gee_collection_object_unref (_tmp2), NULL))), _tmp3)) {
		ValaLocalVariable* _tmp5;
		ValaDataType* _tmp4;
		if (vala_method_get_result_var (m) != NULL) {
			vala_scope_remove (vala_symbol_get_scope (((ValaSymbol*) (m))), vala_symbol_get_name (((ValaSymbol*) (vala_method_get_result_var (m)))));
		}
		_tmp5 = NULL;
		_tmp4 = NULL;
		vala_method_set_result_var (m, (_tmp5 = vala_local_variable_new ((_tmp4 = vala_data_type_copy (vala_method_get_return_type (m))), "result", NULL, NULL)));
		(_tmp5 == NULL ? NULL : (_tmp5 = (vala_code_node_unref (_tmp5), NULL)));
		(_tmp4 == NULL ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL)));
		vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) (m))), vala_symbol_get_name (((ValaSymbol*) (vala_method_get_result_var (m)))), ((ValaSymbol*) (vala_method_get_result_var (m))));
	}
	if (VALA_IS_CREATION_METHOD (m)) {
		ValaCreationMethod* _tmp6;
		ValaCreationMethod* cm;
		if (vala_symbol_get_name (((ValaSymbol*) (m))) == NULL) {
			vala_class_set_default_construction_method (self, m);
			vala_symbol_set_name (((ValaSymbol*) (m)), "new");
		}
		_tmp6 = NULL;
		cm = (_tmp6 = VALA_CREATION_METHOD (m), (_tmp6 == NULL ? NULL : vala_code_node_ref (_tmp6)));
		if (vala_creation_method_get_type_name (cm) != NULL && _vala_strcmp0 (vala_creation_method_get_type_name (cm), vala_symbol_get_name (((ValaSymbol*) (self)))) != 0) {
			char* _tmp8;
			char* _tmp7;
			/* type_name is null for constructors generated by GIdlParser*/
			_tmp8 = NULL;
			_tmp7 = NULL;
			vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) (m))), (_tmp8 = g_strdup_printf ("missing return type in method `%s.%s´", (_tmp7 = vala_symbol_get_full_name (((ValaSymbol*) (self)))), vala_creation_method_get_type_name (cm))));
			_tmp8 = (g_free (_tmp8), NULL);
			_tmp7 = (g_free (_tmp7), NULL);
			vala_code_node_set_error (((ValaCodeNode*) (m)), TRUE);
			(cm == NULL ? NULL : (cm = (vala_code_node_unref (cm), NULL)));
			return;
		}
		(cm == NULL ? NULL : (cm = (vala_code_node_unref (cm), NULL)));
	}
	gee_collection_add (((GeeCollection*) (self->priv->methods)), m);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) (self))), vala_symbol_get_name (((ValaSymbol*) (m))), ((ValaSymbol*) (m)));
}


/**
 * Returns a copy of the list of methods.
 *
 * @return list of methods
 */
static GeeList* vala_class_real_get_methods (ValaObjectTypeSymbol* base) {
	ValaClass * self;
	self = ((ValaClass*) (base));
	return ((GeeList*) (gee_read_only_list_new (VALA_TYPE_METHOD, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, self->priv->methods)));
}


/**
 * Adds the specified property as a member to this class.
 *
 * @param prop a property
 */
void vala_class_add_property (ValaClass* self, ValaProperty* prop) {
	ValaFormalParameter* _tmp1;
	ValaObjectType* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (prop != NULL);
	gee_collection_add (((GeeCollection*) (self->priv->properties)), prop);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) (self))), vala_symbol_get_name (((ValaSymbol*) (prop))), ((ValaSymbol*) (prop)));
	_tmp1 = NULL;
	_tmp0 = NULL;
	vala_property_set_this_parameter (prop, (_tmp1 = vala_formal_parameter_new ("this", ((ValaDataType*) ((_tmp0 = vala_object_type_new (((ValaObjectTypeSymbol*) (self)))))), NULL)));
	(_tmp1 == NULL ? NULL : (_tmp1 = (vala_code_node_unref (_tmp1), NULL)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL)));
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) (prop))), vala_symbol_get_name (((ValaSymbol*) (vala_property_get_this_parameter (prop)))), ((ValaSymbol*) (vala_property_get_this_parameter (prop))));
	if (vala_property_get_field (prop) != NULL) {
		vala_class_add_field (self, vala_property_get_field (prop));
	}
}


/**
 * Returns a copy of the list of properties.
 *
 * @return list of properties
 */
static GeeList* vala_class_real_get_properties (ValaObjectTypeSymbol* base) {
	ValaClass * self;
	self = ((ValaClass*) (base));
	return ((GeeList*) (gee_read_only_list_new (VALA_TYPE_PROPERTY, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, self->priv->properties)));
}


/**
 * Adds the specified signal as a member to this class.
 *
 * @param sig a signal
 */
void vala_class_add_signal (ValaClass* self, ValaSignal* sig) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (sig != NULL);
	gee_collection_add (((GeeCollection*) (self->priv->signals)), sig);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) (self))), vala_symbol_get_name (((ValaSymbol*) (sig))), ((ValaSymbol*) (sig)));
}


/**
 * Returns a copy of the list of signals.
 *
 * @return list of signals
 */
static GeeList* vala_class_real_get_signals (ValaObjectTypeSymbol* base) {
	ValaClass * self;
	self = ((ValaClass*) (base));
	return ((GeeList*) (gee_read_only_list_new (VALA_TYPE_SIGNAL, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, self->priv->signals)));
}


/**
 * Adds the specified class as an inner class.
 *
 * @param cl a class
 */
void vala_class_add_class (ValaClass* self, ValaClass* cl) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (cl != NULL);
	gee_collection_add (((GeeCollection*) (self->priv->classes)), cl);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) (self))), vala_symbol_get_name (((ValaSymbol*) (cl))), ((ValaSymbol*) (cl)));
}


/**
 * Adds the specified struct as an inner struct.
 *
 * @param st a struct
 */
void vala_class_add_struct (ValaClass* self, ValaStruct* st) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (st != NULL);
	gee_collection_add (((GeeCollection*) (self->priv->structs)), st);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) (self))), vala_symbol_get_name (((ValaSymbol*) (st))), ((ValaSymbol*) (st)));
}


/**
 * Adds the specified enum as an inner enum.
 *
 * @param en an enum
 */
void vala_class_add_enum (ValaClass* self, ValaEnum* en) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (en != NULL);
	gee_collection_add (((GeeCollection*) (self->priv->enums)), en);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) (self))), vala_symbol_get_name (((ValaSymbol*) (en))), ((ValaSymbol*) (en)));
}


/**
 * Adds the specified delegate as an inner delegate.
 *
 * @param d a delegate
 */
void vala_class_add_delegate (ValaClass* self, ValaDelegate* d) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (d != NULL);
	gee_collection_add (((GeeCollection*) (self->priv->delegates)), d);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) (self))), vala_symbol_get_name (((ValaSymbol*) (d))), ((ValaSymbol*) (d)));
}


static void vala_class_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaClass * self;
	self = ((ValaClass*) (base));
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_class (visitor, self);
}


static void vala_class_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaClass * self;
	self = ((ValaClass*) (base));
	g_return_if_fail (visitor != NULL);
	{
		GeeList* type_collection;
		int type_it;
		type_collection = self->priv->base_types;
		for (type_it = 0; type_it < gee_collection_get_size (GEE_COLLECTION (type_collection)); type_it = type_it + 1) {
			ValaDataType* type;
			type = ((ValaDataType*) (gee_list_get (GEE_LIST (type_collection), type_it)));
			{
				vala_code_node_accept (((ValaCodeNode*) (type)), visitor);
				(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
			}
		}
	}
	{
		GeeList* p_collection;
		int p_it;
		p_collection = self->priv->type_parameters;
		for (p_it = 0; p_it < gee_collection_get_size (GEE_COLLECTION (p_collection)); p_it = p_it + 1) {
			ValaTypeParameter* p;
			p = ((ValaTypeParameter*) (gee_list_get (GEE_LIST (p_collection), p_it)));
			{
				vala_code_node_accept (((ValaCodeNode*) (p)), visitor);
				(p == NULL ? NULL : (p = (vala_code_node_unref (p), NULL)));
			}
		}
	}
	/* process enums first to avoid order problems in C code */
	{
		GeeList* en_collection;
		int en_it;
		en_collection = self->priv->enums;
		for (en_it = 0; en_it < gee_collection_get_size (GEE_COLLECTION (en_collection)); en_it = en_it + 1) {
			ValaEnum* en;
			en = ((ValaEnum*) (gee_list_get (GEE_LIST (en_collection), en_it)));
			{
				vala_code_node_accept (((ValaCodeNode*) (en)), visitor);
				(en == NULL ? NULL : (en = (vala_code_node_unref (en), NULL)));
			}
		}
	}
	{
		GeeList* f_collection;
		int f_it;
		f_collection = self->priv->fields;
		for (f_it = 0; f_it < gee_collection_get_size (GEE_COLLECTION (f_collection)); f_it = f_it + 1) {
			ValaField* f;
			f = ((ValaField*) (gee_list_get (GEE_LIST (f_collection), f_it)));
			{
				vala_code_node_accept (((ValaCodeNode*) (f)), visitor);
				(f == NULL ? NULL : (f = (vala_code_node_unref (f), NULL)));
			}
		}
	}
	{
		GeeList* c_collection;
		int c_it;
		c_collection = self->priv->constants;
		for (c_it = 0; c_it < gee_collection_get_size (GEE_COLLECTION (c_collection)); c_it = c_it + 1) {
			ValaConstant* c;
			c = ((ValaConstant*) (gee_list_get (GEE_LIST (c_collection), c_it)));
			{
				vala_code_node_accept (((ValaCodeNode*) (c)), visitor);
				(c == NULL ? NULL : (c = (vala_code_node_unref (c), NULL)));
			}
		}
	}
	{
		GeeList* m_collection;
		int m_it;
		m_collection = self->priv->methods;
		for (m_it = 0; m_it < gee_collection_get_size (GEE_COLLECTION (m_collection)); m_it = m_it + 1) {
			ValaMethod* m;
			m = ((ValaMethod*) (gee_list_get (GEE_LIST (m_collection), m_it)));
			{
				vala_code_node_accept (((ValaCodeNode*) (m)), visitor);
				(m == NULL ? NULL : (m = (vala_code_node_unref (m), NULL)));
			}
		}
	}
	{
		GeeList* prop_collection;
		int prop_it;
		prop_collection = self->priv->properties;
		for (prop_it = 0; prop_it < gee_collection_get_size (GEE_COLLECTION (prop_collection)); prop_it = prop_it + 1) {
			ValaProperty* prop;
			prop = ((ValaProperty*) (gee_list_get (GEE_LIST (prop_collection), prop_it)));
			{
				vala_code_node_accept (((ValaCodeNode*) (prop)), visitor);
				(prop == NULL ? NULL : (prop = (vala_code_node_unref (prop), NULL)));
			}
		}
	}
	{
		GeeList* sig_collection;
		int sig_it;
		sig_collection = self->priv->signals;
		for (sig_it = 0; sig_it < gee_collection_get_size (GEE_COLLECTION (sig_collection)); sig_it = sig_it + 1) {
			ValaSignal* sig;
			sig = ((ValaSignal*) (gee_list_get (GEE_LIST (sig_collection), sig_it)));
			{
				vala_code_node_accept (((ValaCodeNode*) (sig)), visitor);
				(sig == NULL ? NULL : (sig = (vala_code_node_unref (sig), NULL)));
			}
		}
	}
	if (self->priv->_constructor != NULL) {
		vala_code_node_accept (((ValaCodeNode*) (self->priv->_constructor)), visitor);
	}
	if (self->priv->_class_constructor != NULL) {
		vala_code_node_accept (((ValaCodeNode*) (self->priv->_class_constructor)), visitor);
	}
	if (self->priv->_static_constructor != NULL) {
		vala_code_node_accept (((ValaCodeNode*) (self->priv->_static_constructor)), visitor);
	}
	if (vala_class_get_destructor (self) != NULL) {
		vala_code_node_accept (((ValaCodeNode*) (vala_class_get_destructor (self))), visitor);
	}
	{
		GeeList* cl_collection;
		int cl_it;
		cl_collection = self->priv->classes;
		for (cl_it = 0; cl_it < gee_collection_get_size (GEE_COLLECTION (cl_collection)); cl_it = cl_it + 1) {
			ValaClass* cl;
			cl = ((ValaClass*) (gee_list_get (GEE_LIST (cl_collection), cl_it)));
			{
				vala_code_node_accept (((ValaCodeNode*) (cl)), visitor);
				(cl == NULL ? NULL : (cl = (vala_code_node_unref (cl), NULL)));
			}
		}
	}
	{
		GeeList* st_collection;
		int st_it;
		st_collection = self->priv->structs;
		for (st_it = 0; st_it < gee_collection_get_size (GEE_COLLECTION (st_collection)); st_it = st_it + 1) {
			ValaStruct* st;
			st = ((ValaStruct*) (gee_list_get (GEE_LIST (st_collection), st_it)));
			{
				vala_code_node_accept (((ValaCodeNode*) (st)), visitor);
				(st == NULL ? NULL : (st = (vala_code_node_unref (st), NULL)));
			}
		}
	}
	{
		GeeList* d_collection;
		int d_it;
		d_collection = self->priv->delegates;
		for (d_it = 0; d_it < gee_collection_get_size (GEE_COLLECTION (d_collection)); d_it = d_it + 1) {
			ValaDelegate* d;
			d = ((ValaDelegate*) (gee_list_get (GEE_LIST (d_collection), d_it)));
			{
				vala_code_node_accept (((ValaCodeNode*) (d)), visitor);
				(d == NULL ? NULL : (d = (vala_code_node_unref (d), NULL)));
			}
		}
	}
}


static char* vala_class_real_get_cprefix (ValaSymbol* base) {
	ValaClass * self;
	self = ((ValaClass*) (base));
	return vala_typesymbol_get_cname (((ValaTypeSymbol*) (self)), FALSE);
}


static char* vala_class_real_get_cname (ValaTypeSymbol* base, gboolean const_type) {
	ValaClass * self;
	const char* _tmp7;
	self = ((ValaClass*) (base));
	if (const_type) {
		if (self->priv->const_cname != NULL) {
			const char* _tmp0;
			_tmp0 = NULL;
			return (_tmp0 = self->priv->const_cname, (_tmp0 == NULL ? NULL : g_strdup (_tmp0)));
		} else {
			if (vala_class_get_is_immutable (self)) {
				char* _tmp2;
				char* _tmp3;
				_tmp2 = NULL;
				_tmp3 = NULL;
				return (_tmp3 = g_strconcat ("const ", (_tmp2 = vala_typesymbol_get_cname (((ValaTypeSymbol*) (self)), FALSE)), NULL), (_tmp2 = (g_free (_tmp2), NULL)), _tmp3);
			}
		}
	}
	if (self->priv->cname == NULL) {
		ValaAttribute* attr;
		attr = vala_code_node_get_attribute (((ValaCodeNode*) (self)), "CCode");
		if (attr != NULL) {
			char* _tmp5;
			_tmp5 = NULL;
			self->priv->cname = (_tmp5 = vala_attribute_get_string (attr, "cname"), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp5);
		}
		if (self->priv->cname == NULL) {
			char* _tmp6;
			_tmp6 = NULL;
			self->priv->cname = (_tmp6 = vala_class_get_default_cname (self), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp6);
		}
		(attr == NULL ? NULL : (attr = (vala_code_node_unref (attr), NULL)));
	}
	_tmp7 = NULL;
	return (_tmp7 = self->priv->cname, (_tmp7 == NULL ? NULL : g_strdup (_tmp7)));
}


/**
 * Returns the default name of this class as it is used in C code.
 *
 * @return the name to be used in C code by default
 */
char* vala_class_get_default_cname (ValaClass* self) {
	char* _tmp0;
	char* _tmp1;
	g_return_val_if_fail (self != NULL, NULL);
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = g_strdup_printf ("%s%s", (_tmp0 = vala_symbol_get_cprefix (vala_symbol_get_parent_symbol (((ValaSymbol*) (self))))), vala_symbol_get_name (((ValaSymbol*) (self)))), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
}


/**
 * Sets the name of this class as it is used in C code.
 *
 * @param cname the name to be used in C code
 */
void vala_class_set_cname (ValaClass* self, const char* cname) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (cname != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->cname = (_tmp1 = (_tmp0 = cname, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp1);
}


static char* vala_class_get_lower_case_csuffix (ValaClass* self) {
	const char* _tmp6;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->lower_case_csuffix == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->lower_case_csuffix = (_tmp0 = vala_symbol_camel_case_to_lower_case (vala_symbol_get_name (((ValaSymbol*) (self)))), (self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL)), _tmp0);
		/* remove underscores in some cases to avoid conflicts of type macros*/
		if (g_str_has_prefix (self->priv->lower_case_csuffix, "type_")) {
			char* _tmp1;
			_tmp1 = NULL;
			self->priv->lower_case_csuffix = (_tmp1 = g_strconcat ("type", g_utf8_offset_to_pointer (self->priv->lower_case_csuffix, g_utf8_strlen ("type_", -1)), NULL), (self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL)), _tmp1);
		} else {
			if (g_str_has_prefix (self->priv->lower_case_csuffix, "is_")) {
				char* _tmp2;
				_tmp2 = NULL;
				self->priv->lower_case_csuffix = (_tmp2 = g_strconcat ("is", g_utf8_offset_to_pointer (self->priv->lower_case_csuffix, g_utf8_strlen ("is_", -1)), NULL), (self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL)), _tmp2);
			}
		}
		if (g_str_has_suffix (self->priv->lower_case_csuffix, "_class")) {
			char* _tmp5;
			char* _tmp4;
			char* _tmp3;
			_tmp5 = NULL;
			_tmp4 = NULL;
			_tmp3 = NULL;
			self->priv->lower_case_csuffix = (_tmp5 = g_strconcat ((_tmp4 = (_tmp3 = g_utf8_offset_to_pointer (self->priv->lower_case_csuffix, ((glong) (0))), g_strndup (_tmp3, g_utf8_offset_to_pointer (_tmp3, g_utf8_strlen (self->priv->lower_case_csuffix, -1) - g_utf8_strlen ("_class", -1)) - _tmp3))), "class", NULL), (self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL)), _tmp5);
			_tmp4 = (g_free (_tmp4), NULL);
		}
	}
	_tmp6 = NULL;
	return (_tmp6 = self->priv->lower_case_csuffix, (_tmp6 == NULL ? NULL : g_strdup (_tmp6)));
}


static char* vala_class_real_get_lower_case_cname (ValaSymbol* base, const char* infix) {
	ValaClass * self;
	char* _tmp1;
	char* _tmp0;
	char* _tmp2;
	self = ((ValaClass*) (base));
	if (infix == NULL) {
		infix = "";
	}
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = g_strdup_printf ("%s%s%s", (_tmp0 = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol (((ValaSymbol*) (self))))), infix, (_tmp1 = vala_class_get_lower_case_csuffix (self))), (_tmp1 = (g_free (_tmp1), NULL)), (_tmp0 = (g_free (_tmp0), NULL)), _tmp2);
}


static char* vala_class_real_get_lower_case_cprefix (ValaSymbol* base) {
	ValaClass * self;
	const char* _tmp2;
	self = ((ValaClass*) (base));
	if (self->priv->lower_case_cprefix == NULL) {
		char* _tmp1;
		char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->lower_case_cprefix = (_tmp1 = g_strdup_printf ("%s_", (_tmp0 = vala_symbol_get_lower_case_cname (((ValaSymbol*) (self)), NULL))), (self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL)), _tmp1);
		_tmp0 = (g_free (_tmp0), NULL);
	}
	_tmp2 = NULL;
	return (_tmp2 = self->priv->lower_case_cprefix, (_tmp2 == NULL ? NULL : g_strdup (_tmp2)));
}


static char* vala_class_real_get_upper_case_cname (ValaTypeSymbol* base, const char* infix) {
	ValaClass * self;
	char* _tmp0;
	char* _tmp1;
	self = ((ValaClass*) (base));
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = g_utf8_strup ((_tmp0 = vala_symbol_get_lower_case_cname (((ValaSymbol*) (self)), infix)), -1), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
}


static char* vala_class_real_get_type_signature (ValaTypeSymbol* base) {
	ValaClass * self;
	const char* _tmp0;
	self = ((ValaClass*) (base));
	_tmp0 = NULL;
	return (_tmp0 = self->priv->type_signature, (_tmp0 == NULL ? NULL : g_strdup (_tmp0)));
}


static gboolean vala_class_real_is_reference_type (ValaTypeSymbol* base) {
	ValaClass * self;
	self = ((ValaClass*) (base));
	return TRUE;
}


static void vala_class_process_ccode_attribute (ValaClass* self, ValaAttribute* a) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (a != NULL);
	if (vala_attribute_has_argument (a, "ref_function")) {
		char* _tmp0;
		_tmp0 = NULL;
		vala_class_set_ref_function (self, (_tmp0 = vala_attribute_get_string (a, "ref_function")));
		_tmp0 = (g_free (_tmp0), NULL);
	}
	if (vala_attribute_has_argument (a, "unref_function")) {
		char* _tmp1;
		_tmp1 = NULL;
		vala_class_set_unref_function (self, (_tmp1 = vala_attribute_get_string (a, "unref_function")));
		_tmp1 = (g_free (_tmp1), NULL);
	}
	if (vala_attribute_has_argument (a, "copy_function")) {
		char* _tmp2;
		_tmp2 = NULL;
		vala_class_set_dup_function (self, (_tmp2 = vala_attribute_get_string (a, "copy_function")));
		_tmp2 = (g_free (_tmp2), NULL);
	}
	if (vala_attribute_has_argument (a, "free_function")) {
		char* _tmp3;
		_tmp3 = NULL;
		vala_class_set_free_function (self, (_tmp3 = vala_attribute_get_string (a, "free_function")));
		_tmp3 = (g_free (_tmp3), NULL);
	}
	if (vala_attribute_has_argument (a, "type_id")) {
		char* _tmp4;
		_tmp4 = NULL;
		self->priv->type_id = (_tmp4 = vala_attribute_get_string (a, "type_id"), (self->priv->type_id = (g_free (self->priv->type_id), NULL)), _tmp4);
	}
	if (vala_attribute_has_argument (a, "marshaller_type_name")) {
		char* _tmp5;
		_tmp5 = NULL;
		self->priv->marshaller_type_name = (_tmp5 = vala_attribute_get_string (a, "marshaller_type_name"), (self->priv->marshaller_type_name = (g_free (self->priv->marshaller_type_name), NULL)), _tmp5);
	}
	if (vala_attribute_has_argument (a, "get_value_function")) {
		char* _tmp6;
		_tmp6 = NULL;
		self->priv->get_value_function = (_tmp6 = vala_attribute_get_string (a, "get_value_function"), (self->priv->get_value_function = (g_free (self->priv->get_value_function), NULL)), _tmp6);
	}
	if (vala_attribute_has_argument (a, "set_value_function")) {
		char* _tmp7;
		_tmp7 = NULL;
		self->priv->set_value_function = (_tmp7 = vala_attribute_get_string (a, "set_value_function"), (self->priv->set_value_function = (g_free (self->priv->set_value_function), NULL)), _tmp7);
	}
	if (vala_attribute_has_argument (a, "const_cname")) {
		char* _tmp8;
		_tmp8 = NULL;
		self->priv->const_cname = (_tmp8 = vala_attribute_get_string (a, "const_cname"), (self->priv->const_cname = (g_free (self->priv->const_cname), NULL)), _tmp8);
	}
	if (vala_attribute_has_argument (a, "cprefix")) {
		char* _tmp9;
		_tmp9 = NULL;
		self->priv->lower_case_cprefix = (_tmp9 = vala_attribute_get_string (a, "cprefix"), (self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL)), _tmp9);
	}
	if (vala_attribute_has_argument (a, "lower_case_csuffix")) {
		char* _tmp10;
		_tmp10 = NULL;
		self->priv->lower_case_csuffix = (_tmp10 = vala_attribute_get_string (a, "lower_case_csuffix"), (self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL)), _tmp10);
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** filename_collection;
			int filename_collection_length1;
			char** filename_it;
			filename_collection = g_strsplit (val, ",", 0);
			filename_collection_length1 = -1;
			for (filename_it = filename_collection; *filename_it != NULL; filename_it = filename_it + 1) {
				const char* _tmp11;
				char* filename;
				_tmp11 = NULL;
				filename = (_tmp11 = *filename_it, (_tmp11 == NULL ? NULL : g_strdup (_tmp11)));
				{
					vala_typesymbol_add_cheader_filename (((ValaTypeSymbol*) (self)), filename);
					filename = (g_free (filename), NULL);
				}
			}
			filename_collection = (_vala_array_free (filename_collection, filename_collection_length1, ((GDestroyNotify) (g_free))), NULL);
		}
		val = (g_free (val), NULL);
	}
	if (vala_attribute_has_argument (a, "type_signature")) {
		char* _tmp12;
		_tmp12 = NULL;
		self->priv->type_signature = (_tmp12 = vala_attribute_get_string (a, "type_signature"), (self->priv->type_signature = (g_free (self->priv->type_signature), NULL)), _tmp12);
	}
	if (vala_attribute_has_argument (a, "type_check_function")) {
		char* _tmp13;
		_tmp13 = NULL;
		vala_class_set_type_check_function (self, (_tmp13 = vala_attribute_get_string (a, "type_check_function")));
		_tmp13 = (g_free (_tmp13), NULL);
	}
	if (vala_attribute_has_argument (a, "param_spec_function")) {
		char* _tmp14;
		_tmp14 = NULL;
		self->priv->param_spec_function = (_tmp14 = vala_attribute_get_string (a, "param_spec_function"), (self->priv->param_spec_function = (g_free (self->priv->param_spec_function), NULL)), _tmp14);
	}
}


/**
 * Process all associated attributes.
 */
void vala_class_process_attributes (ValaClass* self) {
	g_return_if_fail (self != NULL);
	{
		GList* a_collection;
		GList* a_it;
		a_collection = ((ValaCodeNode*) (self))->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0;
			ValaAttribute* a;
			_tmp0 = NULL;
			a = (_tmp0 = ((ValaAttribute*) (a_it->data)), (_tmp0 == NULL ? NULL : vala_code_node_ref (_tmp0)));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0) {
					vala_class_process_ccode_attribute (self, a);
				} else {
					if (_vala_strcmp0 (vala_attribute_get_name (a), "Compact") == 0) {
						vala_class_set_is_compact (self, TRUE);
					} else {
						if (_vala_strcmp0 (vala_attribute_get_name (a), "Immutable") == 0) {
							vala_class_set_is_immutable (self, TRUE);
						}
					}
				}
				(a == NULL ? NULL : (a = (vala_code_node_unref (a), NULL)));
			}
		}
	}
}


static char* vala_class_real_get_type_id (ValaTypeSymbol* base) {
	ValaClass * self;
	const char* _tmp2;
	self = ((ValaClass*) (base));
	if (self->priv->type_id == NULL) {
		if (!vala_class_get_is_compact (self)) {
			char* _tmp0;
			_tmp0 = NULL;
			self->priv->type_id = (_tmp0 = vala_typesymbol_get_upper_case_cname (((ValaTypeSymbol*) (self)), "TYPE_"), (self->priv->type_id = (g_free (self->priv->type_id), NULL)), _tmp0);
		} else {
			char* _tmp1;
			_tmp1 = NULL;
			self->priv->type_id = (_tmp1 = g_strdup ("G_TYPE_POINTER"), (self->priv->type_id = (g_free (self->priv->type_id), NULL)), _tmp1);
		}
	}
	_tmp2 = NULL;
	return (_tmp2 = self->priv->type_id, (_tmp2 == NULL ? NULL : g_strdup (_tmp2)));
}


void vala_class_set_type_id (ValaClass* self, const char* type_id) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (type_id != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->type_id = (_tmp1 = (_tmp0 = type_id, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (self->priv->type_id = (g_free (self->priv->type_id), NULL)), _tmp1);
}


static char* vala_class_real_get_marshaller_type_name (ValaTypeSymbol* base) {
	ValaClass * self;
	const char* _tmp2;
	self = ((ValaClass*) (base));
	if (self->priv->marshaller_type_name == NULL) {
		if (self->priv->_base_class != NULL) {
			char* _tmp0;
			_tmp0 = NULL;
			self->priv->marshaller_type_name = (_tmp0 = vala_typesymbol_get_marshaller_type_name (((ValaTypeSymbol*) (self->priv->_base_class))), (self->priv->marshaller_type_name = (g_free (self->priv->marshaller_type_name), NULL)), _tmp0);
		} else {
			char* _tmp1;
			_tmp1 = NULL;
			self->priv->marshaller_type_name = (_tmp1 = g_strdup ("POINTER"), (self->priv->marshaller_type_name = (g_free (self->priv->marshaller_type_name), NULL)), _tmp1);
		}
	}
	_tmp2 = NULL;
	return (_tmp2 = self->priv->marshaller_type_name, (_tmp2 == NULL ? NULL : g_strdup (_tmp2)));
}


static char* vala_class_real_get_param_spec_function (ValaTypeSymbol* base) {
	ValaClass * self;
	const char* _tmp1;
	self = ((ValaClass*) (base));
	if (self->priv->param_spec_function == NULL) {
		if (!(vala_class_get_is_compact (self) || self->priv->_base_class == NULL)) {
			char* _tmp0;
			_tmp0 = NULL;
			self->priv->param_spec_function = (_tmp0 = vala_typesymbol_get_param_spec_function (((ValaTypeSymbol*) (self->priv->_base_class))), (self->priv->param_spec_function = (g_free (self->priv->param_spec_function), NULL)), _tmp0);
		}
	}
	_tmp1 = NULL;
	return (_tmp1 = self->priv->param_spec_function, (_tmp1 == NULL ? NULL : g_strdup (_tmp1)));
}


void vala_class_set_param_spec_function (ValaClass* self, const char* name) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (name != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->param_spec_function = (_tmp1 = (_tmp0 = name, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (self->priv->param_spec_function = (g_free (self->priv->param_spec_function), NULL)), _tmp1);
}


static char* vala_class_real_get_get_value_function (ValaTypeSymbol* base) {
	ValaClass * self;
	const char* _tmp3;
	self = ((ValaClass*) (base));
	if (self->priv->get_value_function == NULL) {
		if (vala_class_is_fundamental (self)) {
			char* _tmp0;
			_tmp0 = NULL;
			self->priv->get_value_function = (_tmp0 = vala_symbol_get_lower_case_cname (((ValaSymbol*) (self)), "value_get_"), (self->priv->get_value_function = (g_free (self->priv->get_value_function), NULL)), _tmp0);
		} else {
			if (self->priv->_base_class != NULL) {
				char* _tmp1;
				_tmp1 = NULL;
				self->priv->get_value_function = (_tmp1 = vala_typesymbol_get_get_value_function (((ValaTypeSymbol*) (self->priv->_base_class))), (self->priv->get_value_function = (g_free (self->priv->get_value_function), NULL)), _tmp1);
			} else {
				char* _tmp2;
				_tmp2 = NULL;
				self->priv->get_value_function = (_tmp2 = g_strdup ("g_value_get_pointer"), (self->priv->get_value_function = (g_free (self->priv->get_value_function), NULL)), _tmp2);
			}
		}
	}
	_tmp3 = NULL;
	return (_tmp3 = self->priv->get_value_function, (_tmp3 == NULL ? NULL : g_strdup (_tmp3)));
}


static char* vala_class_real_get_set_value_function (ValaTypeSymbol* base) {
	ValaClass * self;
	const char* _tmp3;
	self = ((ValaClass*) (base));
	if (self->priv->set_value_function == NULL) {
		if (vala_class_is_fundamental (self)) {
			char* _tmp0;
			_tmp0 = NULL;
			self->priv->set_value_function = (_tmp0 = vala_symbol_get_lower_case_cname (((ValaSymbol*) (self)), "value_set_"), (self->priv->set_value_function = (g_free (self->priv->set_value_function), NULL)), _tmp0);
		} else {
			if (self->priv->_base_class != NULL) {
				char* _tmp1;
				_tmp1 = NULL;
				self->priv->set_value_function = (_tmp1 = vala_typesymbol_get_set_value_function (((ValaTypeSymbol*) (self->priv->_base_class))), (self->priv->set_value_function = (g_free (self->priv->set_value_function), NULL)), _tmp1);
			} else {
				char* _tmp2;
				_tmp2 = NULL;
				self->priv->set_value_function = (_tmp2 = g_strdup ("g_value_set_pointer"), (self->priv->set_value_function = (g_free (self->priv->set_value_function), NULL)), _tmp2);
			}
		}
	}
	_tmp3 = NULL;
	return (_tmp3 = self->priv->set_value_function, (_tmp3 == NULL ? NULL : g_strdup (_tmp3)));
}


static gboolean vala_class_real_is_reference_counting (ValaTypeSymbol* base) {
	ValaClass * self;
	char* _tmp0;
	gboolean _tmp1;
	self = ((ValaClass*) (base));
	_tmp0 = NULL;
	return (_tmp1 = (_tmp0 = vala_typesymbol_get_ref_function (((ValaTypeSymbol*) (self)))) != NULL, (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
}


gboolean vala_class_is_fundamental (ValaClass* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	if (!vala_class_get_is_compact (self) && self->priv->_base_class == NULL) {
		return TRUE;
	}
	return FALSE;
}


static char* vala_class_real_get_ref_function (ValaTypeSymbol* base) {
	ValaClass * self;
	self = ((ValaClass*) (base));
	if (self->priv->ref_function == NULL && vala_class_is_fundamental (self)) {
		char* _tmp1;
		char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->ref_function = (_tmp1 = g_strconcat ((_tmp0 = vala_symbol_get_lower_case_cprefix (((ValaSymbol*) (self)))), "ref", NULL), (self->priv->ref_function = (g_free (self->priv->ref_function), NULL)), _tmp1);
		_tmp0 = (g_free (_tmp0), NULL);
	}
	if (self->priv->ref_function == NULL && self->priv->_base_class != NULL) {
		return vala_typesymbol_get_ref_function (((ValaTypeSymbol*) (self->priv->_base_class)));
	} else {
		const char* _tmp3;
		_tmp3 = NULL;
		return (_tmp3 = self->priv->ref_function, (_tmp3 == NULL ? NULL : g_strdup (_tmp3)));
	}
}


void vala_class_set_ref_function (ValaClass* self, const char* name) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->ref_function = (_tmp1 = (_tmp0 = name, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (self->priv->ref_function = (g_free (self->priv->ref_function), NULL)), _tmp1);
}


static char* vala_class_real_get_unref_function (ValaTypeSymbol* base) {
	ValaClass * self;
	self = ((ValaClass*) (base));
	if (self->priv->unref_function == NULL && vala_class_is_fundamental (self)) {
		char* _tmp1;
		char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->unref_function = (_tmp1 = g_strconcat ((_tmp0 = vala_symbol_get_lower_case_cprefix (((ValaSymbol*) (self)))), "unref", NULL), (self->priv->unref_function = (g_free (self->priv->unref_function), NULL)), _tmp1);
		_tmp0 = (g_free (_tmp0), NULL);
	}
	if (self->priv->unref_function == NULL && self->priv->_base_class != NULL) {
		return vala_typesymbol_get_unref_function (((ValaTypeSymbol*) (self->priv->_base_class)));
	} else {
		const char* _tmp3;
		_tmp3 = NULL;
		return (_tmp3 = self->priv->unref_function, (_tmp3 == NULL ? NULL : g_strdup (_tmp3)));
	}
}


void vala_class_set_unref_function (ValaClass* self, const char* name) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->unref_function = (_tmp1 = (_tmp0 = name, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (self->priv->unref_function = (g_free (self->priv->unref_function), NULL)), _tmp1);
}


static char* vala_class_real_get_dup_function (ValaTypeSymbol* base) {
	ValaClass * self;
	const char* _tmp0;
	self = ((ValaClass*) (base));
	_tmp0 = NULL;
	return (_tmp0 = self->priv->copy_function, (_tmp0 == NULL ? NULL : g_strdup (_tmp0)));
}


void vala_class_set_dup_function (ValaClass* self, const char* name) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->copy_function = (_tmp1 = (_tmp0 = name, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (self->priv->copy_function = (g_free (self->priv->copy_function), NULL)), _tmp1);
}


char* vala_class_get_default_free_function (ValaClass* self) {
	char* _tmp0;
	char* _tmp1;
	g_return_val_if_fail (self != NULL, NULL);
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = g_strconcat ((_tmp0 = vala_symbol_get_lower_case_cprefix (((ValaSymbol*) (self)))), "free", NULL), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
}


static char* vala_class_real_get_free_function (ValaTypeSymbol* base) {
	ValaClass * self;
	const char* _tmp1;
	self = ((ValaClass*) (base));
	if (self->priv->free_function == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->free_function = (_tmp0 = vala_class_get_default_free_function (self), (self->priv->free_function = (g_free (self->priv->free_function), NULL)), _tmp0);
	}
	_tmp1 = NULL;
	return (_tmp1 = self->priv->free_function, (_tmp1 == NULL ? NULL : g_strdup (_tmp1)));
}


void vala_class_set_free_function (ValaClass* self, const char* name) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (name != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->free_function = (_tmp1 = (_tmp0 = name, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (self->priv->free_function = (g_free (self->priv->free_function), NULL)), _tmp1);
}


static gboolean vala_class_real_is_subtype_of (ValaTypeSymbol* base, ValaTypeSymbol* t) {
	ValaClass * self;
	self = ((ValaClass*) (base));
	g_return_val_if_fail (t != NULL, FALSE);
	if (VALA_TYPESYMBOL (self) == t) {
		return TRUE;
	}
	{
		GeeList* base_type_collection;
		int base_type_it;
		base_type_collection = self->priv->base_types;
		for (base_type_it = 0; base_type_it < gee_collection_get_size (GEE_COLLECTION (base_type_collection)); base_type_it = base_type_it + 1) {
			ValaDataType* base_type;
			base_type = ((ValaDataType*) (gee_list_get (GEE_LIST (base_type_collection), base_type_it)));
			{
				if (vala_data_type_get_data_type (base_type) != NULL && vala_typesymbol_is_subtype_of (vala_data_type_get_data_type (base_type), t)) {
					gboolean _tmp1;
					return (_tmp1 = TRUE, (base_type == NULL ? NULL : (base_type = (vala_code_node_unref (base_type), NULL))), _tmp1);
				}
				(base_type == NULL ? NULL : (base_type = (vala_code_node_unref (base_type), NULL)));
			}
		}
	}
	return FALSE;
}


static gint vala_class_real_get_type_parameter_index (ValaTypeSymbol* base, const char* name) {
	ValaClass * self;
	gint i;
	self = ((ValaClass*) (base));
	g_return_val_if_fail (name != NULL, 0);
	i = 0;
	{
		GeeList* parameter_collection;
		int parameter_it;
		parameter_collection = self->priv->type_parameters;
		for (parameter_it = 0; parameter_it < gee_collection_get_size (GEE_COLLECTION (parameter_collection)); parameter_it = parameter_it + 1) {
			ValaTypeParameter* parameter;
			parameter = ((ValaTypeParameter*) (gee_list_get (GEE_LIST (parameter_collection), parameter_it)));
			{
				if (_vala_strcmp0 (vala_symbol_get_name (((ValaSymbol*) (parameter))), name) == 0) {
					gint _tmp0;
					return (_tmp0 = i, (parameter == NULL ? NULL : (parameter = (vala_code_node_unref (parameter), NULL))), _tmp0);
				}
				i++;
				(parameter == NULL ? NULL : (parameter = (vala_code_node_unref (parameter), NULL)));
			}
		}
	}
	return -1;
}


static void vala_class_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaClass * self;
	self = ((ValaClass*) (base));
	g_return_if_fail (old_type != NULL);
	g_return_if_fail (new_type != NULL);
	{
		gint i;
		i = 0;
		for (; i < gee_collection_get_size (((GeeCollection*) (self->priv->base_types))); i++) {
			ValaDataType* _tmp0;
			gboolean _tmp1;
			_tmp0 = NULL;
			if ((_tmp1 = (_tmp0 = ((ValaDataType*) (gee_list_get (((GeeList*) (self->priv->base_types)), i)))) == old_type, (_tmp0 == NULL ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL))), _tmp1)) {
				gee_list_set (((GeeList*) (self->priv->base_types)), i, new_type);
				return;
			}
		}
	}
}


ValaClass* vala_class_get_base_class (ValaClass* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_base_class;
}


void vala_class_set_base_class (ValaClass* self, ValaClass* value) {
	ValaClass* _tmp2;
	ValaClass* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_base_class = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_base_class == NULL ? NULL : (self->priv->_base_class = (vala_code_node_unref (self->priv->_base_class), NULL))), _tmp2);
}


gboolean vala_class_get_is_abstract (ValaClass* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_is_abstract;
}


void vala_class_set_is_abstract (ValaClass* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_abstract = value;
}


gboolean vala_class_get_is_compact (ValaClass* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	if (self->priv->_base_class != NULL) {
		return vala_class_get_is_compact (self->priv->_base_class);
	}
	return self->priv->_is_compact;
}


void vala_class_set_is_compact (ValaClass* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_compact = value;
}


gboolean vala_class_get_is_immutable (ValaClass* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	if (self->priv->_base_class != NULL) {
		return vala_class_get_is_immutable (self->priv->_base_class);
	}
	return self->priv->_is_immutable;
}


void vala_class_set_is_immutable (ValaClass* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_immutable = value;
}


const char* vala_class_get_type_check_function (ValaClass* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_type_check_function;
}


void vala_class_set_type_check_function (ValaClass* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_type_check_function = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (self->priv->_type_check_function = (g_free (self->priv->_type_check_function), NULL)), _tmp2);
}


gboolean vala_class_get_has_private_fields (ValaClass* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_has_private_fields;
}


static void vala_class_set_has_private_fields (ValaClass* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_has_private_fields = value;
}


ValaMethod* vala_class_get_default_construction_method (ValaClass* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_default_construction_method;
}


void vala_class_set_default_construction_method (ValaClass* self, ValaMethod* value) {
	ValaMethod* _tmp2;
	ValaMethod* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_default_construction_method = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_default_construction_method == NULL ? NULL : (self->priv->_default_construction_method = (vala_code_node_unref (self->priv->_default_construction_method), NULL))), _tmp2);
}


ValaConstructor* vala_class_get_constructor (ValaClass* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_constructor;
}


void vala_class_set_constructor (ValaClass* self, ValaConstructor* value) {
	ValaConstructor* _tmp2;
	ValaConstructor* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_constructor = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_constructor == NULL ? NULL : (self->priv->_constructor = (vala_code_node_unref (self->priv->_constructor), NULL))), _tmp2);
}


ValaConstructor* vala_class_get_class_constructor (ValaClass* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_class_constructor;
}


void vala_class_set_class_constructor (ValaClass* self, ValaConstructor* value) {
	ValaConstructor* _tmp2;
	ValaConstructor* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_class_constructor = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_class_constructor == NULL ? NULL : (self->priv->_class_constructor = (vala_code_node_unref (self->priv->_class_constructor), NULL))), _tmp2);
}


ValaConstructor* vala_class_get_static_constructor (ValaClass* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_static_constructor;
}


void vala_class_set_static_constructor (ValaClass* self, ValaConstructor* value) {
	ValaConstructor* _tmp2;
	ValaConstructor* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_static_constructor = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_static_constructor == NULL ? NULL : (self->priv->_static_constructor = (vala_code_node_unref (self->priv->_static_constructor), NULL))), _tmp2);
}


ValaDestructor* vala_class_get_destructor (ValaClass* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_destructor;
}


void vala_class_set_destructor (ValaClass* self, ValaDestructor* value) {
	ValaDestructor* _tmp2;
	ValaDestructor* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_destructor = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_destructor == NULL ? NULL : (self->priv->_destructor = (vala_code_node_unref (self->priv->_destructor), NULL))), _tmp2);
	if (self->priv->_destructor != NULL) {
		ValaFormalParameter* _tmp4;
		ValaObjectType* _tmp3;
		if (vala_destructor_get_this_parameter (self->priv->_destructor) != NULL) {
			vala_scope_remove (vala_symbol_get_scope (((ValaSymbol*) (self->priv->_destructor))), vala_symbol_get_name (((ValaSymbol*) (vala_destructor_get_this_parameter (self->priv->_destructor)))));
		}
		_tmp4 = NULL;
		_tmp3 = NULL;
		vala_destructor_set_this_parameter (self->priv->_destructor, (_tmp4 = vala_formal_parameter_new ("this", ((ValaDataType*) ((_tmp3 = vala_object_type_new (((ValaObjectTypeSymbol*) (self)))))), NULL)));
		(_tmp4 == NULL ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL)));
		(_tmp3 == NULL ? NULL : (_tmp3 = (vala_code_node_unref (_tmp3), NULL)));
		vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) (self->priv->_destructor))), vala_symbol_get_name (((ValaSymbol*) (vala_destructor_get_this_parameter (self->priv->_destructor)))), ((ValaSymbol*) (vala_destructor_get_this_parameter (self->priv->_destructor))));
	}
}


gboolean vala_class_get_is_error_base (ValaClass* self) {
	ValaAttribute* _tmp0;
	gboolean _tmp1;
	g_return_val_if_fail (self != NULL, FALSE);
	_tmp0 = NULL;
	return (_tmp1 = (_tmp0 = vala_code_node_get_attribute (((ValaCodeNode*) (self)), "ErrorBase")) != NULL, (_tmp0 == NULL ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL))), _tmp1);
}


static void vala_class_class_init (ValaClassClass * klass) {
	vala_class_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_class_finalize;
	g_type_class_add_private (klass, sizeof (ValaClassPrivate));
	VALA_OBJECT_TYPE_SYMBOL_CLASS (klass)->get_methods = vala_class_real_get_methods;
	VALA_OBJECT_TYPE_SYMBOL_CLASS (klass)->get_properties = vala_class_real_get_properties;
	VALA_OBJECT_TYPE_SYMBOL_CLASS (klass)->get_signals = vala_class_real_get_signals;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_class_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_class_real_accept_children;
	VALA_SYMBOL_CLASS (klass)->get_cprefix = vala_class_real_get_cprefix;
	VALA_TYPESYMBOL_CLASS (klass)->get_cname = vala_class_real_get_cname;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cname = vala_class_real_get_lower_case_cname;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cprefix = vala_class_real_get_lower_case_cprefix;
	VALA_TYPESYMBOL_CLASS (klass)->get_upper_case_cname = vala_class_real_get_upper_case_cname;
	VALA_TYPESYMBOL_CLASS (klass)->get_type_signature = vala_class_real_get_type_signature;
	VALA_TYPESYMBOL_CLASS (klass)->is_reference_type = vala_class_real_is_reference_type;
	VALA_TYPESYMBOL_CLASS (klass)->get_type_id = vala_class_real_get_type_id;
	VALA_TYPESYMBOL_CLASS (klass)->get_marshaller_type_name = vala_class_real_get_marshaller_type_name;
	VALA_TYPESYMBOL_CLASS (klass)->get_param_spec_function = vala_class_real_get_param_spec_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_get_value_function = vala_class_real_get_get_value_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_set_value_function = vala_class_real_get_set_value_function;
	VALA_TYPESYMBOL_CLASS (klass)->is_reference_counting = vala_class_real_is_reference_counting;
	VALA_TYPESYMBOL_CLASS (klass)->get_ref_function = vala_class_real_get_ref_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_unref_function = vala_class_real_get_unref_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_dup_function = vala_class_real_get_dup_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_free_function = vala_class_real_get_free_function;
	VALA_TYPESYMBOL_CLASS (klass)->is_subtype_of = vala_class_real_is_subtype_of;
	VALA_TYPESYMBOL_CLASS (klass)->get_type_parameter_index = vala_class_real_get_type_parameter_index;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_class_real_replace_type;
}


static void vala_class_instance_init (ValaClass * self) {
	self->priv = VALA_CLASS_GET_PRIVATE (self);
	self->priv->type_parameters = ((GeeList*) (gee_array_list_new (VALA_TYPE_TYPEPARAMETER, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal)));
	self->priv->base_types = ((GeeList*) (gee_array_list_new (VALA_TYPE_DATA_TYPE, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal)));
	self->priv->constants = ((GeeList*) (gee_array_list_new (VALA_TYPE_CONSTANT, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal)));
	self->priv->fields = ((GeeList*) (gee_array_list_new (VALA_TYPE_FIELD, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal)));
	self->priv->methods = ((GeeList*) (gee_array_list_new (VALA_TYPE_METHOD, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal)));
	self->priv->properties = ((GeeList*) (gee_array_list_new (VALA_TYPE_PROPERTY, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal)));
	self->priv->signals = ((GeeList*) (gee_array_list_new (VALA_TYPE_SIGNAL, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal)));
	self->priv->classes = ((GeeList*) (gee_array_list_new (VALA_TYPE_CLASS, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal)));
	self->priv->structs = ((GeeList*) (gee_array_list_new (VALA_TYPE_STRUCT, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal)));
	self->priv->enums = ((GeeList*) (gee_array_list_new (VALA_TYPE_ENUM, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal)));
	self->priv->delegates = ((GeeList*) (gee_array_list_new (VALA_TYPE_DELEGATE, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal)));
}


static void vala_class_finalize (ValaCodeNode* obj) {
	ValaClass * self;
	self = VALA_CLASS (obj);
	(self->priv->_base_class == NULL ? NULL : (self->priv->_base_class = (vala_code_node_unref (self->priv->_base_class), NULL)));
	self->priv->_type_check_function = (g_free (self->priv->_type_check_function), NULL);
	self->priv->cname = (g_free (self->priv->cname), NULL);
	self->priv->const_cname = (g_free (self->priv->const_cname), NULL);
	self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL);
	self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL);
	self->priv->type_id = (g_free (self->priv->type_id), NULL);
	self->priv->ref_function = (g_free (self->priv->ref_function), NULL);
	self->priv->unref_function = (g_free (self->priv->unref_function), NULL);
	self->priv->param_spec_function = (g_free (self->priv->param_spec_function), NULL);
	self->priv->copy_function = (g_free (self->priv->copy_function), NULL);
	self->priv->free_function = (g_free (self->priv->free_function), NULL);
	self->priv->marshaller_type_name = (g_free (self->priv->marshaller_type_name), NULL);
	self->priv->get_value_function = (g_free (self->priv->get_value_function), NULL);
	self->priv->set_value_function = (g_free (self->priv->set_value_function), NULL);
	self->priv->type_signature = (g_free (self->priv->type_signature), NULL);
	(self->priv->type_parameters == NULL ? NULL : (self->priv->type_parameters = (gee_collection_object_unref (self->priv->type_parameters), NULL)));
	(self->priv->base_types == NULL ? NULL : (self->priv->base_types = (gee_collection_object_unref (self->priv->base_types), NULL)));
	(self->priv->constants == NULL ? NULL : (self->priv->constants = (gee_collection_object_unref (self->priv->constants), NULL)));
	(self->priv->fields == NULL ? NULL : (self->priv->fields = (gee_collection_object_unref (self->priv->fields), NULL)));
	(self->priv->methods == NULL ? NULL : (self->priv->methods = (gee_collection_object_unref (self->priv->methods), NULL)));
	(self->priv->properties == NULL ? NULL : (self->priv->properties = (gee_collection_object_unref (self->priv->properties), NULL)));
	(self->priv->signals == NULL ? NULL : (self->priv->signals = (gee_collection_object_unref (self->priv->signals), NULL)));
	(self->priv->classes == NULL ? NULL : (self->priv->classes = (gee_collection_object_unref (self->priv->classes), NULL)));
	(self->priv->structs == NULL ? NULL : (self->priv->structs = (gee_collection_object_unref (self->priv->structs), NULL)));
	(self->priv->enums == NULL ? NULL : (self->priv->enums = (gee_collection_object_unref (self->priv->enums), NULL)));
	(self->priv->delegates == NULL ? NULL : (self->priv->delegates = (gee_collection_object_unref (self->priv->delegates), NULL)));
	(self->priv->_default_construction_method == NULL ? NULL : (self->priv->_default_construction_method = (vala_code_node_unref (self->priv->_default_construction_method), NULL)));
	(self->priv->_constructor == NULL ? NULL : (self->priv->_constructor = (vala_code_node_unref (self->priv->_constructor), NULL)));
	(self->priv->_class_constructor == NULL ? NULL : (self->priv->_class_constructor = (vala_code_node_unref (self->priv->_class_constructor), NULL)));
	(self->priv->_static_constructor == NULL ? NULL : (self->priv->_static_constructor = (vala_code_node_unref (self->priv->_static_constructor), NULL)));
	(self->priv->_destructor == NULL ? NULL : (self->priv->_destructor = (vala_code_node_unref (self->priv->_destructor), NULL)));
	VALA_CODE_NODE_CLASS (vala_class_parent_class)->finalize (obj);
}


GType vala_class_get_type (void) {
	static GType vala_class_type_id = 0;
	if (vala_class_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaClassClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_class_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaClass), 0, (GInstanceInitFunc) vala_class_instance_init, NULL };
		vala_class_type_id = g_type_register_static (VALA_TYPE_OBJECT_TYPE_SYMBOL, "ValaClass", &g_define_type_info, 0);
	}
	return vala_class_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if (array != NULL && destroy_func != NULL) {
		int i;
		if (array_length >= 0)
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) (array))[i] != NULL)
			destroy_func (((gpointer*) (array))[i]);
		}
		else
		for (i = 0; ((gpointer*) (array))[i] != NULL; i = i + 1) {
			destroy_func (((gpointer*) (array))[i]);
		}
	}
	g_free (array);
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return (str1 != str2);
	}
	return strcmp (str1, str2);
}




