<?php
// $Horde: horde/lib/Lang.php,v 1.13.2.1 2001/12/23 13:46:17 jan Exp $

/**
 * The Lang:: class provides common methods for handling language detection
 * and selection.
 *
 * @author  Jon Parise <jon@horde.org>
 * @author  Chuck Hagenbuch <chuck@horde.org>
 * @version $Revision: 1.13.2.1 $
 * @since   Horde 1.3
 * @package horde
 */
class Lang {

    /**
     * Selects the most preferred language for the current client session.
     *
     * @return string        The selected language abbreviation.
     * @access public
     */
    function select()
    {
        global $HTTP_SERVER_VARS, $nls;
    
        /* Use a directly supplied parameter, if given. */
        $language = Lang::_map(Horde::getFormData('language'));
        
        if (isset($language) && Lang::isValid($language)) {
            // We already have a valid language; no more work needed.

        /* Try browser-accepted languages, then default. */
        } elseif (isset($HTTP_SERVER_VARS['HTTP_ACCEPT_LANGUAGE'])) {

            /* Start with a known default value. */
            $language = $nls['defaults']['language'];

            /* The browser supplies a list, so return the first valid one. */
            $browser_langs = explode(',', $HTTP_SERVER_VARS['HTTP_ACCEPT_LANGUAGE']);
            foreach ($browser_langs as $lang) {
                $lang = Lang::_map(trim($lang));
                if (Lang::isValid($lang)) {
                    $language = $lang;
                    break;
                } elseif (Lang::isValid(Lang::_map(substr($lang, 0, 2)))) {
                    $language = Lang::_map(substr($lang, 0, 2));
                    break;
                }
            }

        /* No dice auto-detecting, so give them the server default. */
        } else {
            $language = $nls['defaults']['language'];
        }

        return basename($language);
    }
    
    /**
     * Sets the language.
     *
     * @param string $lang          (optional) The language abbriviation
     * @access public
     */
    function setLang($lang = null)
    {
        if (@file_exists(HORDE_BASE . '/config/lang.php')) {
            include_once HORDE_BASE . '/config/lang.php';
        } else {
            include_once HORDE_BASE . '/config/lang.php.dist';
        }
        if (!isset($lang) || !Lang::isValid($lang)) {
            $lang = Lang::select();
        }
        $GLOBALS['language'] = $lang;
        putenv('LANG=' . $lang);
        putenv('LANGUAGE=' . $lang);
        @setlocale(LC_ALL, $lang);
    }
        
    /**
     * Sets the gettext domain.
     *
     * @param string $app           The application name
     * @param string $directory     The directory where the application's
     *                              LC_MESSAGES directory resides
     * @param string $charset       The charset
     */
    function setDomain($app, $directory, $charset)
    {
        bindtextdomain($app, $directory);
        textdomain($app);
        if (function_exists('bind_textdomain_codeset')) {
            bind_textdomain_codeset($app, $charset); 
        }
    }
    
    /**
     * Determines whether the supplied language is valid.
     *
     * @param string $language         The abbreviated name of the language.
     *
     * @return  boolean         True if the language is valid, false if it's
     *                          not valid or unknown.
     * @access public
     */
    function isValid($language)
    {
        return !empty($GLOBALS['nls']['languages'][$language]);
    }
    
    /**
     * Maps languages with common two-letter codes (such as nl) to the
     * full gettext code (in this case, nl_NL). Returns the language
     * unmodified if it isn't an alias.
     *
     * @param string $language   The language code to map.
     * @return string            The mapped language code.
     * @access private
     */
	
	function _map($language)
    {
        $aliases = &$GLOBALS['nls']['aliases'];
        
        // First check if the untranslated language can be found
	    if (!empty($aliases[$language])) {
            return $aliases[$language];
        }
        
        // Translate the $language to get broader matches
	    // eg. de-DE should match de_DE
        $trans_lang = str_replace('-', '_', $language);
        
        // See if we get a match for this
	    if (!empty($aliases[$trans_lang])) {
            return $aliases[$trans_lang];
        }
        
        // Next, see if we get a match if we do
	    // a case insensitive match
        
        // Transform the language to upper case
	    $trans_lang_UC = strtoupper($trans_lang);
        
        // Also transform all the aliases to upper case
	    // and check if the $trans_lang_UC can be found
        $aliases_uc = array();
        foreach ($aliases as $alias => $lang) {
            $aliases_uc[strtoupper($alias)] = $lang;
            if (!empty($aliases_uc[$trans_lang_UC])) {
                return $aliases_uc[$trans_lang_UC];
            }
        }
        
        // If we get that far down, the language cannot be found.
        // Return the $language
        return $trans_lang;
	}

}
?>
