/*
   Copyright (C) 2000-2002  Ulric Eriksson <ulric@siag.nu>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.
*/

#include <stdio.h>
#include <ctype.h>
#include <errno.h>
#include <netdb.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <signal.h>
#include <stdlib.h>
#include <stdarg.h>
#include <ctype.h>
#include <time.h>
#include <sys/wait.h>
#include <sys/resource.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <syslog.h>
#include <string.h>

#include "config.h"

#define BUFFER_MAX 	(32*1024)

#define CLIENTS_MAX	2048	/* max clients */
#define CONNECTIONS_MAX	256	/* max simultaneous connections */
#define TIMEOUT		5	/* default timeout for non reachable hosts */
#define BLACKLIST_TIME	30	/* how long to shun a server that is down */
#define KEEP_MAX	100	/* how much to keep from the URI */

typedef struct {
	int downfd, upfd;
	unsigned char *downb, *upb;
	int downn, upn;
	int clt;
	int index;		/* server index */
} connection;

typedef struct {
	int status;		/* last failed connection attempt */
	int port;
	struct in_addr addr;
	int c;			/* connections */
	int maxc;		/* max connections */
	unsigned long sx, rx;	/* bytes sent, received */
} server;

typedef struct {
	time_t last;		/* last time this client made a connection */
	struct in_addr addr;	/* of client */
	int cno;		/* server used last time */
} client;

static int nservers;		/* number of servers */
static int current;		/* current server */
static client *clients;
static server *servers;
static connection *conns;

static int debuglevel;
static int asciidump;
static int foreground;
static int loopflag;

static int clients_max = CLIENTS_MAX;
static int connections_max = CONNECTIONS_MAX;
static int timeout = TIMEOUT;
static int blacklist_time = BLACKLIST_TIME;
static int roundrobin = 0;
static int hash = 0;
static int stubborn = 0;
static int nblock = 1;

static int port;

static char *logfile = NULL;
static FILE *logfp = NULL;
static char *pidfile = NULL;
static FILE *pidfp = NULL;
static char *webfile = NULL;

static void debug(char *fmt, ...)
{
	char b[4096];
	va_list ap;
	va_start(ap, fmt);
	vsnprintf(b, sizeof b, fmt, ap);
	if (foreground) {
		fprintf(stderr, "%s\n", b);
	} else {
		openlog("pen", LOG_CONS, LOG_USER);
		syslog(LOG_DEBUG, "%s\n", b);
		closelog();
	}
	va_end(ap);
}

static void error(char *fmt, ...)
{
	char b[4096];
	va_list ap;
	va_start(ap, fmt);
	vsnprintf(b, sizeof b, fmt, ap);
	if (foreground) {
		fprintf(stderr, "%s\n", b);
	} else {
		openlog("pen", LOG_CONS, LOG_USER);
		syslog(LOG_ERR, "%s\n", b);
		closelog();
	}
	va_end(ap);
	exit(1);
}

static void webstats(void)
{
	FILE *fp;
	int i;

	fp = fopen(webfile, "w");
	if (fp == NULL) return;
	fprintf(fp,
		"<html>\n"
		"<head>\n"
		"<title>Pen status page</title>\n"
		"</head>\n"
		"<body bgcolor=\"#ffffff\">"
		"<h1>Pen status page</h1>\n");
	fprintf(fp,
		"Time %d, %d servers, %d current<p>\n",
		(int)time(NULL), nservers, current);
	fprintf(fp,
		"<table bgcolor=\"#c0c0c0\">\n"
		"<tr>\n"
		"<td bgcolor=\"#80f080\">server</td>\n"
		"<td bgcolor=\"#80f080\">address</td>\n"
		"<td bgcolor=\"#80f080\">status</td>\n"
		"<td bgcolor=\"#80f080\">port</td>\n"
		"<td bgcolor=\"#80f080\">connections</td>\n"
		"<td bgcolor=\"#80f080\">max conn</td>\n"
		"<td bgcolor=\"#80f080\">sent</td>\n"
		"<td bgcolor=\"#80f080\">received</td>\n"
		"</tr>\n");
	for (i = 0; i < nservers; i++) {
		fprintf(fp,
			"<tr>\n"
			"<td>%d</td>\n"
			"<td>%s</td>\n"
			"<td>%d</td>\n"
			"<td>%d</td>\n"
			"<td>%d</td>\n"
			"<td>%d</td>\n"
			"<td>%ld</td>\n"
			"<td>%ld</td>\n"
			"</tr>\n",
			i, inet_ntoa(servers[i].addr),
			servers[i].status, servers[i].port,
			servers[i].c, servers[i].maxc,
			servers[i].sx, servers[i].rx);
	}
	fprintf(fp, "</table>\n");

	fprintf(fp, "<h2>Active clients</h2>");
	fprintf(fp, "Max number of clients: %d<p>", CLIENTS_MAX);
	fprintf(fp,
		"<table bgcolor=\"#c0c0c0\">\n"
		"<tr>\n"
		"<td bgcolor=\"#80f080\">client</td>\n"
		"<td bgcolor=\"#80f080\">address</td>\n"
		"<td bgcolor=\"#80f080\">last used</td>\n"
		"<td bgcolor=\"#80f080\">last server</td>\n"
		"</tr>\n");
	for (i = 0; i < CLIENTS_MAX; i++) {
		if (clients[i].last == 0) continue;
		fprintf(fp,
			"<tr>\n"
			"<td>%d</td>\n"
			"<td>%s</td>\n"
			"<td>%ld</td>\n"
			"<td>%d</td>\n"
			"</tr>\n",
			i, inet_ntoa(clients[i].addr),
			(long)clients[i].last, clients[i].cno);
	}
	fprintf(fp, "</table>\n");

	fprintf(fp, "<h2>Active connections</h2>");
	fprintf(fp, "Max number of connections: %d<p>", CONNECTIONS_MAX);
	fprintf(fp,
		"<table bgcolor=\"#c0c0c0\">\n"
		"<tr>\n"
		"<td bgcolor=\"#80f080\">connection</td>\n"
		"<td bgcolor=\"#80f080\">downfd</td>\n"
		"<td bgcolor=\"#80f080\">upfd</td>\n"
		"<td bgcolor=\"#80f080\">pending data down</td>\n"
		"<td bgcolor=\"#80f080\">pending data up</td>\n"
		"<td bgcolor=\"#80f080\">client</td>\n"
		"<td bgcolor=\"#80f080\">server</td>\n"
		"</tr>\n");
	for (i = 0; i < CONNECTIONS_MAX; i++) {
		if (conns[i].downfd == -1) continue;
		fprintf(fp,
			"<tr>\n"
			"<td>%d</td>\n"
			"<td>%d</td>\n"
			"<td>%d</td>\n"
			"<td>%d</td>\n"
			"<td>%d</td>\n"
			"<td>%d</td>\n"
			"<td>%d</td>\n"
			"</tr>\n",
			i, conns[i].downfd, conns[i].upfd,
			conns[i].downn, conns[i].upn,
			conns[i].clt, conns[i].index);
	}
	fprintf(fp, "</table>\n");
	fprintf(fp,
		"</body>\n"
		"</html>\n");
	fclose(fp);
}

static void textstats(void)
{
	int i;

	debug("Time %d, %d servers, %d current",
		(int)time(NULL), nservers, current);
	for (i = 0; i < nservers; i++) {
		debug("Server %d status:\n"
			"address %s\n"
			"%d\n"
			"port %d\n"
			"%d connections (%d max)\n"
			"%ld sent, %ld received\n",
			i, inet_ntoa(servers[i].addr),
			servers[i].status, servers[i].port,
			servers[i].c, servers[i].maxc,
			servers[i].sx, servers[i].rx);
	}
	debug("Max number of clients: %d", CLIENTS_MAX);
	debug("Active clients:");
	for (i = 0; i < CLIENTS_MAX; i++) {
		if (clients[i].last == 0) continue;
		debug("Client %d status:\n"
			"address %s\n"
			"last used %ld\n"
			"last server %d\n",
			i, inet_ntoa(clients[i].addr),
			(long)clients[i].last, clients[i].cno);
	}
	debug("Max number of connections: %d", CONNECTIONS_MAX);
	debug("Active connections:");
	for (i = 0; i < CONNECTIONS_MAX; i++) {
		if (conns[i].downfd == -1) continue;
		debug("Connection %d status:\n"
			"downfd = %d, upfd = %d\n"
			"pending data %d down, %d up\n"
			"client %d, server %d\n",
			i, conns[i].downfd, conns[i].upfd,
			conns[i].downn, conns[i].upn,
			conns[i].clt, conns[i].index);
	}
}

static void stats(int dummy)
{
	if (webfile) webstats();
	else textstats();
	signal(SIGUSR1, stats);
}

static void restart_log(int dummy)
{
	if (logfp) {
		fclose(logfp);
		logfp = fopen(logfile, "a");
		if (!logfp) error("Can't open logfile %s", logfile);
	}
	signal(SIGHUP, restart_log);
}

static void quit(int dummy)
{
	loopflag = 0;
}

/* Return index of known client, otherwise 0 */
static int lookup_client(struct in_addr cli)
{
	int i;
	unsigned long ad = cli.s_addr;

	for (i = 0; i < clients_max; i++) {
		if (clients[i].addr.s_addr == ad) break;
	}
	if (i == clients_max) i = -1;
	if (debuglevel) debug("Client %s has index %d\n", inet_ntoa(cli), i);
	return i;
}

/* Store client and return index */
static int store_client(struct in_addr cli, int ch)
{
	int i = -1;
	int empty = -1;		/* first empty slot */
	int oldest = -1;	/* in case we need to recycle */

	for (i = 0; i < clients_max; i++) {
		if (clients[i].addr.s_addr == cli.s_addr) break;
		if (empty != -1) continue;
		if (clients[i].last == 0) {
			empty = i;
			continue;
		}
		if (oldest == -1 || (clients[i].last < clients[oldest].last)) {
			oldest = i;
		}
	}

	if (i == clients_max) {
		if (empty != -1) i = empty;
		else i = oldest;
	}

	clients[i].last = time(NULL);
	clients[i].addr = cli;
	clients[i].cno = ch;

	if (debuglevel) {
		debug("Client %s has index %d and server %d\n",
			inet_ntoa(cli), i, ch);
	}
	servers[ch].c++;

	return i;
}

static void dump(unsigned char *p, int n)
{
	int i;

	fprintf(stderr, "%d: ", n);
	for (i = 0; i < n; i++) {
		if (asciidump) {
			fprintf(stderr, "%c",
				(isprint(p[i])||isspace(p[i]))?p[i]:'.');
		} else {
			fprintf(stderr, "%02x ", (int)p[i]);
		}
	}
	fprintf(stderr, "\n");
}


static int getport(char *p)
{
	struct servent *s = getservbyname(p, "tcp");
	if (s == NULL) {
		return atoi(p);
	} else {
		return ntohs(s->s_port);
	}
}

static void setipaddress(struct in_addr *a, char *p)
{
	struct hostent *h = gethostbyname(p);
	if (h == NULL) {
		if ((a->s_addr = inet_addr(p)) == -1) {
			error("unknown or invalid address [%s]\n", p);
		}
	} else {
		memcpy(a, h->h_addr, h->h_length);
	}
}

static void setaddress(struct in_addr *a, int *port, char *s,
		int dp, int *maxc)
{
	char *hs;
	char *ps;
	char *ms;

	char *p, *q;

	struct hostent *h;

	hs = s;

	if ((p = strchr(s, ':')) != NULL) {
		*p = '\0';
		ps = p + 1;
		if ((q = strchr(ps, ':')) != NULL) {
			*q = '\0';
			ms = q + 1;
		} else {
			ms = "";
		}
	} else {
		ps = "";
		ms = "";
	}

	if (!strcmp(ps, "")) ps = NULL;
	if (!strcmp(ms, "")) ms = NULL;

	if (!(h = gethostbyname(s))) {
		if ((a->s_addr = inet_addr(s)) == -1) {
			error("unknown or invalid address [%s]\n", s);
		}
	} else {
		memcpy(a, h->h_addr, h->h_length);
	}

	if (ps) *port = getport(ps);
	else *port = dp;

	if (ms) *maxc = atoi(ms);
}

static void log(FILE *fp, int i, unsigned char *b, int n)
{
	int j;
	if (n > KEEP_MAX) n = KEEP_MAX;
	fprintf(fp, "%s ", inet_ntoa(clients[conns[i].clt].addr));
	fprintf(fp, "%ld ", (long)time(NULL));
	fprintf(fp, "%s ", inet_ntoa(servers[conns[i].index].addr));
	for (j = 0; j < n && b[j] != '\r' && b[j] != '\n'; j++) {
		fprintf(fp, "%c", isascii(b[j])?b[j]:'.');
	}
	fprintf(fp, "\n");
}

static int copy_up(int i)
{
	int rc;
	int from = conns[i].downfd;
	int to = conns[i].upfd;
	int serverindex = conns[i].index;

	char b[BUFFER_MAX];

	rc = read(from, b, BUFFER_MAX);

	if (debuglevel) debug("copy_up(%d)", i);
	if (debuglevel >= 2) dump(b, rc);

	if (rc <= 0) {
		if (nblock && errno == EAGAIN) return 0;
		return (-1);
	} else {
		int n;
		if (logfp) {
			log(logfp, i, b, rc);
			if (debuglevel > 1) log(stderr, i, b, rc);
		}
		n = write(to, b, rc);
		if (n < 0) {
			if (!nblock || errno != EAGAIN) return -1;
			n = 0;
		}
		if (n != rc) {
			if (debuglevel) {
				debug("copy_up saving %d bytes in up buffer\n",
					rc-n);
			}
			conns[i].upn = rc-n;
			conns[i].upb = malloc(rc-n);
			memcpy(conns[i].upb, b+n, rc-n);
		}
		servers[serverindex].sx += rc;
	}
	return (0);
}

static int copy_down(int i)
{
	int rc;
	int from = conns[i].upfd;
	int to = conns[i].downfd;
	int serverindex = conns[i].index;

	char b[BUFFER_MAX];

	rc = read(from, b, BUFFER_MAX);

	if (debuglevel) debug("copy_down(%d)", i);
	if (debuglevel >= 2) dump(b, rc);

	if (rc <= 0) {
		if (nblock && errno == EAGAIN) return 0;
		return (-1);
	} else {
		int n = write(to, b, rc);
		if (n < 0) {
			if (!nblock || errno != EAGAIN) return -1;
			n = 0;
		}
		if (n != rc) {
			if (debuglevel) {
				debug("copy_down saving %d bytes in down buffer\n",
					rc-n);
			}
			conns[i].downn = rc-n;
			conns[i].downb = malloc(rc-n);
			memcpy(conns[i].downb, b+n, rc-n);
		}
		servers[serverindex].rx += rc;
	}
	return (0);
}

static void alarm_handler(int dummy)
{
	;
}

static void store_conn(int downfd, int upfd, int clt, int index)
{
	int i;

	if (nblock) {
		if (fcntl(downfd, F_SETFL, O_NONBLOCK) == -1)
			error("Can't fcntl, errno = %d", errno);
		if (fcntl(upfd, F_SETFL, O_NONBLOCK) == -1)
			error("Can't fcntl, errno = %d", errno);
	}
	for (i = 0; i < connections_max; i++) {
		if (conns[i].upfd == -1) break;
	}
	if (i < connections_max) {
		conns[i].upfd = upfd;
		conns[i].downfd = downfd;
		conns[i].clt = clt;
		conns[i].index = index;
		current = index;
	} else if (debuglevel) {
		debug("Connection table full (%d slots), can't store connection.\n"
		      "Try restarting with -x %d",
		      connections_max, 2*connections_max);
	}
}

static void close_conn(int i)
{
	int index = conns[i].index;
	servers[index].c -= 1;
	if (conns[i].upfd > 0) close(conns[i].upfd);
	if (conns[i].downfd > 0) close(conns[i].downfd);
	conns[i].upfd = conns[i].downfd = -1;
}

static int next_server(int i)
{
	i++;
	if (i >= nservers) i = 0;
	return i;
}

static void usage(void)
{
	printf("usage:\n"
	       "  pen [-b sec] [-c N] [-t sec] [-x N] [-w dir] [-adfhrs] \\\n"
	       "          [host:]port h1[:p1[:maxc1]] [h2[:p2[:maxc2]]] ...\n"
	       "\n"
	       "  -a        debugging dumps in ascii format\n"
	       "  -b sec    blacklist time in seconds [%d]\n"
	       "  -c N      max number of clients [%d]\n"
	       "  -d        debugging on (repeat -d for more)\n"
	       "  -f        stay in foregound\n"
	       "  -h        use hash for initial server selection\n"
	       "  -l file   logging on\n"
	       "  -n        do not make sockets nonblocking\n"
	       "  -r        bypass client tracking in server selection\n"
	       "  -s        stubborn selection, i.e. don't fail over\n"
	       "  -t sec    connect timeout in seconds [%d]\n"
	       "  -p file   write pid to file\n"
	       "  -x N      max number of simultaneous connections [%d]\n"
	       "  -w file   save statistics in HTML format in a file\n"
	       "\n"
	       "example:\n"
	       "  pen smtp mailhost1:smtp mailhost2:25 mailhost3\n"
	       "\n",
	       BLACKLIST_TIME, CLIENTS_MAX, TIMEOUT, CONNECTIONS_MAX);

	exit(0);
}

static void background(void)
{
#ifdef HAVE_DAEMON
	daemon(0, 0);
#else
	int childpid;
	if ((childpid = fork()) < 0) {
		error("Can't fork");
	} else {
		if (childpid > 0) {
			exit(0);	/* parent */
		}
	}
	setsid();
	signal(SIGCHLD, SIG_IGN);
#endif
}

static void init(int argc, char **argv)
{
	int i;
	int server;

	conns = calloc(connections_max, sizeof *conns);
	clients = calloc(clients_max, sizeof *clients);
	servers = calloc(argc, sizeof *servers);
	if (conns == NULL || clients == NULL || servers == NULL)
		error("Can't allocate memory");

	nservers = 0;
	current = 0;

	server = 0;

	for (i = 1; i < argc; i++) {
		servers[server].status = 0;
		servers[server].c = 0;	/* connections... */
		servers[server].maxc = 0;	/* max connections... */
		setaddress(&servers[server].addr, &servers[server].port,
			   argv[i], port, &servers[server].maxc);
		servers[server].sx = 0;
		servers[server].rx = 0;

		nservers++;
		server++;
	}

	for (i = 0; i < clients_max; i++) {
		clients[i].last = 0;
		clients[i].addr.s_addr = 0;
		clients[i].cno = 0;
	}
	for (i = 0; i < connections_max; i++) {
		conns[i].upfd = -1;
		conns[i].downfd = -1;
		conns[i].upn = 0;
		conns[i].downn = 0;
	}

	if (debuglevel) {
		debug("servers:");
		for (i = 0; i < nservers; i++) {
			debug("%2d %s:%d:%d", i,
				inet_ntoa(servers[i].addr),
				servers[i].port, servers[i].maxc);
		}
	}
}

/* return upstream file descriptor */
int try_server(int index)
{
	struct sockaddr_in serv_addr;
        int upfd;
	int n;
        int now = (int)time(NULL);
	if (debuglevel) debug("Trying server %d at time %d", index, now);
        if (now-servers[index].status < blacklist_time) {
		if (debuglevel) debug("Server %d is blacklisted", index);
		return -1;
	}
        if (servers[index].maxc != 0 &&
            (servers[index].c >= servers[index].maxc)) {
		if (debuglevel) debug("Server %d is overloaded", index);
		return -1;
	}
        upfd = socket(AF_INET, SOCK_STREAM, 0);
        if (upfd < 0) error("Error opening socket");
        memset(&serv_addr, 0, sizeof serv_addr);
        serv_addr.sin_family = AF_INET;
        serv_addr.sin_addr.s_addr = servers[index].addr.s_addr;
        serv_addr.sin_port = htons(servers[index].port);
        signal(SIGALRM, alarm_handler);
        alarm(timeout);
	n = connect(upfd, (struct sockaddr *)&serv_addr, sizeof serv_addr);
	alarm(0);	/* cancel scheduled timeout, if there is one */
        if (n == -1) {
		if (debuglevel) debug("Connect to server %d failed", index);
                servers[index].status = (int)time(NULL);
                return -1;
        }
	if (debuglevel) debug("Successful connect to server %d", index);
        return upfd;
}

static void add_client(int downfd, struct sockaddr_in *cli_addr)
{
        int upfd = -1, i, index, n;

	if (roundrobin) {
		if (debuglevel) debug("Bypassing client tracking");
	} else {
	        i = lookup_client(cli_addr->sin_addr);
		if (debuglevel) debug("lookup_client returns %d", i);
		if (i != -1) {
			index = clients[i].cno;
			upfd = try_server(index);
			if (upfd != -1) goto Success;
		}
		if (hash) {
			index = cli_addr->sin_addr.s_addr % nservers;
			upfd = try_server(index);
			if (upfd != -1) goto Success;
		}
	}
	if (!stubborn) {
        	index = current;
        	do {
        	        index = next_server(index);
        	        if ((upfd = try_server(index)) != -1) goto Success;
        	} while (index != current);
	}
        /* if we get here, we're dead */
        if (downfd != -1) close(downfd);
        if (upfd != -1) close(upfd);
        return;
Success:
        n = store_client(cli_addr->sin_addr, index);
        store_conn(downfd, upfd, n, index);
        return;
}

static int open_listener(char *a)
{
	int listenfd;
	struct sockaddr_in serv_addr;
	char b[1024], *p;
	int one = 1;

	memset(&serv_addr, 0, sizeof serv_addr);
	serv_addr.sin_family = AF_INET;
	p = strchr(a, ':');
	if (p) {
		strcpy(b, a);
		p = strchr(b, ':');
		*p = '\0';
		port = getport(p+1);
		setipaddress(&serv_addr.sin_addr, b);
		sprintf(b, inet_ntoa(serv_addr.sin_addr));
	} else {
		port = getport(a);
		serv_addr.sin_addr.s_addr = htonl(INADDR_ANY);
		sprintf(b, "0.0.0.0");
	}
	serv_addr.sin_port = htons(port);

	if ((listenfd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		error("can't open stream socket");
	}

	if (debuglevel) debug("local address=[%s:%d]", b, port);

	setsockopt(listenfd, SOL_SOCKET, SO_REUSEADDR, (char *)&one, sizeof one);
	if (bind(listenfd, (struct sockaddr *) &serv_addr,
		 sizeof serv_addr) < 0) {
		error("can't bind local address");
	}

	listen(listenfd, 5);
	return listenfd;
}

static int flush_down(int i)
{
	int n = write(conns[i].downfd, conns[i].downb, conns[i].downn);
	if (n > 0) {
		conns[i].downn -= n;
		if (conns[i].downn == 0) free(conns[i].downb);
		else memcpy(conns[i].downb, conns[i].downb+n, conns[i].downn);
	}
	return n;
}

static int flush_up(int i)
{
	int n = write(conns[i].upfd, conns[i].upb, conns[i].upn);
	if (n > 0) {
		conns[i].upn -= n;
		if (conns[i].upn == 0) free(conns[i].upb);
		else memcpy(conns[i].upb, conns[i].upb+n, conns[i].upn);
	}
	return n;
}

static void mainloop(int listenfd)
{
	int downfd, clilen;
	struct sockaddr_in cli_addr;
	fd_set w_read, w_write, w_error;
	int i, w_max;
	clilen = sizeof cli_addr;
	signal(SIGUSR1, stats);
	signal(SIGHUP, restart_log);
	signal(SIGTERM, quit);
	signal(SIGPIPE, SIG_IGN);

	loopflag = 1;

	while (loopflag) {
		int n;

		FD_ZERO(&w_read);
		FD_ZERO(&w_write);
		FD_ZERO(&w_error);
		FD_SET(listenfd, &w_read);	/* new connections */
		w_max = listenfd+1;

		/* add sockets from open connections */
		for (i = 0; i < connections_max; i++) {
			if (conns[i].downfd == -1) continue;
			if (conns[i].upn == 0) {
				FD_SET(conns[i].downfd, &w_read);
				if (conns[i].downfd+1 > w_max) {
					w_max = conns[i].downfd+1;
				}
			} else {
				FD_SET(conns[i].upfd, &w_write);
				if (conns[i].upfd+1 > w_max) {
					w_max = conns[i].upfd+1;
				}
			}
			if (conns[i].downn == 0) {
				FD_SET(conns[i].upfd, &w_read);
				if (conns[i].upfd+1 > w_max) {
					w_max = conns[i].upfd+1;
				}
			} else {
				FD_SET(conns[i].downfd, &w_write);
				if (conns[i].downfd+1 > w_max) {
					w_max = conns[i].downfd+1;
				}
			}
		}

		/* Wait for a connection from a client process. */
		n = select(w_max, &w_read, &w_write, /*&w_error*/0, 0);
		if (n < 0 && errno != EINTR) {
			perror("select");
			error("Error on select");
		}
		if (n <= 0) continue;

		if (FD_ISSET(listenfd, &w_read)) {
			downfd = accept(listenfd,
				(struct sockaddr *) &cli_addr, &clilen);
			if (downfd < 0) {
				if (debuglevel) perror("accept");
				continue;
			}

			add_client(downfd, &cli_addr);
		}

		/* check sockets from open connections */
		for (i = 0; i < connections_max; i++) {
			if (conns[i].downfd == -1) continue;
			if (FD_ISSET(conns[i].downfd, &w_read)) {
				if (copy_up(i) < 0) {
					close_conn(i);
					continue;
				}
			}
			if (FD_ISSET(conns[i].upfd, &w_read)) {
				if (copy_down(i) < 0) {
					close_conn(i);
					continue;
				}
			}
			if (FD_ISSET(conns[i].downfd, &w_write)) {
				if (flush_down(i) < 0) {
					close_conn(i);
					continue;
				}
			}
			if (FD_ISSET(conns[i].upfd, &w_write)) {
				if (flush_up(i) < 0) {
					close_conn(i);
					continue;
				}
			}
		}
	}
}

static int options(int argc, char **argv)
{
	int c;

	while ((c = getopt(argc, argv, "b:c:l:p:t:w:x:adfhnrs")) != EOF) {
		switch (c) {
		case 'a':
			asciidump = 1;
			break;
		case 'b':
			blacklist_time = atoi(optarg);
			break;
		case 'c':
			clients_max = atoi(optarg);
			break;
		case 'd':
			debuglevel++;
			break;
		case 'f':
			foreground = 1;
			break;
		case 'h':
			hash = 1;
			break;
		case 'l':
			logfile = optarg;
			break;
		case 'n':
			nblock = 0;
			break;
		case 'p':
			pidfile = optarg;
			break;
		case 'r':
			roundrobin = 1;
			break;
		case 's':
			stubborn = 1;
			break;
		case 't':
			timeout = atoi(optarg);
			if (timeout < 1) {
				usage();
			}
			break;
		case 'x':
			connections_max = atoi(optarg);
			break;
		case 'w':
			webfile = optarg;
			break;
		case '?':
		default:
			usage();
		}
	}

	return optind;
}

int main(int argc, char **argv)
{
	int i, listenfd;

	struct rlimit r;
	int n = options(argc, argv);
	argc -= n;
	argv += n;

	if (argc < 1) {
		usage();
	}

	getrlimit(RLIMIT_CORE, &r);
	r.rlim_cur = r.rlim_max;
	setrlimit(RLIMIT_CORE, &r);

	signal(SIGCHLD, SIG_IGN);

	if (!foreground) {
		background();
	}

	listenfd = open_listener(argv[0]);
	init(argc, argv);

	if (logfile) {
		logfp = fopen(logfile, "a");
		if (!logfp) error("Can't open logfile %s", logfile);
	}
	if (pidfile) {
		pidfp = fopen(pidfile, "w");
		if (!pidfp) {
			error("Can't create pidfile %s", pidfile);
			exit(1);
		}
		fprintf(pidfp, "%d", (int)getpid());
		fclose(pidfp);
	}

	mainloop(listenfd);
	if (debuglevel) debug("Exiting, cleaning up...");
	if (logfp) fclose(logfp);
	for (i = 0; i < CONNECTIONS_MAX; i++) {
		close_conn(i);
	}
	close(listenfd);
	if (pidfile) {
		unlink(pidfile);
	}
	return 0;
}
