# make_bterror.h.pl
#
#
# This script generates the bterror.h header file for the libbt Bit Torrent Library.
# It processes the tab-delimited file "errorlist.txt" to generate the bterror.h
# file used during compilation. The "errorlist.txt" file contains the error identifier,
# a tab, followed by an error message that corresponds to that identifier. Each error
# entry must be on a seperate line.
# Error codes that contain spaces will be converted to underscores.
# If no error message is supplied, a default error message is generated based on the error
# code identifier.
#
# EX: An input file (with <TAB> replaced by Chr(9) ascii) like this:
#
# load_file<TAB>Could not load file!
# peer_send_handshake<TAB>
# check hash<TAB>Hash check failed.
#
# Would produce a bterror.h file similar to this:
#
# (header info deleted for brevity)
# enum errorcodes {
#	BTERR_BASE = 500,
#	BTERR_UNKNOWN = BTERR_BASE,
#	BTERR_LOAD_FILE, "BTERROR: Could not load file!",
#	BTERR_PEER_SEND_HANDSHAKE, "BTERROR in function: peer_send_handshake",
#	BTERR_CHECK_HASH, "BTERROR: Hash check failed."
# };
#
# 


@errArray;

print "Reading errorlist.txt...\n";

open(INFILE, "errorlist.txt") || die "Can't open errorlist.txt";

while(!eof INFILE)
{
    $thisline = readline(INFILE);
    chomp $thisline;
    push @errArray, [ split(chr(9), $thisline) ];
}

print "Writing include/bterror.h...\n";

open(OUTFILE, ">include/bterror.h") || die "Can't create include/bterror.h";

print OUTFILE "#ifndef _BTERROR__H\n";
print OUTFILE "#define _BTERROR__H\n";
print OUTFILE "#include <time.h>\n";
print OUTFILE "\n";
print OUTFILE "/* This file was automatically generated by make_bterror.h.pl\n";
print OUTFILE "\n";
print OUTFILE "   To add new error codes and messages, please use the errorlist.txt\n";
print OUTFILE "   in the libbt root directory. Just add a new line at the end in the\n";
print OUTFILE "   the form of: ErrorCode<TAB>ErrorMessage (no quotes or other wierd\n";
print OUTFILE "   characters). Then type: perl make_bterror.h.pl to automatically\n";
print OUTFILE "   generate a new bterror.h file.\n";
print OUTFILE "\n";
print OUTFILE "   See bterr() for how BTERR is implemented.\n";
print OUTFILE "*/\n";
print OUTFILE "\n";
print OUTFILE "#define BTERR(sock, errorcode, flags) bterr(sock, errorcode, flags, __FILE__, __LINE__)\n";
print OUTFILE "#define BTERR_NOERR 0\n";
print OUTFILE "#define BTERR_DIE_FLAG 0x0001\n";
print OUTFILE "#define BTERR_LOGIT_FLAG 0x0010\n";
print OUTFILE "#if WIN32\n";
print OUTFILE "#define BTERR_WINSOCK_FLAG 0x1000\n";
print OUTFILE "#else\n";
print OUTFILE "/* Linux will ignore this flag */\n";
print OUTFILE "#define BTERR_WINSOCK_FLAG 0\n";
print OUTFILE "#endif\n";
print OUTFILE "\n";
print OUTFILE "\n";
print OUTFILE "enum errorcodes {\n";
print OUTFILE "    BTERR_BASE=500,\n";
print OUTFILE "    BTERR_UNKNOWN = BTERR_BASE,\n";

for $i ( 0 .. $#errArray )
{
    $errorcode = @errArray->[$i][0];
    $errorcode =~ s/ /_/;
    $errorcode = "    BTERR_" . uc($errorcode) . ",\n";
    print OUTFILE $errorcode;
}
    
print OUTFILE "    BTERR_LAST\n";
print OUTFILE "};\n";
print OUTFILE "\n";
print OUTFILE "#ifdef BTERROR_BODY\n";
print OUTFILE "char *bterror_string[] = {\n";
print OUTFILE "    \"BTERROR: Unknown Error\",\n";
    
for $i ( 0 .. $#errArray )
{
    $errormsg = @errArray->[$i][1];
    if ($errormsg)
    {
        $errormsg = "    \"BTERROR: " . $errormsg . "\",\n";
    } else
    {
        $errormsg = "    \"BTERROR in function: " . @errArray->[$i][0] . "\",\n";
    }
    print OUTFILE $errormsg;

}
    
print OUTFILE "    \"BTERROR: Last Error Placeholder\"\n";
print OUTFILE "};\n";
print OUTFILE "#else\n";
print OUTFILE "extern char *bterror_string[];\n";
print OUTFILE "#endif\n";
print OUTFILE "\n";
print OUTFILE "int now(void);\n";
print OUTFILE "void bterr(int sock, int errorcode, int flags, char *file, int line);\n";
print OUTFILE "\n";
print OUTFILE "#endif /* _BTERROR__H */\n";
    
close INFILE;
close OUTFILE;

print "Done!\n";
