/*
 * dstumbler v1.0 [main.c]
 * by h1kari - (c) Dachb0den Labs 2001
 */

/*
 * Copyright (c) 2001 Dachb0den Labs.
 *      David Hulton <h1kari@dachb0den.com>.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by David Hulton.
 * 4. Neither the name of the author nor the names of any co-contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY David Hulton AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL David Hulton OR THE VOICES IN HIS HEAD
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <time.h>

#ifdef __OpenBSD__
#include <curses.h>
#else
#include <ncurses.h>
#endif

#include "dstumbler.h"
#include "screen.h"

extern char *optarg;

/*
 * parse arguments and jump to ap scan loop
 */
int
main(int argc, char **argv)
{
  int c;

  /* process command line arguments */
  if(argc < 2)
    print_usage(argv[0]);

  init_vars();

  if(argv[1][0] == '-' && argv[1][1] != 'd')
    print_usage(argv[0]);
  else if(strcmp(argv[1], "-d") != 0)
  {
    optind++;
    device = argv[1];
  }

  while((c = getopt(argc, argv, "ospnm:dg:l:")) != -1)
  {
    switch(c)
    {
      case 'o':
        monmode = 1;
        break;
      case 's':
        scanmode = 0;
        break;
      case 'n':
        ncurse = 0;
        break;
      case 'm':
        macreset = 1;
        macint = MAX(atoi(optarg), 0);
        break;
      case 'd':
        nodevice = 1;
        break;
      case 'g':
        usegps = 1;
        gps = optarg;
        break;
      case 'l':
        uselog = 1;
        logfile = optarg;
        break;
      case '?':
      case 'h':
      default:
        print_usage(argv[0]);
    }
  }

  /* option overrides */
  if(nodevice)
  {
    device = NULL;
    scanmode = 0;
    monmode = 0;
    usegps = 0;
  }

  if(monmode)
  {
    scanmode = 0;
    macreset = 0;
    macint = 0;
  }
    

  /* initialize curses interface and start up devices */
  init_curses();

  redraw_menu();
  PRINTBANNER();
  smart_redraw();

  if(!nodevice)
    wi_start(device);

  if(usegps)
    gps_start(gps);

  if(monmode)
    mon_start(device);

  if(uselog && (log_fd = log_start(logfile)) == NULL)
    exitclean(2);

  /* enter the input/output loop */
  start_loop(device);

  exitclean(0); 
  return -1;
}

/*
 * print all of the usage info and exit
 */
void
print_usage(const char *progname)
{
  fprintf(stderr,
   "usage: %s <device> [-d] [-osn] [-m <int>] [-g <gps device>] [-l <logfile>]\n"
   " -d: run dstumbler without specifying a wireless device\n"
   " -o: specify the use of prism2 card in monitor mode\n"
   " -s: disable scan mode on the card, instead do old style stat polling\n"
   " -n: use basic ascii characters for limited terminal fonts\n"
   " -m: randomly set mac address at specified interval or 0 for startup\n"
   " -g: specify gps device to use\n"
   " -l: specify logfile to use for realtime logging\n", progname);
  exit(2);
}

/*
 * start the input/output handling loop
 */
#define SETMONCHAN() \
 if(monmode && chanlock && (apchange || apnew))\
 {\
   ch = aps[(aps_new && autosel) ? aps_new : aps_cur]->chan;\
   setdebugchan(iface, ch);\
 }
void
start_loop(const char *iface)
{
  int r, c, max_fd;
  struct timeval tout, now, then;
  time_t start, t;
  fd_set rset;

  /*
   * record the time of when we start up for running various functions
   * at certain intervals.
   */
  start = time(NULL);

  /* calculate the max fd first, so we dont repetedly do it in the loop */
  max_fd = STDIN_FILENO;

  if(usegps)
    max_fd = MAX(max_fd, gps_fd);

  while(1)
  {
    t = time(NULL) - start;

    if(macint && !(t % macint))
      wi_setrandmacaddr(iface);

    /*
     * half the poll speed if we're dealing with a non-prism2 and have to poll
     * for both weped and non-weped
     */
    gettimeofday(&now, NULL);
    ADDTVAL(now, then, (monmode ? SNIFFSPEED :
     (prism2 || scanmode ? POLLSPEED : POLLSPEED / 2)));

    /*
     * now loop with the now and then to guarantee we wait for the specified
     * timeout.
     */
    while(CMPTVAL(now, then) > -1)
    {
      FD_ZERO(&rset);
      FD_SET(STDIN_FILENO, &rset);

      if(usegps)
        FD_SET(gps_fd, &rset);

      tout.tv_sec = then.tv_sec - now.tv_sec;
      tout.tv_usec = then.tv_usec - now.tv_usec;

      /* if our usec is < 0, lets borrow from the seconds so it's positive */
      if(tout.tv_usec < 0)
      {
        tout.tv_sec--;
        tout.tv_usec = 1000000 + tout.tv_usec;
      }

      if((r = select(max_fd + 1, &rset, NULL, NULL, &tout)) == -1 && !sigwinch)
      {
        alert("error: error with select(): %s", strerror(errno));
        exitclean(2);
      }

      if(r == 0)
        break;
      else if(FD_ISSET(STDIN_FILENO, &rset) && (c = getch()))
        parse_cmd(c);
      else if(usegps && FD_ISSET(gps_fd, &rset))
        gps_parse(gps_fd);

      if(sigwinch)
        refreshclean();

      gettimeofday(&now, NULL);
    }

    if(nodevice)
    {
      smart_redraw();
      continue;
    }

    /*
     * if we're in monitor mode, grab the next packet off the wire so we can
     * analyze
     */
    if(monmode && mon_next(iface) == -1)
    {
      SETMONCHAN();
      smart_redraw();
      continue;
    }

    parse_ap(iface);
    parse_node(iface);

    SETMONCHAN();
    smart_redraw();

    apchange = apnew = 0;
  }
}

/*
 * initialize curses and draw windows
 */
void
init_curses(void)
{
  main_scr = initscr();

  signal(SIGHUP, exitclean);
  signal(SIGINT, exitclean);
  signal(SIGKILL, exitclean);
  signal(SIGPIPE, exitclean);
  signal(SIGALRM, exitclean);
  signal(SIGTERM, exitclean);
  signal(SIGWINCH, handle_sigwinch);

  keypad(stdscr, TRUE);
  nodelay(stdscr, TRUE);
  cbreak();
  nonl();
  noecho();
  curs_set(0);

  getscrmaxyx();

  /*
   * initialize colors if user is using a color terminal
   */
  if(has_colors())
  {
    use_default_colors();
    start_color();
    init_pair(COLOR_WHITE, COLOR_WHITE, COLOR_DEFAULT);
    init_pair(COLOR_RED, COLOR_RED, COLOR_DEFAULT);
    init_pair(COLOR_MAGENTA, COLOR_MAGENTA, COLOR_DEFAULT);
    init_pair(COLOR_GREEN, COLOR_GREEN, COLOR_DEFAULT);
    init_pair(COLOR_CYAN, COLOR_CYAN, COLOR_DEFAULT);
    init_pair(COLOR_BLUE, COLOR_BLUE, COLOR_DEFAULT);
    init_pair(COLOR_YELLOW, COLOR_YELLOW, COLOR_DEFAULT);
  }

  if(max_lines < MIN_SCREEN_Y || max_cols < MIN_SCREEN_X)
  {
    wprintw(main_scr, "error: sorry, your screen size must be at least %dx%d\n"
     "press any key to exit..\n", MIN_SCREEN_Y, MIN_SCREEN_X);
    nodelay(main_scr, FALSE);
    getch();
    exitclean(2);
  }

  draw_screen();
}

/*
 * draw/redraw all windows
 */
void
draw_screen(void)
{
  int i;

  node_scr_lines = (nodemode ? NODE_SCR_LINES_TOGGLE : 0);
  middle_border_lines = (nodemode ? MIDDLE_BORDER_LINES_TOGGLE : 0);

  menu_scr_cols = (!menumode ? MENU_SCR_COLS_TOGGLE : 0);
  right_border_cols = (!menumode ? RIGHT_BORDER_COLS_TOGGLE : 0);

  /*
   * initialize and draw borders
   */
  top_border = subwin(main_scr, TOP_BORDER_LINES, TOP_BORDER_COLS,
   TOP_BORDER_Y, TOP_BORDER_X);
  wattrset(top_border, BORDER_COLOR);
  for(i = 0; i < AP_SCR_COLS; i++)
    waddch(top_border, ncurse ? ACS_HLINE : BORDER_HLINE);
  waddch(top_border, ncurse ? (!menumode ? ACS_PLUS : ACS_BTEE) : BORDER_PLUS);
  for(i = 0; i < INFO_SCR_COLS; i++)
    waddch(top_border, ncurse ? ACS_HLINE : BORDER_HLINE);
  wrefresh(top_border);

  top_right_border = subwin(main_scr, TOP_RIGHT_BORDER_LINES,
   TOP_RIGHT_BORDER_COLS, TOP_RIGHT_BORDER_Y, TOP_RIGHT_BORDER_X);
  wattrset(top_right_border, BORDER_COLOR);
  waddstr(top_right_border, "aroicm");
  waddch(top_right_border, ncurse ? ACS_VLINE : BORDER_VLINE);
  wrefresh(top_right_border);

  bottom_border = subwin(main_scr, BOTTOM_BORDER_LINES, BOTTOM_BORDER_COLS,
   BOTTOM_BORDER_Y, BOTTOM_BORDER_X);
  wattrset(bottom_border, BORDER_COLOR);
  for(i = 0; i < GRAPH_SCR_COLS; i++)
    waddch(bottom_border, ncurse ? ACS_HLINE : BORDER_HLINE);

  if(!menumode)
  {
    waddch(bottom_border, ncurse ? ACS_BTEE : BORDER_BTEE);
    for(i = 0; i < MENU_SCR_COLS; i++)
      waddch(bottom_border, ncurse ? ACS_HLINE : BORDER_HLINE);
  }

  wrefresh(bottom_border);

  if(!menumode)
  {
    right_border = subwin(main_scr, RIGHT_BORDER_LINES, RIGHT_BORDER_COLS,
     RIGHT_BORDER_Y, RIGHT_BORDER_X);
    wattrset(right_border, BORDER_COLOR);
  }

  if(nodemode)
  {
    middle_border = subwin(main_scr, MIDDLE_BORDER_LINES, MIDDLE_BORDER_COLS,
     MIDDLE_BORDER_Y, MIDDLE_BORDER_X);
    wattrset(middle_border, BORDER_COLOR);
    for(i = 0; i < MIDDLE_BORDER_COLS; i++)
      waddch(middle_border, ncurse ? ACS_HLINE : BORDER_HLINE);
    wrefresh(middle_border);

    if(!menumode)
    {
      for(i = 0; i < NODE_SCR_LINES; i++)
        waddch(right_border, ncurse ? ACS_VLINE : BORDER_VLINE);
      waddch(right_border, ncurse ? ACS_RTEE : BORDER_PLUS);
      for(i = 0; i < GRAPH_SCR_LINES; i++)
        waddch(right_border, ncurse ? ACS_VLINE : BORDER_VLINE);
      wrefresh(right_border);
    }
  }
  else if(!menumode)
  {
    for(i = 0; i < RIGHT_BORDER_LINES; i++)
      waddch(right_border, ncurse ? ACS_VLINE : BORDER_VLINE);
    wrefresh(right_border);
  }

  /*
   * now initialize and set options for the other subwindows
   */
  ap_scr = subwin(main_scr, AP_SCR_LINES, AP_SCR_COLS, AP_SCR_Y, AP_SCR_X);
  info_scr = subwin(main_scr, INFO_SCR_LINES, INFO_SCR_COLS, INFO_SCR_Y,
   INFO_SCR_X);
  graph_scr = subwin(main_scr, GRAPH_SCR_LINES, GRAPH_SCR_COLS, GRAPH_SCR_Y,
   GRAPH_SCR_X);
  wattrset(graph_scr, GRAPH_COLOR);
  alert_scr = subwin(main_scr, ALERT_SCR_LINES, ALERT_SCR_COLS, ALERT_SCR_Y,
   ALERT_SCR_X);

  if(nodemode)
    node_scr = subwin(main_scr, NODE_SCR_LINES, NODE_SCR_COLS, NODE_SCR_Y,
     NODE_SCR_X);

  if(!menumode)
    menu_scr = subwin(main_scr, MENU_SCR_LINES, MENU_SCR_COLS, MENU_SCR_Y,
     MENU_SCR_X);

  scrollok(ap_scr, TRUE);
  scrollok(graph_scr, TRUE);
}

/*
 * make sure all the global variables are initialized properly
 */
#define APS_INIT_SIZE (sizeof(struct aps_s *) * PREALLOC_APS)
void
init_vars(void)
{
  /*
   * pre-allocate PREALLOC_APS amount of pointers for the aps array and zero
   * it out so there isn't any confusion.
   */
  if((aps = (struct aps_s **)malloc(APS_INIT_SIZE)) == NULL)
  {
    fprintf(stderr, "error: unable to allocate memory: %s\n", strerror(errno));
    exit(2);
  }

  memset((char *)aps, 0, APS_INIT_SIZE);

  /*
   * aps_max contains the current amount of bytes allocated for aps
   * aps_len is the current amount of allocated aps in the array
   * and aps_cur and aps_new contain the index of their corresponding
   * aps.
   */
  aps_max = PREALLOC_APS;
  aps_len = aps_cur = aps_new = aps_lst = aps_win = 0;

  /* flags for when aps are added */
  apchange = apnew = 0;

  /* similar vars for nodes */
  node_new = node_lst = node_win = 0;
  nodechange = nodenew = 0;

  /*
   * set redraw on all windows to 0, if they're set to 1 in other functions
   * when smart_redraw is called, the corresponding windows will be updated
   */
  redraw.ap_scr = redraw.info_scr = redraw.graph_scr = redraw.menu_scr =
   redraw.alert_scr = redraw.node_scr = redraw.all = 0;

  /* set these windows to null so we know if they're allocated or not */
  node_scr = middle_border = menu_scr = right_border = NULL;

  /* backup previous nic info */
  backup.wi_started = 0;
  backup.gps_started = 0;

  /* command line options :/ */
  prism2 = 0;
  monmode = 0;
  chanlock = 0;
  scanmode = 1;
  macreset = 0;
  macint = 0;
  ncurse = 1; /* default to full ascii character set */
  autosel = 0;
  resolvmfg = 0;
  nodevice = 0;
  usegps = 0;
  uselog = 0;
  nodemode = 0;
  audiomode = 0;
  menumode = 0;

  /* set chans defaults */
  ch = 1;

  /* set default gps stuff */
  gps_fd = -1;
  gps = NULL;

  /* set default audio stuff */
  sp_fd = -1;

  /* and default log stuff */
  log_fd = NULL;
  logfile = NULL;

  ns.dir = 'N';
  ns.coordinates = 0;
  ew.dir = 'E';
  ew.coordinates = 0;

  sigwinch = 0;
}

/*
 * just set the sigwinch flag so after a select we can handle the signal
 * by calling refreshclean().
 */
void
handle_sigwinch(int val)
{
  if(val != SIGWINCH || sigwinch)
    return;

  sigwinch++;
}

/*
 * handle screen refreshes and resizes when the code is worked out to do so.
 * make sure that this is only run once at a time, since there are often
 * multiple resizes when someone's resizing a window.
 */
void
refreshclean(void)
{
  getscrmaxyx();

  if(max_lines < MIN_SCREEN_Y || max_cols < MIN_SCREEN_X)
  {
    alert("error: sorry, your screen size must be at least %dx%d\n",
     MIN_SCREEN_Y, MIN_SCREEN_X);
    return;
  }

  /*
   * first kill the windows so we can redraw the full screen and adjust
   * in case the screen dimensions have changed.
   */
  erase();
  refresh();

  /* delete all of the windows */
  delwin(ap_scr);
  delwin(info_scr);
  delwin(graph_scr);
  delwin(alert_scr);
  delwin(top_border);
  delwin(top_right_border);
  delwin(bottom_border);

  if(node_scr != NULL)
  {
    delwin(node_scr);
    node_scr = NULL;
  }

  if(middle_border != NULL)
  {
    delwin(middle_border);
    middle_border = NULL;
  }

  if(menu_scr != NULL)
  {
    delwin(menu_scr);
    menu_scr = NULL;
  }

  if(right_border != NULL)
  {
    delwin(right_border);
    right_border = NULL;
  }

  /* resize main_scr and just redraw the screen */
  wresize(main_scr, max_lines, max_cols);
  draw_screen();

  /* rewrite all data to the subwindows */
  APCHANGE(aps_cur);
  REDRAWAP();
  REDRAWNODE();

  if(!menumode)
    redraw_menu();

  PRINTBANNER();

  if(autosel)
    redraw_toggle(autosel, 0, 'a');
  if(resolvmfg)
    redraw_toggle(resolvmfg, 1, 'r');
  if(nodemode)
    redraw_toggle(nodemode, 2, 'o');
  if(audiomode)
    redraw_toggle(audiomode, 3, 'i');
  if(chanlock)
    redraw_toggle(chanlock, 4, 'c');
  if(menumode)
    redraw_toggle(menumode, 5, 'm');

  smart_redraw();

  sigwinch = 0;
}

/*
 * make sure if we receive a signal or ctrl+c or something, we exit cleanly
 * otherwise ncurses will go apeshit.
 */
void
exitclean(int val)
{
  int i, j, k;

  endwin();

  if(val != EXIT_WIERR && !nodevice)
    wi_stop(device);

  if(usegps)
    gps_stop();

  if(monmode)
    mon_stop();

  if(uselog)
    log_stop(log_fd);

  /* free all of our malloc'ed memory */
  for(i = 0; i < aps_len; i++)
  {
    for(j = 0; j < aps[i]->log_len; j++)
      free(aps[i]->log[j]);

    for(j = 0; j < aps[i]->node_len; j++)
    {
      for(k = 0; k < aps[i]->node[j]->log_len; k++)
        free(aps[i]->node[j]->log[k]);

      free(aps[i]->node[j]);
    }

    free(aps[i]->log);
    free(aps[i]);
  }

  free(aps);

  putchar('\n');

  exit(val < 0 ? 2 : val);
}
