/*
 * dstumbler v1.0 [file.c]
 * by h1kari - (c) Dachb0den Labs 2001
 * based on generous contributions by Kevin Kadow <dstumbler@msg.net>
 */

/*
 * Copyright (c) 2001 Dachb0den Labs.
 *      David Hulton <h1kari@dachb0den.com>.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by David Hulton.
 * 4. Neither the name of the author nor the names of any co-contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY David Hulton AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL David Hulton OR THE VOICES IN HIS HEAD
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"

#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <unistd.h>

#include "dstumbler.h"

/*
 * write an ap log to the logfile
 */
void
log_ap(FILE *fd, struct aps_s *ap, struct log_s *log)
{
  struct tm tmval;

  gmtime_r((time_t *)&log->seen.tv_sec, &tmval);

  fprintf(fd, "%c %4.7f\t%c %4.7f\t( %s )\t%s\t"
   "( %02x:%02x:%02x:%02x:%02x:%02x )\t"
   "%d:%d:%d (GMT)\t[ %d %d %d ]\t# %d\t%d\t%ld.%ld\t%d\t%d\t%d\t( %s )\n",
   log->ns.dir, log->ns.coordinates, log->ew.dir, log->ew.coordinates,
   ap->ssid, ap->adhoc == 1 ? "ad-hoc" : "BBS",
   ap->bssid[0], ap->bssid[1], ap->bssid[2],
   ap->bssid[3], ap->bssid[4], ap->bssid[5],
   tmval.tm_hour, tmval.tm_min, tmval.tm_sec,
   log->quality[0], log->quality[1], log->quality[2],
   ap->weped, ap->chan, log->seen.tv_sec, log->seen.tv_usec,
   ap->keyed, ap->srate, ap->interval, ap->note);

  fflush(fd);
}

/*
 * parse an ap log from a logfile
 */
#define TEMP_BSS_LEN 8
void
unlog_ap(FILE *fd, char *buf, struct aps_s *ap)
{
  int i;
  u_int temp_bssid[MACSIZE];
  char temp_bss[TEMP_BSS_LEN];

  if(sscanf(buf, "%c %f %c %f (%*c%[^)]) %s "
   "( %2x:%2x:%2x:%2x:%2x:%2x ) %*d:%*d:%*d (GMT) [ %d %d %d ] "
   "# %d %d %ld.%ld %d %d %d ( %[^)])",
   &ap->ns.dir, &ap->ns.coordinates, &ap->ew.dir, &ap->ew.coordinates,
   ap->ssid, temp_bss,
   &temp_bssid[0], &temp_bssid[1], &temp_bssid[2],
   &temp_bssid[3], &temp_bssid[4], &temp_bssid[5],
   &ap->quality[0], &ap->quality[1], &ap->quality[2],
   &ap->weped, &ap->chan, &ap->lastseen.tv_sec, &ap->lastseen.tv_usec,
   &ap->keyed, &ap->srate, &ap->interval, ap->note) < 9)
  {
    alert("error: line in backup file is corrupt or not compatible");
    log_stop(fd);
    exitclean(2);
  }

  for(i = 0; i < MACSIZE; i++)
    ap->bssid[i] = temp_bssid[i];

  if(ap->ssid[strlen(ap->ssid) - 1] == ' ')
    ap->ssid[strlen(ap->ssid) - 1] = '\0';

  if(ap->note[strlen(ap->note) - 1] == ' ')
    ap->note[strlen(ap->note) - 1] = '\0';

  if(strcmp(temp_bss, "ad-hoc") == 0)
    ap->adhoc = 1;
  else
    ap->adhoc = 0;
}

/*
 * write a node log to the logfile
 */
void
log_node(FILE *fd, struct aps_s *ap, struct node_s *node, struct log_s *log)
{
  struct tm tmval;

  gmtime_r((time_t *)&log->seen.tv_sec, &tmval);

  fprintf(fd, "#NODE %c %4.7f\t%c %4.7f\t"
   "( %02x:%02x:%02x:%02x:%02x:%02x )\t%s\t"
   "( %02x:%02x:%02x:%02x:%02x:%02x )\t%d:%d:%d (GMT)\t"
   "[ %d %d %d ]\t# %d\t%d\t%ld.%ld\t%d\t%d\t%d\t%d\t%d\t( %s )\n",
   log->ns.dir, log->ns.coordinates, log->ew.dir, log->ew.coordinates,
   node->mac[0], node->mac[1], node->mac[2],
   node->mac[3], node->mac[4], node->mac[5],
   node->adhoc == 1 ? "ad-hoc" : "BSS",
   ap->bssid[0], ap->bssid[1], ap->bssid[2],
   ap->bssid[3], ap->bssid[4], ap->bssid[5],
   tmval.tm_hour, tmval.tm_min, tmval.tm_sec,
   log->quality[0], log->quality[1], log->quality[2],
   node->weped, ap->chan, log->seen.tv_sec,
   log->seen.tv_usec, node->keyed, node->srate, node->interval,
   node->any, node->datalen, node->note);

  fflush(fd);
}

/*
 * parse a node log from a logfile
 */
#define TEMP_BSS_LEN 8
void
unlog_node(FILE *fd, char *buf, struct node_s *node, u_char *bssid)
{
  int i, temp_bssid[MACSIZE], temp_mac[MACSIZE];
  char temp_bss[TEMP_BSS_LEN];

  if(sscanf(buf, "#NODE %c %f %c %f ( %2x:%2x:%2x:%2x:%2x:%2x ) %s "
   "( %2x:%2x:%2x:%2x:%2x:%2x ) %*d:%*d:%*d (GMT) [ %d %d %d ] "
   "# %d %*d %ld.%ld %d %d %d %d %d ( %[^)])",
   &node->ns.dir, &node->ns.coordinates,
   &node->ew.dir, &node->ew.coordinates,
   &temp_mac[0], &temp_mac[1], &temp_mac[2],
   &temp_mac[3], &temp_mac[4], &temp_mac[5], temp_bss,
   &temp_bssid[0], &temp_bssid[1], &temp_bssid[2],
   &temp_bssid[3], &temp_bssid[4], &temp_bssid[5],
   &node->quality[0], &node->quality[1], &node->quality[2],
   &node->weped, &node->lastseen.tv_sec, &node->lastseen.tv_usec,
   &node->keyed, &node->srate, &node->interval, &node->any,
   &node->datalen, node->note) < 9)
  {
    alert("error: line in backup file is corrupt or not compatible");
    fclose(fd);
    exitclean(2);
  }

  for(i = 0; i < MACSIZE; i++)
    node->mac[i] = temp_mac[i];

  for(i = 0; i < MACSIZE; i++)
    bssid[i] = temp_bssid[i];

  if(node->note[strlen(node->note) - 1] == ' ')
    node->note[strlen(node->note) - 1] = '\0';

  if(strcmp(temp_bss, "ad-hoc") == 0)
    node->adhoc = 1;
  else
    node->adhoc = 0;
}

/*
 * open the log file
 */
FILE *
log_start(char *logfile)
{
  time_t curr_time;
  struct tm tmval;
  FILE *fd;

  if(logfile == NULL)
    return NULL;

  curr_time = time(NULL);

  if((fd = fopen(logfile, "a")) == NULL)
  {
    alert("error: unable to open log file for writing: %s", strerror(errno));
    return NULL;
  }

  gmtime_r((time_t *)&curr_time, &tmval);

  /* print file header */
  fprintf(fd,
   "# $Creator: dstumbler v1.0\n"
   "# $Format: wi-scan with extensions\n"
   "# Latitude\tLongitude\t( SSID/Node-MAC )\tType\t( BSSID )\tTime (GMT)\t"
   "[ SNR Sig Noise ]\t# WEPed\tChan\tEpochTime\tAuth\tSupportedRate\t"
   "Interval\tNode-AnyNet\tNode-Datalen\tNote\n"
   "# $DateGMT: %d-%d-%d\n",
   tmval.tm_year + 1900, tmval.tm_mon + 1, tmval.tm_mday);

  return fd;
}

/*
 * close the log file
 */
void
log_stop(FILE *fd)
{
  if(fd != NULL)
    fclose(fd);

  fd = NULL;
}
