/***************************************************************************
                          rfilecxf.cpp  -  description
                             -------------------
    begin                : Sun Nov 7 1999
    copyright            : (C) 1999 by Andreas Mustun
    email                : mustun@ribbonsoft.com
 ***************************************************************************/

#include "rfilecxf.h"

#include <stdlib.h>

#include <qstring.h>
#include <qtextstream.h>
#include <qdatetime.h>
#include <qfileinfo.h>

#include "rappwin.h"
#include "rfilebase.h"
#include "rfonts.h"
#include "rgraphic.h"
#include "rlog.h"
#include "rmath.h"
#include "rprgdef.h"
#include "rstatuspanel.h"
#include "rstring.h"

// Constructor:
//
RFileCxf::RFileCxf(const QString& _name, RGraphic* _graphic)
  :RFileBase(_name)
{
  graphic=_graphic;
}



// Destructor:
//
RFileCxf::~RFileCxf()
{

}



// load a cxf file (read in buffer / read from buffer):
//
bool
RFileCxf::load()
{
  if(readFileInBuffer()) {
    separateBuf();
    return readFromBuffer();
  }
  else {
    return false;
  }
}



// read a cxf file from buffer:
//
bool
RFileCxf::readFromBuffer()
{
  // Create border elements:
  //
  int layer = graphic->addLayer( "Border" );

  graphic->setColor(7);

  //graphic->addLine( 0.0, -1.0, 2814.0, -1.0, layer, false );
  //graphic->addLine( 0.0, 10.0, 2814.0, 10.0, layer, false );

  QString numStr;
  int c;
  int normalFont = RFonts::getRFonts()->getFontNumber("normal");
  for( c=0; c<256; ++c ) {
    graphic->setColor(7);

    graphic->addLine( c*30.0, -10.0, c*30.0, 20.0, layer, false );

    graphic->setColor(13);
    graphic->setStyle(Qt::DashDotDotLine);

    graphic->addLine( c*30.0+10.0,-3.0, c*30.0+20.0,-3.0, layer, false );
    graphic->addLine( c*30.0+10.0, 0.0, c*30.0+20.0, 0.0, layer, false );
    graphic->addLine( c*30.0+10.0, 6.0, c*30.0+20.0, 6.0, layer, false );
    graphic->addLine( c*30.0+10.0, 9.0, c*30.0+20.0, 9.0, layer, false );

    graphic->addLine( c*30.0+10.0, -3.0, c*30.0+10.0, 9.0, layer, false );
    graphic->addLine( c*30.0+20.0, -3.0, c*30.0+20.0, 9.0, layer, false );

    graphic->setColor(0);
    graphic->setStyle(Qt::SolidLine);

    // ASCII number:
    numStr.sprintf("%03d", c);
    graphic->addText( c*30.0+15.0, -15.0,
                      normalFont,
                      numStr, E_STRAIGHT|E_CENTER,
                      4.0, 0.0, 0.0, 0.4, 0.0, 0.0, layer, false );

    // ASCII char:
    if(c>31 && RFonts::getRFonts()->getLetterElementNumber(normalFont, QChar(c))>0) {
      graphic->addText( c*30.0+15.0, -23.0,
                        normalFont,
                        QChar(c), E_STRAIGHT|E_CENTER,
                        6.0, 0.0, 0.0, 0.4, 0.0, 0.0, layer, false );
    }
  }

  // Add the characters:
  //
  layer = graphic->addLayer( "Letters" );
  graphic->setColor(0);
  int fontNumber = RFonts::getRFonts()->getFontNumber( QFileInfo(name()).baseName() );

  for( c=32; c<256; ++c ) {
    // Add the letter from this font:
    //
    if(RFonts::getRFonts()->getLetterElementNumber(fontNumber, QChar(c))>0) {
      int el = graphic->addText( c*30.0+10.0, 0.0,
                                 fontNumber,
                                 QChar(c),
                                 E_STRAIGHT|E_LEFT,
                                 9.0, 0.0, 0.0, 0.4, 0.4, 14.0, layer, false );

      // Tag all letters for splitting to elements later:
      if(el!=-1 && graphic->elementCurrent()) {
        graphic->elementAt(el)->setFlag( E_TAGGED );
        graphic->elementCurrent()->setFlag( E_TAGGED );
      }
    }
  }

  graphic->editTextElements( true );
  graphic->setPageSizeX(30.0);
  graphic->setPageSizeY(30.0);

  graphic->setPageOriginX(0.0);
  graphic->setPageOriginY(-10.0);

  return true;
}



// save a cxf file:
//
bool
RFileCxf::save()
{
  bool      ret;          // returned value
  RElement *el;           // pointer which walks through elements
  int       lc;           // Letter counter 0-255
  int       elNum;        // Number of elements of current letter
  double     letterOffset; // Letter offset

  //statusPanel()->iniProgress(graphic->count(), tr("Saving File..."));

  RLOG("\nRFileCxf::Save()");

  fPointer = fopen(name(), "wt");
  if(fPointer!=NULL) {

    // Write CXF-header:

    // HEADER:
    //
    fprintf(fPointer, "# Format:      CAM Expert Font\n"
                      "# Creator:     %s\n"
                      "# Version:     %s\n"
                      "# Name:        %s\n"
                      "# Dimensions:  0.1 0.6 1.4\n",
                      DEF_APPNAME, DEF_VERSION, QFileInfo(name()).baseName().latin1());

    // Initially untag all elements:
    //
    graphic->tagNothing(true);

    // LETTERS:
    //
    for(lc=32; lc<=255; lc++) {

      elNum = 0;
      letterOffset = 30.0*lc+10.0;

      // Tag current letter:
      //
      graphic->tagRange(letterOffset-10.0, -10.0, letterOffset-10.0+30.0, 20.0, true, true);
      graphic->tagLayer( graphic->getLayerNum("Border"), false, true );

      // Count tagged ELEMENTS:
      //
      for(el=graphic->elementFirst(); el!=0; el=graphic->elementNext()) {
        if(el->getFlag(E_ACTIVE) && !el->getFlag(E_UNDO) && el->getFlag(E_TAGGED) &&
           (el->getElementTyp()==T_LINE || el->getElementTyp()==T_ARC || el->getElementTyp()==T_CIRCLE)) {
          ++elNum;
          if(el->getElementTyp()==T_CIRCLE) ++elNum;
        }
      }

      if(elNum>0) {

        // Save letter header:
        //
        fprintf(fPointer, "\n[%c] %d\n", lc, elNum);

        // Save tagged ELEMENTS:
        //
        for(el=graphic->elementFirst(); el!=0; el=graphic->elementNext()) {
          if(el->getFlag(E_ACTIVE) && !el->getFlag(E_UNDO) && el->getFlag(E_TAGGED)) {

            statusPanel()->setProgress(graphic->elementAt(), 200);

            switch(el->getElementTyp()) {

              // Line:
              //
              case T_LINE:
                fprintf(fPointer,
                        "L %s,%s,%s,%s\n",
                        strFloatToString(el->getX1()-letterOffset).latin1(),
                        strFloatToString(el->getY1()).latin1(),
                        strFloatToString(el->getX2()-letterOffset).latin1(),
                        strFloatToString(el->getY2()).latin1() );
                break;

              // Arc:
              //
              case T_ARC:
                if(!el->getFlag(E_REVERSED)) {
                  fprintf(fPointer, "A");
                }
                else {
                  fprintf(fPointer, "AR");
                }

                fprintf(fPointer,
                        " %s,%s,%s,%s,%s\n",
                        strFloatToString(el->getCx()-letterOffset).latin1(),
                        strFloatToString(el->getCy()).latin1(),
                        strFloatToString(el->getCr()).latin1(),
                        strFloatToString(el->getA1()).latin1(),
                        strFloatToString(el->getA2()).latin1() );
                break;

              // Circle:
              //
              case T_CIRCLE:
                fprintf(fPointer,
                        "A %s,%s,%s,0,180\n",
                        strFloatToString(el->getCx()-letterOffset).latin1(),
                        strFloatToString(el->getCy()).latin1(),
                        strFloatToString(el->getCr()).latin1() );
                fprintf(fPointer,
                        "A %s,%s,%s,180,360\n",
                        strFloatToString(el->getCx()-letterOffset).latin1(),
                        strFloatToString(el->getCy()).latin1(),
                        strFloatToString(el->getCr()).latin1() );
                break;

              default:
                break;
            }
          }
        }
      }
      // Untag all elements:
      //
      graphic->tagNothing(true);
    }


    // End:
    //
    fprintf(fPointer, "\n#EOF\n");

    fclose(fPointer);

    statusPanel()->delProgress();

    // Reread fonts:
    //
    RFonts::getRFonts()->init();

    ret=true;
  }
  else {
    ret=false;
  }

  return ret;

}



// convert the graphic to a font:
//
void
RFileCxf::convertToFont()
{
  if(!graphic) return;

  RLOG( "\nconvert" );

  RElement *el;
  RElement *ell, *elr;             // Separators at left/ right of a letter
  double xl, xr;                   // Separator coords at left/ right of a letter
  int ch=32;                       // Current character

  do {
    // Search border elements of next letter:
    ell=elr=0;
    xl=xr=32000.0;
    for( el=graphic->elementFirst(); el!=0; el=graphic->elementNext() ) {
      if( !el->getFlag(E_UNDO) && graphic->getLayerName( el->getLayer() )=="separator" ) {
        if( el->getX1()<xl ) {
          xl = el->getX1();
          ell = el;
        }
      }
    }
    for( el=graphic->elementFirst(); el!=0; el=graphic->elementNext() ) {
      if( !el->getFlag(E_UNDO) && graphic->getLayerName( el->getLayer() )=="separator" ) {
        if( el->getX1()<xr && el!=ell ) {
          xr = el->getX1();
          elr = el;
        }
      }
    }

    if(!ell || !elr) break;

    xl = ell->getX1();
    xr = elr->getX1()-2.0;

    RLOG( "\nxl: " );
    RLOG( xl );
    RLOG( "\nxr: " );
    RLOG( xr );
    RLOG( "\nch: " );
    RLOG( ch );

    // Tag contours between or 1/2 between border elements:
    graphic->tagNothing(true);
    for( el=graphic->elementFirst(); el!=0; el=graphic->elementNext() ) {
      if( !el->getFlag(E_UNDO) && !el->getFlag(E_TAGGED) && el->getY1()<100.0 &&
           graphic->getLayerName( el->getLayer() )!="separator" &&
          (el->getX1()>=xl && el->getX1()<xr ||
           el->getX2()>=xl && el->getX2()<xr    ) ) {
        RElement* tEl = el;
        graphic->tagContour( el, true, true );
        el = tEl;
        graphic->setCurrentElement( el );
      }
    }

    // Move letter to the nw position:
    graphic->fillPreviewWithTagged( xl,0.0 );
    graphic->editMove( 0, 0.0,0.0, 10.0+(ch-32)*30.0,200.0 );

    // Delete left separator:
    ell->tag();
    graphic->editDelete(false);

    ch++;
  }while(true);

  graphic->repaint();
}
