//===============================================================
//
//		ExprParser		-- Written by Giovanni Dicanio
//
//		A C++ class to parse mathematical expressions.
//
//		(v. 0.3)
//
//
//		Copyright (c) by Giovanni Dicanio
//
//===============================================================



//---------------------------------------------------------------
// File:			exprparserfunctions.cpp
// Description:		Implementations of specific functions (e.g.:
//					sin, cos, etc.) parsing.
// Author:			Giovanni Dicanio
// First date:		20th January 2000
// Last update:		1st February 2000
//---------------------------------------------------------------



//---------------------------------------------------------------
//						INCLUDE SECTION
//---------------------------------------------------------------

#include "exprparser.h"		// class main header
#include <math.h>				// standard C math library



//---------------------------------------------------------------
//				MEMBER FUNCTIONS IMPLEMENTATION SECTION
//---------------------------------------------------------------


ExprParser::real ExprParser::doFunctionSin()
{
	real result;
	real param;

	skipBlanks();
	if (*m_currChar != '(')
	{
		setError(OPEN_PARENTHESIS_EXPECTED);
		return 0;	// dummy value
	}
	
	goAhead();
	param = doExpression();

	if (*m_currChar != ')')
	{
		setError(CLOSE_PARENTHESIS_EXPECTED);
		return 0;
	}
	goAhead();
	skipBlanks();

	result = sin(convertAngleInRadians(param));

	return result;
}


ExprParser::real ExprParser::doFunctionCos()
{
	real result;
	real param;

	skipBlanks();
	if (*m_currChar != '(')
	{
		setError(OPEN_PARENTHESIS_EXPECTED);
		return 0;	// dummy value
	}
	
	goAhead();
	param = doExpression();

	if (*m_currChar != ')')
	{
		setError(CLOSE_PARENTHESIS_EXPECTED);
		return 0;
	}
	goAhead();
	skipBlanks();

	result = cos(convertAngleInRadians(param));

	return result;
}


ExprParser::real ExprParser::doFunctionTan()
{
	real result;
	real param;

	skipBlanks();
	if (*m_currChar != '(')
	{
		setError(OPEN_PARENTHESIS_EXPECTED);
		return 0;	// dummy value
	}
	
	goAhead();
	param = doExpression();

	if (*m_currChar != ')')
	{
		setError(CLOSE_PARENTHESIS_EXPECTED);
		return 0;
	}
	goAhead();
	skipBlanks();

	// tan = sin / cos; beware of cos !!
	param = convertAngleInRadians(param);	// get angle in radians
	if (cos(param) == 0.0)
	{
		setError(DOMAIN_ERROR);
		return 0;
	}

	result = tan(param);

	return result;
}


ExprParser::real ExprParser::doFunctionCot()
{
	real result;
	real param;

	skipBlanks();
	if (*m_currChar != '(')
	{
		setError(OPEN_PARENTHESIS_EXPECTED);
		return 0;	// dummy value
	}
	
	goAhead();
	param = doExpression();

	if (*m_currChar != ')')
	{
		setError(CLOSE_PARENTHESIS_EXPECTED);
		return 0;
	}
	goAhead();
	skipBlanks();

	// cot = cos / sin; beware of sin !!
	param = convertAngleInRadians(param);	// get angle in radians
	if (sin(param) == 0.0)
	{
		setError(DOMAIN_ERROR);
		return 0;
	}

	result = 1.0 / tan(param);

	return result;
}


ExprParser::real ExprParser::doFunctionASin()
{
	real result;
	real param;

	skipBlanks();
	if (*m_currChar != '(')
	{
		setError(OPEN_PARENTHESIS_EXPECTED);
		return 0;	// dummy value
	}
	
	goAhead();
	param = doExpression();

	if (*m_currChar != ')')
	{
		setError(CLOSE_PARENTHESIS_EXPECTED);
		return 0;
	}
	goAhead();
	skipBlanks();

	if (fabs(param) > 1.0)	// asin(x), |x| <= 1
	{
		setError(DOMAIN_ERROR);
		return 0;
	}

	result = convertAngleInCurrMode(asin(param));

	return result;
}


ExprParser::real ExprParser::doFunctionACos()
{
	real result;
	real param;

	skipBlanks();
	if (*m_currChar != '(')
	{
		setError(OPEN_PARENTHESIS_EXPECTED);
		return 0;	// dummy value
	}
	
	goAhead();
	param = doExpression();

	if (*m_currChar != ')')
	{
		setError(CLOSE_PARENTHESIS_EXPECTED);
		return 0;
	}
	goAhead();
	skipBlanks();

	if (fabs(param) > 1.0)	// acos(x), |x| <= 1
	{
		setError(DOMAIN_ERROR);
		return 0;
	}

	result = convertAngleInCurrMode(acos(param));

	return result;
}


ExprParser::real ExprParser::doFunctionATan()
{
	real result;
	real param;

	skipBlanks();
	if (*m_currChar != '(')
	{
		setError(OPEN_PARENTHESIS_EXPECTED);
		return 0;	// dummy value
	}
	
	goAhead();
	param = doExpression();

	if (*m_currChar != ')')
	{
		setError(CLOSE_PARENTHESIS_EXPECTED);
		return 0;
	}
	goAhead();
	skipBlanks();

	result = convertAngleInCurrMode(atan(param));

	return result;
}


ExprParser::real ExprParser::doFunctionSqrt()
{
	real result;
	real param;

	skipBlanks();
	if (*m_currChar != '(')
	{
		setError(OPEN_PARENTHESIS_EXPECTED);
		return 0;	// dummy value
	}
	
	goAhead();
	param = doExpression();

	if (*m_currChar != ')')
	{
		setError(CLOSE_PARENTHESIS_EXPECTED);
		return 0;
	}
	goAhead();
	skipBlanks();


	if (param < 0.0)				// sqrt(x), x >= 0
	{
		setError(DOMAIN_ERROR);
		return 0;
	}

	result = sqrt(param);

	return result;
}


ExprParser::real ExprParser::doFunctionSqr()
{
	real result;
	real param;

	skipBlanks();
	if (*m_currChar != '(')
	{
		setError(OPEN_PARENTHESIS_EXPECTED);
		return 0;	// dummy value
	}
	
	goAhead();
	param = doExpression();

	if (*m_currChar != ')')
	{
		setError(CLOSE_PARENTHESIS_EXPECTED);
		return 0;
	}
	goAhead();
	skipBlanks();

	result = param*param;

	return result;
}


ExprParser::real ExprParser::doFunctionPow()
{
	real result;
	real param1, param2;

	skipBlanks();
	if (*m_currChar != '(')
	{
		setError(OPEN_PARENTHESIS_EXPECTED);
		return 0;	// dummy value
	}
	
	goAhead();
	param1 = doExpression();
	if (*m_currChar != ',')
	{
		setError(COMMA_EXPECTED);
		return 0;
	}

	goAhead();
	skipBlanks();
	param2 = doExpression();

	if (*m_currChar != ')')
	{
		setError(CLOSE_PARENTHESIS_EXPECTED);
		return 0;
	}
	goAhead();
	skipBlanks();

	result = pow(param1, param2);

	return result;
}


ExprParser::real ExprParser::doFunctionExp()
{
	real result;
	real param;

	skipBlanks();
	if (*m_currChar != '(')
	{
		setError(OPEN_PARENTHESIS_EXPECTED);
		return 0;	// dummy value
	}
	
	goAhead();
	param = doExpression();

	if (*m_currChar != ')')
	{
		setError(CLOSE_PARENTHESIS_EXPECTED);
		return 0;
	}
	goAhead();
	skipBlanks();

	result = exp(param);

	return result;
}


ExprParser::real ExprParser::doFunctionLn()
{
	real result;
	real param;

	skipBlanks();
	if (*m_currChar != '(')
	{
		setError(OPEN_PARENTHESIS_EXPECTED);
		return 0;	// dummy value
	}
	
	goAhead();
	param = doExpression();

	if (*m_currChar != ')')
	{
		setError(CLOSE_PARENTHESIS_EXPECTED);
		return 0;
	}
	goAhead();
	skipBlanks();

	if (param <= 0)		// log(x); x > 0 !
	{
		setError(DOMAIN_ERROR);
		return 0;
	}
	
	result = log(param);

	return result;
}


//---------------------------------------------------------------
// END OF FILE: exprparserfunctions.cpp
//---------------------------------------------------------------
