/*
** ifind (inode find)
** The @stake Sleuth Kit (TASK)
**
** Given an image  and block number, identify which inode it is used by
** 
** Brian Carrier [carrier@atstake.com]
** Copyright (c) 2002 Brian Carrier, @stake Inc.  All rights reserved
**
**
** TCTUILs
** Brian Carrier [carrier@cerias.purdue.edu]
** Copyright (c) 2001 Brian Carrier.  All rights reserved
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**
** 1. Redistributions of source code must retain the above copyright notice,
**    this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 3. The name of the author may not be used to endorse or promote
**    products derived from this software without specific prior written
**    permission.     
**
**
** THIS SOFTWARE IS NOT AFFILIATED WITH PURDUE UNIVERSITY OR THE CENTER FOR
** EDUCATION IN INFORMATION ASSURANCE AND SECURITY (CERIAS) AND THEY BEAR
** NO RESPONSIBILITY FOR ITS USE OR MISUSE.
**
**
** THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
** WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
** MERCHANTABILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
**
** IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
** INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
** (INCLUDING, BUT NOT LIMITED TO, LOSS OF USE, DATA, OR PROFITS OR
** BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
** OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
** ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
*/

#include "fs_tools.h"
#include "error.h"

static DADDR_T block;			/* the block to find */
static INUM_T curinode;			/* the inode being analyzed */

static u_int32_t curtype;		/* the type currently being analyzed: NTFS */
static u_int16_t curid;

FILE *logfp;

#define FIND_ALL	0x1
#define FOUND_ONE	0x2
static u_int8_t localflags;

static void usage(char *prog) {
	printf("usage: %s [-avV] [-f fstype] image unit_addr\n", prog);
	printf("\t-a: find all inodes\n");
	printf("\t-v: Verbose output to stderr\n");
	printf("\t-V: Print version\n");
	printf("\t-f fstype: Image file system type\n");
	printf("Supported file system types:\n");
    fs_print_types();

	exit(1);
}


/*
 * file walk action for non-ntfs
 */
static u_int8_t
find_inode_file_act(FS_INFO *fs, DADDR_T addr, char *buf, 
  int size, int flags, char *ptr)
{
	/* Drop references to block zero (sparse)
	 * This becomes an issue with fragments and looking for fragments
	 * within the first block.  They will be triggered by sparse 
	 * entries, even though the first block can not be allocated
	 */
	if (!addr)
		return WALK_CONT;

	if ((block >= addr) && (block < (addr + size / fs->block_size))) {
		printf("%i\n", (int)curinode);

		if (!(localflags & FIND_ALL)) {
			fs->close(fs);
			exit (0);
		}
		localflags |= FOUND_ONE;
	}
	return WALK_CONT;
}


/* 
 * file walk action callback for ntfs  
 *
 */
static u_int8_t
find_inode_ntfs_file(FS_INFO *fs, DADDR_T addr, char *buf, 
  int size, int flags, char *ptr)
{
	if (addr == block) {	
		printf("%i-%i-%i\n", (int)curinode, (int)curtype, (int)curid);

		if (!(localflags & FIND_ALL)) {
			fs->close(fs);
			exit (0);
		}
		localflags |= FOUND_ONE;
	}
	return WALK_CONT;
}



/*
** find_inode
**
** Main function that calls the direct and indirect calls
*/
static u_int8_t
find_inode(FS_INFO *fs, INUM_T inum, FS_INODE *fsi, int flags, char *ptr) 
{
	int myflags = (FS_FLAG_AONLY | FS_FLAG_CONT | FS_FLAG_SLACK |
	  FS_FLAG_ALLOC | FS_FLAG_UNALLOC );

	curinode = inum;

	/* NT Specific Stuff: search all ADS */
	if ((fs->ftype & FSMASK) == NTFS_TYPE) {
		FS_DATA *data = fsi->attr;

		while ((data) && (data->flags & FS_DATA_INUSE)) {
			curtype = data->type;
			curid = data->id;
			if (data->flags & FS_DATA_NONRES) {
				fs->file_walk(fs, fsi, data->type, data->id, myflags, 
				  find_inode_ntfs_file, ptr);
			}
			data = data->next;
		}
		return WALK_CONT;
	}
	else {
		fs->file_walk(fs, fsi, 0, 0, myflags, 
		  find_inode_file_act, ptr);
	}

	return WALK_CONT;
}


/*
 * if the block is a meta data block, then report that, otherwise
 * this is where we say that the inode was not found
 */
static u_int8_t
block_act (FS_INFO *fs, DADDR_T addr, char *buf, int flags, char *ptr)
{
	if (flags & FS_FLAG_META)
		printf("Meta Data\n");
	else
		printf("Inode not found\n");

	return WALK_STOP;
}

int
main(int argc, char **argv) 
{
	char   *fstype = DEF_FSTYPE;
	u_int32_t flags = 0xffffffff;
	FS_INFO	*fs;
	char ch;
	extern int optind;
	progname = argv[0];

	localflags = 0;

    while ((ch = getopt(argc, argv, "af:V")) > 0) {  
		switch (ch) {
		 case 'a':
			localflags |= FIND_ALL;
			break;
        case 'f':
            fstype = optarg;
            break;
		case 'v':
			verbose++;
			logfp = stderr;
			break;
		case 'V':
			print_version();
			exit(0);
		case '?':  
			default:
			usage(argv[0]);
		}
	}
	
	if (optind+2 != argc) 
		usage(argv[0]);

	fs = fs_open(argv[optind++], fstype);
	block = atoi(argv[optind]);

	if (block > fs->last_block) {
		printf("Block %lu is larger than last block in image (%lu)\n",
		  (ULONG)block, (ULONG)fs->last_block);
		fs->close(fs);
		return 0;
	}
	else if (block == 0) {
		printf("Inode not found\n");
		fs->close(fs);
		return 0;
	}

	fs->inode_walk(fs, fs->first_inum, fs->last_inum,
		flags, find_inode, (char *) 0);

	/* 
	 * If we did not find an inode yet, we call block_walk for the 
	 * block to find out the associated flags so we can identify it as
	 * a meta data block */
	if (0 == (localflags & FOUND_ONE)) {
		fs->block_walk(fs, block, block,
			FS_FLAG_UNALLOC | FS_FLAG_ALLOC | FS_FLAG_META | FS_FLAG_AONLY, 
			block_act, (char *) 0);
	}

	fs->close(fs);

	return 0;
}

