/*	$Id: pickup.c,v 1.7 2007/01/13 10:50:08 mbalmer Exp $	*/

/*
 * Copyright (c) 2006, 2007 Marc Balmer <mbalmer@openbsd.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/time.h>
#include <sys/types.h>
#include <sys/queue.h>
#include <sys/resource.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include <ctype.h>
#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <paths.h>
#include <pwd.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <time.h>
#include <unistd.h>

#include "pathnames.h"
#include "tabled.h"

#include "imsg.h"

#include <poll.h>
#include <sys/syslimits.h>

#define MAX_FILE	1024
#define MAXLEN		256
#define MAXSTR		256

#define LOG_UNSET	-1

extern char	*sock;

extern int	 verbose;
extern char	*cfgfile;
extern int	 log_facility;

struct imsgbuf	*ibuf_main;

extern char *__progname;

static void pf_table_add(const char *, const char *, long);
static void pf_table_clr(const char *, const char *, long);
static void imsg_commit();

void
p_sighdlr(int signum)
{
	switch (signum) {
	case SIGHUP:
		break;
	}
}

static void
pf_table_add(const char *table, const char *addr, long duration)
{
	pid_t pid;
	struct pftable_msg m;

	if (table == NULL || addr == NULL)
		return;
	
	pid = getpid();

	strlcpy(m.pftable, table, PFTABLE_LEN);
	strlcpy(m.addr, addr, ADR_LEN);
	m.duration = duration;
	m.len = 0;
	
	if (imsg_compose(ibuf_main, IMSG_PFTABLE_ADD, 0, pid, -1, &m,
	    sizeof(m)) != -1) 
		imsg_commit();
}

static void
pf_table_clr(const char *table, const char *addr, long duration)
{
	pid_t pid;
	struct pftable_msg m;

	if (table == NULL || addr == NULL)
		return;
	
	pid = getpid();

	strlcpy(m.pftable, table, PFTABLE_LEN);
	strlcpy(m.addr, addr, ADR_LEN);
	m.duration = duration;
	m.len = 0;
	
	if (imsg_compose(ibuf_main, IMSG_PFTABLE_DEL, 0, pid, -1, &m,
	    sizeof(m)) != -1)
		imsg_commit();
}

static void
imsg_commit()
{
	struct pollfd	pfd[1];
	int nfds;

	for (;;) {
		bzero(pfd, sizeof(pfd));
		pfd[0].fd = ibuf_main->fd;
		pfd[0].events = POLLOUT;

		if ((nfds = poll(pfd, 1, INFTIM)) == -1)
			if (errno != EINTR)
				errx(1, "engine: poll error");

		if (nfds > 0 && ibuf_main->w.queued) {
			if (msgbuf_write(&ibuf_main->w) < 0)
				errx(1, "pipe write error");
			else
				break; /* XXX All bytes sent? */
		}
	}
}

int
p_main(int pipe_m2e[2])
{
	struct stat	 stb;
	struct passwd	*pw;
	FILE		*fp;
	char		 line[1024];
	char		 *p, *cmd, *table, *adr;
	int		 nullfd;
	int		 pid;

	switch (pid = fork()) {
	case -1:
		errx(1, "can't fork");
	case 0:
		break;
	default:
		return pid;
	}

	if ((pw = getpwnam(TABLED_USER)) == NULL)
		errx(1, "getpwnam");

	if ((nullfd = open(_PATH_DEVNULL, O_RDWR, 0)) == -1)
		exit(1);

	if (stat(pw->pw_dir, &stb) == -1)
		errx(1, "stat");
	if (stb.st_uid != 0 || (stb.st_mode & (S_IWGRP|S_IWOTH)) != 0)
		errx(1, "bad privsep dir permissions");

	if ((fp = fopen(sock, "r")) == NULL)
		errx(1, "open of fifo %s failed", sock);

	if (chroot(pw->pw_dir) == -1)
		errx(1, "chroot");
	if (chdir("/") == -1)
		errx(1, "chdir(\"/\")");

	if (!verbose) {
		dup2(nullfd, STDIN_FILENO);
		dup2(nullfd, STDOUT_FILENO);
		dup2(nullfd, STDERR_FILENO);
	}
	close(nullfd);

	setproctitle("pickup");

	/* imsg stuff */
	close(pipe_m2e[0]);
	if ((ibuf_main = malloc(sizeof(struct imsgbuf))) == NULL)
		errx(1, "memory error");
	imsg_init(ibuf_main, pipe_m2e[1]);

	/* Do the work */
	while (fgets(line, sizeof(line), fp) != NULL) {
		p = line;
		if (((cmd = strsep(&p, " \t")) != NULL) &&
		    ((table = strsep(&p, " \t")) != NULL) &&
		    ((adr = strsep(&p, " \t\n")) != NULL)) {
			if (!strcmp(cmd, "add"))
				pf_table_add(table, adr, 0);
			else if (!strcmp(cmd, "clr"))
				pf_table_clr(table, adr, 0);
		}
	}
	fclose(fp);

	closelog();
	_exit(2);
}
