#!/usr/bin/perl
#
# passiveOS.pl version 0.2
#
# Reads in snort logs and attempts to determine what OS the
# originating packet came from.
#
# Usage: ./passiveOS.pl [-l <log_dir>] [-a <alert_file>]
#
# You can actually parse both the log_dir and the alert file at the 
# same time if you want.
#
# PKTCNT is the count of packets for that IP.  The higher the PKTCNT the
# more accurate the windowsize average should be.  You can turn on PKTCNT and
# other flags for detected OSes by setting $detail=1 below.
# This program expects the fingerprints.dat file in its current directory
#
# Changes: Now uses Net::SnortLog and can parse the alert file
#
# Craig Smith (March 2000) - Licenced under the GPL
use Net::SnortLog;
require 'getopts.pl';

Getopts('l:a:');
die "Usage: $0 [-l <log_dir>] [-a <alert_file>]" if !$opt_l && !$opt_a;

## Our massive amount of variables :)
#
my $detail=0;			# Verbose Detail
my $version=0.2;		# Our version number
my $line;			# Line holder for log files
my $srcip;			# Targets source ip 
my $oldip;			# Retain last used IP
my $ttl;			# TTL
my $oldttl;			# Retain last TTL
my $tos;			# Type of Service
my $oldtos;			# Retain last TOS
my $win;			# Window size
my $df;				# Don't Fragment Flag
my $olddf;			# Retain last DF
my $avgwin;			# Get AVG. window size
my $wincnt;			# How many windows do we have to avg.
my @files2check;		# Listing of valid files to check
my @ips;			# IPs flags and data to calculate
my $diff=30;			# Amount of allowable hops in TTL
my $pktcnt;			# Total packets counted for each IP
my $guess;			# OS Guess
my @printdata;             # Holds hold fingerprint info into an array
my %fingerprint;           # holds are fingerprint info from file 

my $snort_data = new Net::SnortLog;	# Our special Snort OO
my $packet = {};		# Anon Hash reference used by Snort.pm

print "$0 version $version - Craig Smith (March 2000)\n";
## Read in the fingerprint database
#
open FPDATA, "./fingerprints.dat" || die "Couldn't open fingerprints : $!";
while(<FPDATA>) {
  $line = $_;
  chomp($line);
  if($line=~/^#/) { next; }
  if($line) {
        if($line=~/(\S*)(\s*)(\S*)(\s*)(\S*)(\s*)(\S*)(\s*)(\S*)(\s*)(\S*)(\s*)(\S*)/) {
          $fingerprint = {};
          $fingerprint->{OS} = $1;
          $fingerprint->{VERSION} = $3;
          $fingerprint->{PLATFORM} = $5;
          $fingerprint->{TTL} = $7;
          $fingerprint->{WINDOW} = $9;
          $fingerprint->{DF} = $11;
          $fingerprint->{TOS} = $13;
          # WINDOW is special, we need to check for ranges
          if($fingerprint->{WINDOW}=~/(\d*)-(\d*)/) {
            $fingerprint->{WINDOW_MIN}=$1;
            $fingerprint->{WINDOW_MAX}=$2;
          } else {      # Exact number
            $fingerprint->{WINDOW_MIN}=$fingerprint->{WINDOW};
            $fingerprint->{WINDOW_MAX}=$fingerprint->{WINDOW};
          }
         push @printdata, $fingerprint;
        } else {        # Invalid line
          die "Invalid line in fingerprint file";
        }
  }
}
close FPDATA;

if($opt_l) {
  print "STAGE 1: Reading in snort log and parsing files...\n";
  ## Read in each TCP file from the snort log directory
  #
  @files2check=$snort_data->get_log_files($opt_l, "TCP");
}   #  You can specify both if you want
if($opt_a) {
  print "STAGE 1: Loading and parsing Alert log...\n";
  push @files2check, $opt_a;
}

## Parse all of our different files
#
foreach $file (@files2check) {
  $snort_data->parse_file($file);
}

## Analyze our data and put it into a nice little array
#
while($packet=$snort_data->next_pkt) {
  $df=$packet->{DF} ? "y" : "n";
  push @ips, join(':', $packet->{SRCIP}, $packet->{TTL}, $df, hex $packet->{WIN}) if $packet->{WIN};
}

## Sort array and do all calculations now
#
print "STAGE 2: Determine source IP base OS\n";
@ips = sort @ips;
foreach my $ip (@ips) {
  ($srcip, $ttl, $df, $win)=split(':', $ip);
   if(($srcip ne $oldip) && $oldip) {	# We've switched IPs
	$avgwin=int $avgwin/$wincnt if $wincnt;	# Calculate avg for printing
	if($avgwin) {
	  $guess="Unknown OS";		# Default guess
	  foreach (@printdata) {
	    if(
        	($oldttl <= $_->{TTL} && $oldttl >= ($_->{TTL}-$diff)) &&
	        ($avgwin >= $_->{WINDOW_MIN} && 
		 $avgwin <= $_->{WINDOW_MAX}) &&
	        ($olddf eq $_->{DF})) {
        	  $guess = "$_->{OS} $_->{VERSION} [$_->{PLATFORM}]";
	        }
	  }
	} # Endif $avgwin
    	  if($guess !~ /Unknown/) {
		print "$oldip -> $guess";
		print "- TTL($oldttl) AVGWIN($avgwin) DF($olddf) PKTCNT($pktcnt)" if $detail;
		print "\n";
    	  } else {
    		print "$oldip -> $guess (TTL($oldttl) TOS($oldtos) AVGWIN($avgwin) DF($olddf) PKTCNT($pktcnt))\n";
    	  }
	$avgwin=0;$wincnt=0;$pktcnt=0;		# Reset averages
   }
   $pktcnt++;
   $oldip=$srcip;			# Save new IP as old
   $oldttl=$ttl;
   $oldtos=$tos;
   $olddf=$df;
    # Calculate average windowsize
   $avgwin += $win if $win;	# Only Calculate if we just found it
   $wincnt++ if $win;		# Add 1 to our win count
}


