/* Copyright (c) 2000  Kevin Sullivan <nite@gis.net>
 *
 * Please refer to the COPYRIGHT file for more information.
 */

#include <stdio.h>
#ifndef MCURSES
#include <ncurses.h>
#endif
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/socket.h>
#include <errno.h>

#include "defines.h"
#include "codes.h"
#include "colors.h"
#include "scheck.h"
#include "timer.h"
#include "nap.h"

#ifdef MCURSES
#include "wind.h"
#endif

extern info_t info;
extern int srch;
extern WINDOW *wchan;

static sock_t *head = NULL;

/* add a connection to end of global connection list. s is file
   descriptor, nm is name, t is type (S_R or S_W), func is handler
   function. d is left blank. nm is copied (strdup). 
   Also, if this is an upload or download connection, notify server (ugly!) */
/* Note that upload/dowload notification is done in nap.c:dosend() and
   nap.c:doget() for connections initiated by a remote client. */
void addsock(int s, char *nm, unsigned char t, int (*func)(WINDOW *, sock_t *))
{
  sock_t *cur, *cur1, *sv;
  
  if (head)
  {
    for (cur=head,cur1=NULL;cur!=NULL;cur=cur->next)
      cur1 = cur;
    cur = (sock_t *)malloc(sizeof(sock_t));
    cur1->next = cur;
  }
  else
  {
    head = (sock_t *)malloc(sizeof(sock_t));
    cur = head;
  }
  cur->s = s;
  cur->t = t;
  cur->nm = strdup(nm);
  cur->func = func;
  cur->next = NULL;
  cur->d = NULL;
  
  sv = findsock("server");
  if (sv)
  {
    if (cur->nm[0] == 'u' && cur->nm[1] == ' ')
      sendpack(sv->s, F_UP, NULL);
    if (cur->nm[0] == 'd' && cur->nm[1] == ' ')
      sendpack(sv->s, F_DOWN, NULL);
  }
}

/* delete a connection. It is uniquely identified by its file
   descriptor s.  This routine also notifies the server if an upload
   or download is being shut down (this is ugly, since this should
   rather be done in connection with the task structure, not the
   connection structure). It also sets srch to 0 in case we are
   deleting the server connection. It also shuts down the socket of
   the connection, and closes it, if it is indeed a socket connection
   (file descriptor >2). */
void delsock(int s)
{
  int i;
  sock_t *cur, *cur1=NULL, *sv;
  
  sv = findsock("server");
  
  for (i=0,cur=head;cur!=NULL;cur=cur->next)
  {
    if (cur->s == s)
    {
      if (cur->nm[0] == 'd' && cur->nm[1] == ' ')
      {
        if (sv)
          sendpack(sv->s, F_DOWNDONE, NULL);
      }
      if (cur->nm[0] == 'u' && cur->nm[1] == ' ')
      {
        if (sv)
          sendpack(sv->s, F_UPDONE, NULL);
      }
      if (!strcmp(cur->nm, "server"))
        srch = 0;
      if (cur1)
        cur1->next = cur->next;
      else
        head = cur->next;
      free(cur->nm);
      free(cur);
      /* In many cases, the socket is closed by a connection handler before
       * the call to delsock(). (See the many places in nap.c where a
       * connection handler calls close(m->s) before return(0)).
       * It would be safer to only call close() once, but this second call 
       * to close() hasn't caused problems so far because delsock() is always
       * called immediately after a connection handler returns 0. i.e. there
       * is no opportunity for the socket descriptor to be reused in the mean
       * time (or is there?) */
      if (s > 2)
      {
        shutdown(s, 2);
        close(s);
      }
      return;
    }
    cur1 = cur;
  }
}

/* finds a connection by name in the connection list */
sock_t *findsock(char *nm)
{
  sock_t *cur;
  
  for (cur=head;cur!=NULL;cur=cur->next)
    if (!strcasecmp(nm, cur->nm))
      return(cur);
  
  return(NULL);
}

/* prints the list of connections to the specified window - actually,
   it only prints their names. */
void psocks(WINDOW *win)
{
  sock_t *cur;
  
  for (cur=head;cur!=NULL;cur=cur->next)
    wp(win, "%s\n", cur->nm);
  drw(win);
}

/* this is the main event loop of the nap program. It is here that we
   wait for network activity and then tend to the various different
   connections (including user input). In addition, we call tevent()
   (in timer.c) once a second or so, to take care of scheduled events.

   This function is called precisely once, from main() in nap.c. Once
   we leave here, nap shuts down and terminates.

   up is the input window (equal to the global variable winput), and
   win is the output window of the terminal interface (equal to the
   global variable wchan). Why not just refer to the global variables,
   as do some of the procedures that are called from here? */

void sockfunc(WINDOW *win, WINDOW *up)
{
  fd_set fs, fw; /* sets of file descriptors to watch, see "man select" */
  int r;
  sock_t *cur;
  struct timeval sec; /* note Kevin used <time.h> timespec here instead of
			 <select.h> timeval - this was a type error */
  
  while (1)
  {
    dochecks();  /* notify terminal that it has been resized, if necessary.
		    Why not do this in drw()? */
    drw(up);     /* redraw input window - why? */

    /* if the --autorestart option is set, and there is no server, try to 
       reconnect to a server. */

    if (info.autorestart) { 
      for (cur=head;cur!=NULL;cur=cur->next) {
	if (!strcasecmp(cur->nm, "server"))
	  break;
      }
      if (cur==NULL) {
	sleep(1);  /* sleep to avoid excessive load in case of infinite loop */
	wp(win, "Connection to server lost, reconnecting...\n");
	drw(win);
	dreconnect(-1, "reconnect", NULL, 0, win);
      }
    }
    
    /* if all sockets are gone, return. Note that "input" is the only
       entry that is not a socket. */

    if (!head)
      return;
    else if (!strcasecmp(head->nm, "input") && !head->next)
    {
      delsock(head->s);
      return;
    }
    
    /* let fs be the set of "read" file descriptors, and fw the list
       of "write" file descriptors, from the connection list */

    FD_ZERO(&fs);
    FD_ZERO(&fw);
    for (cur=head;cur!=NULL;cur=cur->next)
    {
      if (cur->t == S_R)
        FD_SET(cur->s, &fs);
      else if (cur->t == S_W)
        FD_SET(cur->s, &fw);
    }
    
    /* prepare timeout of 1 second, then wait for activity on connections.
       On fs connections, wait for characters to be available for reading.
       On fw connections, wait for ability to write. */
    sec.tv_sec = 1;
    sec.tv_usec = 0;
    r = select(FD_SETSIZE, &fs, &fw, NULL, &sec);

    /* the reason the previous call to "select" needs a timeout is so that
       we can call tevent(0) once a second or so to tend to scheduled events */
    tevent(0);

    /* if there was no network activity, continue main loop */
    if (r <= 0)
      continue;
    
    /* else do something for every connection that had activity. 
       Essentially, just call the handler (cur->func) for that connection.
       */
    for (cur=head;cur!=NULL;cur=cur->next)
    {
      if (cur->t == S_R && FD_ISSET(cur->s, &fs))
      {
	FD_CLR(cur->s, &fs);
        r = cur->func(win, cur);
        if (!r)
          delsock(cur->s);  /* would like to change this so cur->func deletes
			       its own connection when done with it */
        else if (r == -1)
	  goto panic;
	cur=head;  /* need to start from beginning since sockets might
		      have been deleted in the meantime */
      }
      else if (cur->t == S_W && FD_ISSET(cur->s, &fw))
      {
	FD_CLR(cur->s, &fs);
        r = cur->func(win, cur);
        if (!r)
          delsock(cur->s);
        else if (r == -1)
	  goto panic;
	cur=head;
      }
    }
  }

  /* end of loop not reachable */

 panic:                 /* we go here if something really goes wrong */
  while (head!=NULL)
    delsock(head->s);
  return;
}

/* this essentially seems to be a repetition of the above, except it
   returns after sec seconds. It is called only in one place in
   cmds.c, and can maybe be done better with timers? */

void sockfunct(WINDOW *win, WINDOW *up, int sec)
{
  fd_set fs, fw;
  int r, adj=0;
  sock_t *cur;
  struct timeval tv;
  time_t t, t1;
  
  while (1)
  {
    tv.tv_sec = sec-adj;
    tv.tv_usec = 0;
    if (tv.tv_sec <= 0)
      return;
      
    dochecks();
    drw(up);
    FD_ZERO(&fs);
    FD_ZERO(&fw);
    
    if (!head)
      return;
    else if (!strcasecmp(head->nm, "input") && !head->next)
    {
      delsock(head->s);
      return;
    }
    
    for (cur=head;cur!=NULL;cur=cur->next)
    {
      if (cur->t == S_R)
        FD_SET(cur->s, &fs);
      else if (cur->t == S_W)
        FD_SET(cur->s, &fw);
    }
    
    time(&t);
    
    r = select(FD_SETSIZE, &fs, &fw, NULL, &tv);
    tevent(0);
    if (r == -1)
      continue;
    else if (!r)
      return;
    
    time(&t1);
    
    adj+=(t1-t);
    
    for (cur=head;cur!=NULL;cur=cur->next)
    {
      if (cur->t == S_R && FD_ISSET(cur->s, &fs))
      {
	FD_CLR(cur->s, &fs);
        r = cur->func(win, cur);
        if (!r)
          delsock(cur->s);

        else if (r == -1)
	  goto panic;
	cur=head;
      }
      else if (cur->t == S_W && FD_ISSET(cur->s, &fw))
      {
	FD_CLR(cur->s, &fw);
        r = cur->func(win, cur);
        if (!r)
	  delsock(cur->s);
        else if (r == -1)
	  goto panic;
	cur=head;
      }
    }
  }

  /* end of loop not reachable */
  
 panic:                 /* we go here if something really goes wrong */
  while (head!=NULL)
    delsock(head->s);
  return;

}
