/* $Id: search.c,v 1.2 2007/10/25 11:43:32 merdely Exp $ */
#include <stdio.h>
#include <stdlib.h>
#define LDAP_DEPRECATED 1
#include <ldap.h>
#include <err.h>
#include <login_cap.h>

#include "login_ldap.h"

char * handle_referral(struct auth_ctx *, char **, char *, int);
int q = 0;

char *
search(struct auth_ctx *ctx, char *base, char *flt, int scp)
{
	int rc;
	int i = 0;
	LDAPMessage *res, *ent;
	int mcode;
	char **refs = NULL;
	char *userdn = NULL;

	dlog(1, "%d: search (%s, %s)", q, base ? base : "no base", flt);

	rc = ldap_search_st(ctx->ld, base, scp, flt, NULL, 0, &ctx->tv, &res);
	if (rc != LDAP_SUCCESS && rc != LDAP_REFERRAL) {
		dlog(1, "search result: bam %d %s", rc,ldap_err2string(rc));
		return(NULL);
	}

	q++;

	/*
	 * 2 things can happen here, the user is found in the current server
	 * or we get a referral.
	 * 
	 * if the user is found, we should get LDAP_RES_SEARCH_RESULT with 
	 * message code LDAP_SUCCESS, indicating a match was found. if this
	 * is the case, we'll get another message LDAP_RES_SEARCH_ENTRY from
	 * which we obtain the dn of the user that matched the filter. 
	 * actually they seem to come in the order ENTRY, RESULT but thats 
	 * not really important.
	 * 	
	 * if we get a referral, we pass the referral list to handle_referral
	 * which will bind to the referred to server and recursively call 
	 * search() (ie this function) again. the referral list will be
	 * delivered in a single message of LDAP_RES_SEARCH_RESULT with a
	 * message code of LDAP_REFERRAL. handle_referral will return the
	 * dn of the user from the referred to server if found, or NULL if 
	 * not.
	 */ 
	for (ent = ldap_first_message(ctx->ld, res); ent != NULL; ent = ldap_next_message(ctx->ld, ent)) {

		dlog(1, "%d: msgid %d, type %02x",q, i++, ldap_msgtype(ent));

		switch (ldap_msgtype(ent)) {

		case LDAP_RES_SEARCH_RESULT:

			rc = ldap_parse_result(ctx->ld, ent, &mcode, NULL, NULL, &refs, NULL, 0 );
			if (rc != LDAP_SUCCESS) {
				dlog(0, "ldap_parse_result %s", ldap_err2string(rc));
				break;
			}

			/* 
			 * if its not a referral we're done with this message
			 */ 
			if (mcode != LDAP_REFERRAL) {
				if (mcode != LDAP_SUCCESS)
					dlog(0, "%d: unhandled search result %x %s",q, mcode, ldap_err2string(mcode));
				break;
			}

			if (refs == NULL) {
				dlog(1, "no refs!");
				break;
			}

			if (mcode == LDAP_REFERRAL && ctx->noref) {
				dlog(1, "got referred when refs are off!");
				ldap_value_free(refs);
				break;
			}

			dlog(1, "%d: Calling handle_referral", q);
			userdn = handle_referral(ctx, refs, flt, scp);
			dlog(1, "%d: Referral returned userdn %s", q, userdn ? userdn : "null");
			ldap_value_free(refs);
			break;
		case LDAP_RES_SEARCH_ENTRY:
			userdn = ldap_get_dn(ctx->ld, ent);
			dlog(1, "%d: SEARCH_ENTRY userdn %s",q, userdn ? userdn : "none");
			break;
		}
	}

	ldap_msgfree(res);
	dlog(1, "%d: returning userdn = %s",q, userdn ? userdn : "no user dn");
	q--;
	return(userdn);
}

struct auth_ctx *
auth_ctx_dup(struct auth_ctx *ctx)
{
	struct auth_ctx *tmp;

	tmp = (struct auth_ctx *)malloc(sizeof(struct auth_ctx));
	if (tmp == NULL) {
		perror("no mem!");
		return(NULL);
	}

	memcpy(tmp, ctx, sizeof(struct auth_ctx));
	return(tmp);
}

char *
handle_referral(struct auth_ctx *ctx, char **refs, char *flt, int scp)
{
	int i;
	LDAPURLDesc *lurl;
	char *userdn = NULL;
	struct auth_ctx *tmpctx;

	tmpctx = auth_ctx_dup(ctx);
	if (tmpctx == NULL) {
		dlog(0, "auth_ctx_dup failed!");
		return(NULL);
	}

	for (i = 0; refs[i] != NULL; i++) {

		dlog(1, "ref: %s", refs[i]);

		if (!ldap_is_ldap_url(refs[i])) {
			dlog(0, "referrer given is not an ldap url!");
			continue;
		}

		if (ldap_url_parse(refs[i], &lurl) != 0) {
			dlog(0, "could not parse referral url!");
			continue;
		}		

		dlog(1, "referred to:\n\thost %s\n\tport %d\n\tbase %s\n\tfilter %s",
			lurl->lud_host ? lurl->lud_host : "no host", 
			lurl->lud_port,
			lurl->lud_dn ? lurl->lud_dn : "no dn",
			flt ? flt : "no filter");

		tmpctx->s.host = lurl->lud_host ? lurl->lud_host : NULL;
		tmpctx->s.port = lurl->lud_port;
		/* leave the version as it is */
	
		if (strcmp(lurl->lud_scheme, "ldaps") == 0)
			tmpctx->s.mode = MODE_SSL;
		else
			tmpctx->s.mode = MODE_CLEAR;
		
		if (!do_conn(tmpctx)) {
			dlog(0, "could not connect to referrer host!");
			continue;
		}

		if (!bind_password(tmpctx, ctx->keepcreds ? ctx->binddn : NULL, 
		    ctx->keepcreds ? ctx->bindpw : NULL)) {
			dlog(0, "ref bind failed!");
			continue;
		}

		userdn = search(tmpctx, lurl->lud_dn, flt, scp);
		if (userdn != NULL) {
			dlog(1, "search of referral gave me %s", userdn);
		}

		unbind(tmpctx);
		ldap_free_urldesc(lurl);
	}

	free(tmpctx);
	return(userdn);
}	

#ifdef __TEST_MODE
int
main(int argc, char **argv)
{
	int port = 389;
	char *host = "gibblets.ifost.org.au";
	char *basedn = NULL;
	LDAP *ld;
	char *userdn;
	char *binddn = "cn=ldapadmin,dc=ifost,dc=org,dc=au";
	char *bindpass = "secret";
	char *filter = "(objectclass=*)";
	char ch;
	extern char *optarg;
	extern int optind;

	while ((ch = getopt(argc, argv, "b:h:p:D:w:")) != -1) {
		switch(ch) {
		case 'h':
			host = optarg;
			break;
		case 'p':
			port = atoi(optarg);
			break;
		case 'b':
			basedn = optarg;
			break;
		case 'D':
			binddn = optarg;
			break;
		case 'w':
			bindpass = optarg;
			break;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc == 1) 
		filter = argv[0];
	
	ld = host_connect(host, port);
	if (ld == NULL)
		errx(1, "host_connect");

	/*if (!bind(ld, "cn=ldapadmin,dc=ifost,dc=org,dc=au", "secret")) { */
	if (!bind(ld, binddn, bindpass)) {
		dlog(1, "bind failed!");
		return(0);
	}
	
	userdn = search(ld, basedn, filter, LDAP_SCOPE_SUBTREE);
	dlog(1, "userdn %s", userdn ? userdn : "no user dn returned");

	return(0);

}
#endif
