/*
 * Standard library
 *
 * This library provides C implementations of the standard library
 * functions.
 *
 * (C) 2006, Pascal Schmidt <arena-language@ewetel.net>
 * see file ../doc/LICENSE for license
 */
#ifndef LIBSTD_STD_H
#define LIBSTD_STD_H

#include <sys/types.h>
#include <sys/stat.h>

#include "../libmisc/misc.h"
#include "../libruntime/runtime.h"
#include "../libeval/eval.h"

/*
 * Handle management structure
 */
typedef struct {
  int	count;
  void	**entries;
} handles;

/*
 * Handle management functions
 */
handles *handle_init(void);
void handle_free(handles *h);
int handle_add(handles *h, void *data);
void *handle_get(handles *h, int handle);
void handle_del(handles *h, int handle);

/*
 * Dictionary functions
 */
void dict_teardown(void);
value *dict_open(unsigned int, value **);
value *dict_read(unsigned int, value **);
value *dict_write(unsigned int, value **);
value *dict_remove(unsigned int, value **);
value *dict_exists(unsigned int, value **);
value *dict_close(unsigned int, value **);

/*
 * Runtime environment functions
 */
value *rt_type_of(unsigned int, value **);
value *rt_tmpl_of(unsigned int, value **);
value *rt_is_void(unsigned int, value **);
value *rt_is_bool(unsigned int, value **);
value *rt_is_int(unsigned int, value **);
value *rt_is_float(unsigned int, value **);
value *rt_is_string(unsigned int, value **);
value *rt_is_array(unsigned int, value **);
value *rt_is_struct(unsigned int, value **);
value *rt_is_fn(unsigned int, value **);
value *rt_is_a(unsigned int, value **);
value *rt_is_local(unsigned int, value **);
value *rt_is_global(unsigned int, value **);
value *rt_is_function(unsigned int, value **);
value *rt_is_var(unsigned int, value **);
value *rt_is_tmpl(unsigned int, value **);
value *rt_set(unsigned int, value **);
value *rt_get(unsigned int, value **);
value *rt_get_static(unsigned int, value **);
value *rt_unset(unsigned int, value **);
value *rt_assert(unsigned int, value **);
value *rt_global(unsigned int, value **);
value *rt_cast_to(unsigned int, value **);
value *rt_versions(unsigned int, value **);

/*
 * System environment functions
 */
value *sys_exit(unsigned int, value **);
value *sys_getenv(unsigned int, value **);
value *sys_system(unsigned int, value **);

/*
 * Random number functions
 */
value *rnd_random(unsigned int, value **);
value *rnd_srandom(unsigned int, value **);

/*
 * Print functions
 */
value *print_print(unsigned int, value **);
value *print_dump(unsigned int, value **);
value *print_sprintf(unsigned int, value **);
value *print_printf(unsigned int, value **);

/*
 * String functions
 */
value *str_length(unsigned int, value **);
value *str_concat(unsigned int, value **);
value *str_leftmost(unsigned int, value **);
value *str_rightmost(unsigned int, value **);
value *str_strpos(unsigned int, value **);
value *str_span(unsigned int, value **);
value *str_cspan(unsigned int, value **);
value *str_pbrk(unsigned int, value **);
value *str_coll(unsigned int, value **);
value *str_lower(unsigned int, value **);
value *str_upper(unsigned int, value **);
value *str_is_alnum(unsigned int, value **);
value *str_is_alpha(unsigned int, value **);
value *str_is_cntrl(unsigned int, value **);
value *str_is_digit(unsigned int, value **);
value *str_is_graph(unsigned int, value **);
value *str_is_lower(unsigned int, value **);
value *str_is_print(unsigned int, value **);
value *str_is_punct(unsigned int, value **);
value *str_is_space(unsigned int, value **);
value *str_is_upper(unsigned int, value **);
value *str_is_xdigit(unsigned int, value **);
value *str_mid(unsigned int, value **);
value *str_left(unsigned int, value **);
value *str_right(unsigned int, value **);
value *str_ord(unsigned int, value **);
value *str_chr(unsigned int, value **);
value *str_explode(unsigned int, value **);
value *str_implode(unsigned int, value **);
value *str_ltrim(unsigned int, value **);
value *str_rtrim(unsigned int, value **);
value *str_trim(unsigned int, value **);

/*
 * Math functions
 */
value *math_exp(unsigned int, value **);
value *math_log(unsigned int, value **);
value *math_log10(unsigned int, value **);
value *math_sqrt(unsigned int, value **);
value *math_ceil(unsigned int, value **);
value *math_floor(unsigned int, value **);
value *math_fabs(unsigned int, value **);
value *math_sin(unsigned int, value **);
value *math_cos(unsigned int, value **);
value *math_tan(unsigned int, value **);
value *math_asin(unsigned int, value **);
value *math_acos(unsigned int, value **);
value *math_atan(unsigned int, value **);
value *math_sinh(unsigned int, value **);
value *math_cosh(unsigned int, value **);
value *math_tanh(unsigned int, value **);
value *math_abs(unsigned int, value **);

/*
 * File functions
 */
void file_teardown(void);
value *file_open(unsigned int, value **);
value *file_seek(unsigned int, value **);
value *file_tell(unsigned int, value **);
value *file_read(unsigned int, value **);
value *file_write(unsigned int, value **);
value *file_setbuf(unsigned int, value **);
value *file_flush(unsigned int, value **);
value *file_is_eof(unsigned int, value **);
value *file_is_error(unsigned int, value **);
value *file_clearerr(unsigned int, value **);
value *file_close(unsigned int, value **);
value *file_remove(unsigned int, value **);
value *file_rename(unsigned int, value **);
value *file_errno(unsigned int, value **);
value *file_strerror(unsigned int, value **);
value *file_getc(unsigned int, value **);
value *file_gets(unsigned int, value **);

/*
 * Array functions
 */
value *array_mkarray(unsigned int, value **);
value *array_sort(unsigned int, value **);
value *array_is_sorted(unsigned int, value **);
value *array_unset(unsigned int, value **);
value *array_compact(unsigned int, value **);
value *array_search(unsigned int, value **);
value *array_merge(unsigned int, value **);
value *array_reverse(unsigned int, value **);

/*
 * Struct functions
 */
value *struct_mkstruct(unsigned int, value **);
value *struct_get(unsigned int, value **);
value *struct_set(unsigned int, value **);
value *struct_unset(unsigned int, value **);
value *struct_fields(unsigned int, value **);
value *struct_methods(unsigned int, value **);
value *struct_is_field(unsigned int, value **);
value *struct_is_method(unsigned int, value **);
value *struct_merge(unsigned int, value **);

/*
 * Date and time functions
 */
value *time_time(unsigned int, value **);
value *time_gmtime(unsigned int, value **);
value *time_localtime(unsigned int, value **);
value *time_mktime(unsigned int, value **);
value *time_asctime(unsigned int, value **);
value *time_ctime(unsigned int, value **);
value *time_strftime(unsigned int, value **);

/*
 * Functions on functions
 */
value *fn_is_builtin(unsigned int, value **);
value *fn_is_userdef(unsigned int, value **);
value *fn_call(unsigned int, value **);
value *fn_call_array(unsigned int, value **);
value *fn_call_method(unsigned int, value **);
value *fn_call_method_array(unsigned int, value **);
value *fn_prototype(unsigned int, value **);
value *fn_map(unsigned int, value **);
value *fn_filter(unsigned int, value **);
value *fn_foldl(unsigned int, value **);
value *fn_foldr(unsigned int, value **);
value *fn_take_while(unsigned int, value **);
value *fn_drop_while(unsigned int, value **);

/*
 * Locale functions
 */
value *lc_getlocale(unsigned int, value **);
value *lc_setlocale(unsigned int, value **);
value *lc_localeconv(unsigned int, value **);

/*
 * List functions
 */
value *list_nil(unsigned int, value **);
value *list_cons(unsigned int, value **);
value *list_head(unsigned int, value **);
value *list_tail(unsigned int, value **);
value *list_last(unsigned int, value **);
value *list_init(unsigned int, value **);
value *list_take(unsigned int, value **);
value *list_drop(unsigned int, value **);
value *list_length(unsigned int, value **);
value *list_null(unsigned int, value **);
value *list_elem(unsigned int, value **);
value *list_intersperse(unsigned int, value **);
value *list_replicate(unsigned int, value **);

/*
 * Register functions
 */
void stdlib_register(int argc, char **argv);

#endif
