/*
 * File operations, based on standard C facilities
 * (C) 2006, Pascal Schmidt <arena-language@ewetel.net>
 * see file ../doc/LICENSE for license
 */

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "stdlib.h"

/*
 * List of open file handles
 */
handles *files;

/*
 * Flag for initialization of file handles
 */
static int files_init = 0;

/*
 * Initialize standard filehandles
 */
static void init_files(void)
{
  int res1, res2, res3;

  files = handle_init();
  files_init = 1;

  res1 = handle_add(files, stdin);
  res2 = handle_add(files, stdout);
  res3 = handle_add(files, stderr);
  
  sanity(res1 == 1 && res2 == 2 && res3 == 3);
}

/*
 * Add file structure to list
 *
 * Adds a file structure pointer to the internal list. Reuses
 * the first free slot if found, otherwise the list is extended
 * by one entry. Returns an integer handle for the file
 * structure, or -1 if an out of memory condition occurs.
 */
static int add_file(FILE *fp)
{
  if (!files_init) {
    init_files();
  }
  return handle_add(files, fp);
}

/*
 * Get file structure from list
 *
 * This function obtains a file structure pointer from the internal
 * list. Bounds are checked.
 */
static FILE *get_file(int handle)
{
  if (!files_init) {
    init_files();
  }
  return handle_get(files, handle);
}

/*
 * Remove file structure from list
 *
 * This function removes a file structure pointer from the internal
 * list. Bounds are checked.
 */
static void del_file(int handle)
{
  if (!files_init) {
    init_files();
  }
  handle_del(files, handle);
}

/*
 * Tear down memory used by file functions
 */
void file_teardown(void)
{
  if (files_init) {
    int i;
    
    for (i = 0; i < files->count; i++) {
      fclose((FILE *) files->entries[i]);
      handle_del(files, i + 1);
    }
    handle_free(files);
  }
}

/*
 * Open file with given mode
 *
 * Opens the given filename with the given mode and returns a file
 * handle. Modes are as in C. On error, void is returned.
 */
value *file_open(unsigned int argc, value **argv)
{
  char *filename = argv[0]->value_u.string_val.value;
  char *filemode = argv[1]->value_u.string_val.value;
  FILE *fp;
  int handle;

  if (!value_str_compat(argv[0]) || !value_str_compat(argv[1])) {
    return value_make_void();
  }
  
  fp = fopen(filename, filemode);
  handle = add_file(fp);
  if (handle == -1) {
    return value_make_void();
  } 

  return value_make_int(handle);
}

/*
 * Set file position
 *
 * This function sets the file position for the given file handle.
 * Positive values are offsets from the start of the file, negative
 * offsets are from the end of the file. Returns true on success,
 * false on failure.
 */
value *file_seek(unsigned int argc, value **argv)
{
  FILE *fp = get_file(argv[0]->value_u.int_val);
  int pos  = argv[1]->value_u.int_val;
  
  if (!fp) {
    return value_make_bool(0);
  }
  
  return value_make_bool(!fseek(fp, pos, pos >= 0 ? SEEK_SET : SEEK_END));
}

/*
 * Get file position
 *
 * Gets the current file position for the given file handle. On
 * error, void is returned.
 */
value *file_tell(unsigned int argc, value **argv)
{
  FILE *fp = get_file(argv[0]->value_u.int_val);
  long pos;
  
  if (!fp) {
    return value_make_void();
  }

  pos = ftell(fp);
  if (pos == -1) {
    return value_make_void();
  }
  return value_make_int(pos);
}

/*
 * Read from file
 *
 * Reads at most n bytes from the given file descriptor. The result
 * is a string of at most n characters. On error, void is returned.
 * On read errors, a short string is returned.
 */
value *file_read(unsigned int argc, value **argv)
{
  FILE *fp = get_file(argv[0]->value_u.int_val);
  int max  = argv[1]->value_u.int_val;
  char *buf;
  int units;
  value *res;
  
  if (!fp) {
    return value_make_void();
  }
  
  buf = oom(calloc(max, 1));
  units = fread(buf, 1, max, fp);
  res = value_make_memstring(buf, units);
  free(buf);

  return res;
}

/*
 * Write string to file
 *
 * This function writes a string to the given file handle. The number
 * of bytes written is returned. On error, void is returned.
 */
value *file_write(unsigned int argc, value **argv)
{
  FILE *fp = get_file(argv[0]->value_u.int_val);
  int len  = argv[1]->value_u.string_val.len;
  const char *data = argv[1]->value_u.string_val.value;
  size_t written;
  
  if (!fp) {
    return value_make_void();
  }
  
  if (!len) {
    return value_make_int(0);
  }
  
  written = fwrite(data, 1, len, fp);
  return value_make_int(written);
}

/*
 * Set buffering
 *
 * Enables or disables I/O buffering for the given file handle.
 * Returns true on success or false on failure.
 */
value *file_setbuf(unsigned int argc, value **argv)
{
  FILE *fp = get_file(argv[0]->value_u.int_val);
  int res = -1;
  
  if (fp) {
    if (argv[1]->value_u.bool_val) {
      res = setvbuf(fp, NULL, _IOFBF, BUFSIZ);
    } else {
      res = setvbuf(fp, NULL, _IONBF, BUFSIZ);
    }
  }
  return value_make_bool(res == 0);
}

/*
 * Flush file buffer
 *
 * Flushes the I/O buffer for the given file handle. Returns true
 * on success and false on failure.
 */
value *file_flush(unsigned int argc, value **argv)
{
  FILE *fp = get_file(argv[0]->value_u.int_val);
  
  if (!fp) {
    return value_make_bool(0);
  }
  return value_make_bool(fflush(fp) == 0);
}

/*
 * Check end-of-file condition
 *
 * Returns true if the given file handle points to the end of the
 * corresponding file. The result is only meaninful if the file
 * has been read before. On error, EOF is assumed to be reached.
 */
value *file_is_eof(unsigned int argc, value **argv)
{
  FILE *fp = get_file(argv[0]->value_u.int_val);
  
  if (!fp) {
    return value_make_bool(1);
  }
  return value_make_bool(feof(fp));
}

/*
 * Check error condition
 *
 * Returns true if the given file handle had I/O errors during one
 * of the operations executed before. If the status cannot be
 * determined, it is assumed that no errors were encountered.
 */
value *file_is_error(unsigned int argc, value **argv)
{
  FILE *fp = get_file(argv[0]->value_u.int_val);

  if (!fp) {
    return value_make_bool(0);
  }
  return value_make_bool(ferror(fp));
}

/*
 * Clear status indicators
 *
 * This function clears the end-of-file and error indicators for
 * the given file handle.
 */
value *file_clearerr(unsigned int argc, value **argv)
{
  FILE *fp = get_file(argv[0]->value_u.int_val);

  if (fp) {
    clearerr(fp);
  }
  return value_make_void();
}

/*
 * Close file
 *
 * This function closes the given file handle. It returns true on
 * success and false if an error occurs during closing. In any
 * case, the file handle is closed when the function returns. If
 * the given file handle does not exit, success is reported.
 */
value *file_close(unsigned int argc, value **argv)
{
  FILE *fp = get_file(argv[0]->value_u.int_val);

  if (!fp) {
    return value_make_bool(1);
  } else {
    del_file(argv[0]->value_u.int_val);
  }
  return value_make_bool(!fclose(fp));
}

/*
 * Remove file
 *
 * This function tries to remove the given filename from the system.
 * It returns true on success and false on failure.
 */
value *file_remove(unsigned int argc, value **argv)
{
  const char *name = argv[0]->value_u.string_val.value;
  
  if (!value_str_compat(argv[0])) {
    return value_make_bool(0);
  }
  return value_make_bool(!remove(name));
}

/*
 * Rename file
 *
 * This functions tries to rename the file given as the first
 * argument to the name given as the second argument. It returns
 * true on success and false on failure.
 */
value *file_rename(unsigned int argc, value **argv) 
{
  const char *src = argv[0]->value_u.string_val.value;
  const char *dst = argv[1]->value_u.string_val.value;
  
  if (!value_str_compat(argv[0]) || !value_str_compat(argv[1])) {
    return value_make_bool(0);
  }
  return value_make_bool(!rename(src, dst));
}

/*
 * Get error number of last function call
 */
value *file_errno(unsigned int argc, value **argv)
{
  return value_make_int(errno);
}

/*
 * Get error message for error number
 */
value *file_strerror(unsigned int argc, value **argv)
{
  return value_make_string(strerror(argv[0]->value_u.int_val));
}

/*
 * Get character from file
 */
value *file_getc(unsigned int argc, value **argv)
{
  FILE *fp = get_file(argv[0]->value_u.int_val);
  char buf[2];
  int got;

  if (!fp) {
    return value_make_void();
  }
  got = fgetc(fp);
  if (got == EOF) {
    return value_make_void();
  }
  buf[0] = got;
  buf[1] = 0;
  return value_make_string(buf);
}

/*
 * Get line from file
 */
value *file_gets(unsigned int argc, value **argv)
{
  FILE *fp = get_file(argv[0]->value_u.int_val);
  char buf[65536];
  char *res;
  
  if (!fp) {
    return value_make_void();
  }
  res = fgets(buf, 65536, fp);
  if (!res) {
    return value_make_void();
  }
  return value_make_string(buf);
}
