/*
 * Symbol table stack
 * (C) 2006, Pascal Schmidt <arena-language@ewetel.net>
 * see file ../doc/LICENSE for license
 */

#include <stdlib.h>

#include "runtime.h"

/*
 * global symbol table
 */
static symtab *global_table = NULL;
/*
 * number of local symbol tables
 */
static unsigned int local_depth = 0;
/*
 * local symbol tables
 */
static symtab **local_tables = NULL;

/*
 * Tear down symbol table stack
 *
 * Deallocates all memory used by the symbol table stack.
 */
static void symtab_stack_teardown(void)
{
  sanity(global_table);
  
  while (local_depth > 0) {
    symtab_stack_leave();
  }

  free(local_tables);
  local_tables = NULL;
  
  symtab_free(global_table);
  global_table = NULL;
}

/*
 * Initialize symbol table stack
 *
 * This function allocates memory for the global symbol table
 * and resets the local symbol table stack to depth 0.
 */
void symtab_stack_init(void)
{
  if (!global_table) {
    global_table = oom(symtab_alloc(0));

    local_depth = 0;
    local_tables = NULL;
    
    atexit(symtab_stack_teardown);
  }
}


/*
 * Enter local symbol table
 *
 * This function adds a new local symbol table to the stack. The
 * new symbol table obscures the previous toplevel local table.
 */
void symtab_stack_enter(void)
{
  symtab **local;
  symtab *newtab;
 
  newtab = symtab_alloc(7);
 
  local = oom(realloc(local_tables,
    (local_depth + 1) * sizeof(symtab *)));

  local_tables = local;
  local_tables[local_depth] = newtab;
  local_depth++;
}

/*
 * Leave local symbol table
 *
 * This function removes and frees the topmost local symbol table.
 * The previous local symbol table becomes visible again.
 */
void symtab_stack_leave(void)
{
  if (local_depth == 0) {
    return;
  }
  local_depth--;
  symtab_free(local_tables[local_depth]);
}

/*
 * Pop symbol table stack
 *
 * This function returns the topmost local symbol table and makes
 * the previous local symbol table visible again.
 */
symtab *symtab_stack_pop(void)
{
  sanity(local_depth > 0);
  
  --local_depth;
  return local_tables[local_depth];
}

/*
 * Get stack depth
 *
 * This function returns the current depth of the local symbol
 * table stack. It returns 0 if there are no local symbol tables
 * at the moment.
 */
unsigned int symtab_stack_depth(void)
{
  return local_depth;
}

/*
 * Add symbol table entry to stack
 *
 * This functions adds the given entry to the topmost symbol
 * table of the stack. This means the global table if no
 * local symbol table exists.
 */
void symtab_stack_add(symtab_entry *entry)
{
  sanity(entry);

  if (local_depth > 0) {
    symtab_add(local_tables[local_depth-1], entry);
  } else {
    symtab_add(global_table, entry);
  }
}

/*
 * Put symbol table entry into global table
 *
 * This function adds the given entry to the global symbol
 * table, no matter whether a current local table exists.
 */
void symtab_stack_add_global(const char *name, value *val)
{
  sanity(name && val);
  symtab_add_variable(global_table, name, val);
}

/*
 * Add variable entry to stack
 *
 * This functions adds the given variable to the topmost symbol
 * table of the stack. This means the global table if no local
 * symbol table exists.
 */
void symtab_stack_add_variable(const char *name, value *val)
{
  sanity(name && val);

  if (local_depth > 0) {
    symtab_add_variable(local_tables[local_depth-1], name, val);
  } else {
    symtab_add_variable(global_table, name, val);
  }
}

/*
 * Add function entry to stack
 *
 * This function adds the given function to the topmost symbol
 * table of the stack. This means the global table if no local
 * symbol table exists.
 */
void symtab_stack_add_function(const char *name, const signature *sig)
{
  sanity(name && sig);

  if (local_depth > 0) {
    symtab_add_function(local_tables[local_depth-1], name, sig);
  } else {
    symtab_add_function(global_table, name, sig);
  }
}

/*
 * Add template entry to stack
 *
 * This function adds the given template to the topmost symbol table
 * of the stack. This means the global table if no local symbol
 * table exists.
 */
void symtab_stack_add_template(const char *name, const char *parent, void *def)
{
  sanity(name && def);

  if (local_depth > 0) {
    symtab_add_template(local_tables[local_depth-1], name, parent, def);
  } else {
    symtab_add_template(global_table, name, parent, def);
  }
}

/*
 * Lookup symbol in stack
 *
 * This function looks for the given symbol name in the topmost
 * local symbol table. If the symbol is not found there, it is
 * also searched for in the global table.
 */
symtab_entry *symtab_stack_lookup(const char *symbol)
{
  symtab_entry *entry = NULL;

  if (local_depth > 0) {
    entry = symtab_lookup(local_tables[local_depth-1], symbol);
  }
  if (!entry) {
    entry = symtab_lookup(global_table, symbol);
  }
  return entry;
}

/*
 * Check locality
 *
 * This function returns 1 if the given symbol name would be
 * resolved from a local symbol table. If the global table is
 * active at the moment, the answer is always "yes".
 */
int symtab_stack_local(const char *symbol)
{
  symtab_entry *entry;
  
  if (local_depth > 0) {
    entry = symtab_lookup(local_tables[local_depth-1], symbol);
    return (entry != NULL);
  }
  return 1;
}

/*
 * Delete symbol from stack
 *
 * This function deletes the given symbol name from the topmost
 * symbol table in the stack. This means the global table if no
 * local symbol table exists.
 */
void symtab_stack_delete(const char *symbol)
{
  if (local_depth > 0) {
    symtab_delete(local_tables[local_depth-1], symbol);
  } else {
    symtab_delete(global_table, symbol);
  }
}
